#' OneMap interface with MDSMap package with possibilitie of multipoint distances estimation
#'
#' For a given sequence of markers, apply mds method described in Preedy and Hackett (2016)
#' using MDSMap package to ordering markers and estimates the genetic distances with OneMap
#' multipoint approach. Also gives MDSMap input file format for directly analysis in this package.
#'
#' @param input.seq an object of class \code{sequence}
#' @param out.file path to the generated MDSMap input file.
#' @param mds.graph.file path to the graphic generated by MDSMap
#' @param p Integer - the penalty for deviations from the sphere - higher p
#' forces points more closely onto a sphere.
#' @param n Vector of integers or strings containing markers to be omitted from
#' the analysis.
#' @param displaytext Shows markers names in analysis graphic view
#' @param hmm Multipoint genetic distance estimation
#' @param weightfn Character string specifying the values to use for the weight
#' matrix in the MDS 'lod2' or 'lod'.
#' @param mapfn Character string specifying the map function to use on the
#' recombination fractions 'haldane' is default, 'kosambi' or 'none'.
#' @param ispc Logical determining the method to be used to estimate the map. By default 
#' this is TRUE and the method of principal curves will be used. If FALSE then the 
#' constrained MDS method will be used.
#' @param mds.seq When some pair of markers do not follow the linkage criteria, 
#' if \code{TRUE} one of the markers is removed and mds is performed again.
#' @return An object of class \code{sequence}, which is a list containing the
#' following components: \item{seq.num}{a \code{vector} containing the
#' (ordered) indices of markers in the sequence, according to the input file.}
#' \item{seq.phases}{a \code{vector} with the linkage phases between markers
#' in the sequence, in corresponding positions. \code{-1} means that there are
#' no defined linkage phases.} \item{seq.rf}{a \code{vector} with the
#' recombination frequencies between markers in the sequence. \code{-1} means
#' that there are no estimated recombination frequencies.}
#' \item{seq.like}{log-likelihood of the corresponding linkage map.}
#' \item{data.name}{name of the object of class \code{onemap} with the raw
#' data.} \item{twopt}{name of the object of class \code{rf_2pts} with the
#' 2-point analyses.}
#' 
#' @details For better description about MDS method, see MDSMap package vignette.
#' 
#' @author Cristiane Taniguti, \email{chtaniguti@@usp.br} 
#' @seealso  \url{https://CRAN.R-project.org/package=MDSMap}.
#'
#'
#' @references 
#' Preedy, K. F.  and  Hackett, C. A.  (2016). A rapid marker ordering approach for high-density
#' genetic linkage maps in experimental autotetraploid populations using multidimensional
#' scaling. \emph{Theoretical and Applied Genetics} 129: 2117-2132
#'
#' Mollinari, M., Margarido, G. R. A., Vencovsky, R. and Garcia, A. A. F.
#' (2009) Evaluation of algorithms used to order markers on genetics maps.
#' \emph{Heredity} 103: 494-502.
#'
#' Wu, R., Ma, C.-X., Painter, I. and Zeng, Z.-B. (2002a) Simultaneous maximum
#' likelihood estimation of linkage and linkage phases in outcrossing species.
#' \emph{Theoretical Population Biology} 61: 349-363.
#'
#' Wu, R., Ma, C.-X., Wu, S. S. and Zeng, Z.-B. (2002b). Linkage mapping of
#' sex-specific differences. \emph{Genetical Research} 79: 85-96
#'
#'@import MDSMap
#'@importFrom utils write.table
#'
#'@export
mds_onemap <- function(input.seq, out.file= "out.file", mds.graph.file="NULL.pdf", p = NULL, n=NULL, ispc=TRUE,
                       displaytext=FALSE, weightfn='lod2', mapfn='haldane', hmm = TRUE, mds.seq=TRUE){
  
  ## checking for correct object
  if(!("sequence" %in% class(input.seq)))
    stop(deparse(substitute(input.seq))," is not an object of class 'sequence'")
  
  
  n_ind <- get(input.seq$data.name)$n.ind
  obj.class <- class(get(input.seq$data.name))
  if(obj.class[2]=="outcross"){
    mat<-get_mat_rf_out(input.seq, LOD=TRUE,  max.rf = 0.501, min.LOD = -0.1)
  } else {
    mat<-get_mat_rf_in(input.seq, LOD=TRUE,  max.rf = 0.501, min.LOD = -0.1)
  }
  n_mk <- nrow(mat)
  
  mat.rf <- mat.lod <- matrix(rep(NA, n_mk*n_mk), nrow = n_mk)
  colnames(mat.rf) <- colnames(mat.lod) <- rownames(mat.rf) <- rownames(mat.lod) <- colnames(mat)
  mat.lod[lower.tri(mat.lod)] <- mat[lower.tri(mat)]
  mat.rf[upper.tri(mat.rf)] <- mat[upper.tri(mat)]
  
  df <- reshape2::melt(mat.lod, na.rm = TRUE)
  df.rf <- reshape2::melt(mat.rf, na.rm = TRUE)
  
  df <- cbind(df.rf, df$value)
  df <- df[with(df, order(Var1, Var2)),]
  
  n_col <- dim(df)[1]
  vector.file <- apply(df, 1, function(x) paste(x, collapse = " "))
  file.out <- c(paste(n_mk, n_col, collapse = " "), vector.file)
  
  write.table(file.out, file = out.file, col.names = FALSE,
              row.names = FALSE, quote = FALSE)
  
  pdf(mds.graph.file)
  mds_map <- MDSMap::estimate.map(out.file, p = p, n=n, ispc = ispc, 
                                  displaytext = displaytext)
  dev.off()
  if(hmm==TRUE){
    ord_mds <- match(as.character(mds_map$locimap[,2]), colnames(get(input.seq$data.name)$geno)) 
    seq_mds <- make_seq(get(input.seq$twopt), ord_mds)
    seq_mds$twopt <- input.seq$twopt
    mds_map <- map(seq_mds, mds.seq = mds.seq)
  }
  
  if(!is.list(mds_map)) {
    new.seq <- make_seq(get(input.seq$twopt), mds_map)
    new.seq$twopt <- input.seq$twopt
    mds_map <- mds_onemap(new.seq, out.file= out.file, mds.graph.file=mds.graph.file, p = NULL, n=NULL, ispc=TRUE,
                          displaytext=displaytext, weightfn=weightfn, mapfn=mapfn, hmm = hmm, mds.seq=mds.seq)
  }
  return(mds_map)
}
