% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/misc.R
\name{magneticField}
\alias{magneticField}
\title{Earth magnetic declination, inclination, and intensity}
\usage{
magneticField(longitude, latitude, time)
}
\arguments{
\item{longitude}{longitude in degrees east (negative for degrees west).  The
dimensions must conform to lat.}

\item{latitude}{latitude in degrees north, a number, vector, or matrix.}

\item{time}{The time at which the field is desired. This may be a
single value or a vector or matrix that is structured to match
\code{longitude} and \code{latitude}. The value may a decimal year,
a POSIXt time, or a Date time.}
}
\value{
A list containing \code{declination}, \code{inclination}, and
\code{intensity}.
}
\description{
Implements the 12th generation International Geomagnetic Reference Field
(IGRF), based on a reworked version of a Fortran program downloaded from a
NOAA website (see reference 1).
}
\details{
The code (subroutine \code{igrf12syn}) seems to have
been written by Susan Macmillan of the British Geological Survey.  Comments
in the source code indicate that it employs coefficients agreed to in
December 2014 by the IAGA Working Group V-MOD.  Other comments in that code
suggest that the valid time interval is from years 1900 to 2020,
with only the values from 1945 to 2010 being considered definitive.

Reference 2 suggests that a new version to the underlying source
code might be expected in 2019 or 2020, but a check on January 31,
2019, showed that version 12, as incorporated in oce since
2015, remains the active version.
}
\examples{
library(oce)
# 1. Today's value at Halifax NS
magneticField(-(63+36/60), 44+39/60, Sys.Date())

# 2. World map of declination in year 2000.
\donttest{
data(coastlineWorld)
par(mar=rep(0.5, 4)) # no axes on whole-world projection
mapPlot(coastlineWorld, projection="+proj=robin", col="lightgray")
# Construct matrix holding declination
lon <- seq(-180, 180)
lat <- seq(-90, 90)
dec2000 <- function(lon, lat)
    magneticField(lon, lat, 2000)$declination
dec <- outer(lon, lat, dec2000) # hint: outer() is very handy!
# Contour, unlabelled for small increments, labeled for
# larger increments.
mapContour(lon, lat, dec, col='blue', levels=seq(-180, -5, 5),
           lty=3, drawlabels=FALSE)
mapContour(lon, lat, dec, col='blue', levels=seq(-180, -20, 20))
mapContour(lon, lat, dec, col='red', levels=seq(5, 180, 5),
           lty=3, drawlabels=FALSE)
mapContour(lon, lat, dec, col='red', levels=seq(20, 180, 20))
mapContour(lon, lat, dec, levels=180, col='black', lwd=2, drawlabels=FALSE)
mapContour(lon, lat, dec, levels=0, col='black', lwd=2)
}

}
\references{
\enumerate{
\item The underlying Fortran code is from \code{igrf12.f}, downloaded the NOAA
website (\url{https://www.ngdc.noaa.gov/IAGA/vmod/igrf.html}) on June 7,
\item 
\item Witze, Alexandra. \dQuote{Earth's Magnetic Field Is Acting up and Geologists Don't Know Why.}
Nature 565 (January 9, 2019): 143.
\url{https://doi.org/10.1038/d41586-019-00007-1}.
}
}
\seealso{
Other things related to magnetism: 
\code{\link{applyMagneticDeclination}()}
}
\author{
Dan Kelley wrote the R code and a fortran wrapper to the
\code{igrf12.f} subroutine, which was written by Susan Macmillan of the
British Geological Survey and distributed ``without limitation'' (email from
SM to DK dated June 5, 2015).
}
\concept{things related to magnetism}
