% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ctd.R
\docType{methods}
\name{[[,ctd-method}
\alias{[[,ctd-method}
\title{Extract Parts of a CTD Object}
\usage{
\S4method{[[}{ctd}(x, i, j, ...)
}
\arguments{
\item{x}{A \code{ctd} object, i.e. one inheriting from \code{\link{ctd-class}}.}

\item{i}{Character string indicating the name of item to extract.}

\item{j}{Optional additional information on the \code{i} item.}

\item{...}{Optional additional information (ignored).}
}
\description{
The \code{[[} method works for all \code{oce} objects, i.e.
objects inheriting from \code{\link{oce-class}}.  The purpose,
as with the related replacement method, \code{\link{[[<-}}, 
is to insulate users from the internal details of \code{oce}
objects, by looking for items within the various storage
slots of the object. Items not actually stored can also be 
extracted, including derived data, units of measurement,
and data-quality flags.

The method uses a two-step process to try to find the
requested information. First, a class-specific function 
is used to try to access the requested information (see
\dQuote{Details of the specialized ... method}). 
Second, if no match is found, a general function is used
(see \sQuote{Details of the general method}). If neither
method can locates the requested item, \code{NULL} is returned.
}
\section{Details of the general method}{


If the specialized method produces no matches, the following generalized
method is applied. As with the specialized method, the procedure hinges
first on the value of \code{i}.

First, a check is made as to whether \code{i} names one of the standard
\code{oce} slots, and returns the slot contents if so. Thus, 
\code{x[["metadata"]]} will retrieve the \code{metadata} slot, 
while \code{x[["data"]]} and \code{x[["processingLog"]]} return
those slots.

Next, if \code{i} is a string ending in the
\code{"Unit"}, then the characters preceding that string
are taken to be the name of an item in the data object, and a list
containing the unit is returned. This list consists of an item
named \code{unit}, which is an \code{\link{expression}}, and
an item named \code{scale}, which is a string describing the
measurement scale.  If the string ends in \code{" unit"}, e.g.
\code{x[["temperature unit"]]}, then just the expression is returned, 
and if it ends in \code{" scale"}, then just the scale is returned.

Next, if \code{i} is a string ending in \code{"Flag"}, then the corresponding
data-quality flag is returned (or \code{NULL} if there is no such flag). 
For example, \code{x[["salinityFlag"]]} returns a vector of salinity
flags if \code{x} is a ctd object.

If none of the preceding conditions are met, a check is done
to see if the \code{metadata} slot contains an item with
the provided name, and that is returned, if so. A direct match
is required for this condition.

Finally, the \code{data} slot is checked to see if it contains 
an item with the name indicated by \code{i}. In this case, a partial
match will work; this is accomplished by using \code{\link{pmatch}}.

If none of the above-listed conditions holds, then \code{NULL} is returned.
}

\section{Details of the specialized \code{ctd} method}{


Some uses of \code{[[,ctd-method} involve direct retrieval of
items within the \code{data} slot of the \code{ctd} object, 
while other uses involve calculations based on items in that
\code{data} slot. For an example, all \code{ctd} objects
should hold an item named \code{temperature} in the \code{data}
slot, so for example \code{x[["temperature"]]} will retrieve that
item. By contrast, \code{x[["sigmaTheta"]]} is taken to be a
request to compute \eqn{\sigma_\theta}{sigma[theta]}, and so
it yields a call to \code{\link{swTheta}(x)} even if
the \code{data} slot of \code{x} might happen to contain an item
named \code{theta}. This can be confusing at first, but it tends
to lead to fewer surprises in everyday work, for otherwise the
user would be forced to check the contents of any \code{ctd}
object under analysis, to determine whether that item will be looked
up or computed. Nothing is lost in this scheme, since the data
within the object are always accessible with \code{\link{oceGetData}}.

It should be noted that the accessor is set up to retrieve quantities
in conventional units. For example, \code{\link{read.ctd.sbe}} is
used on a \code{.cnv} file that stores pressure in psi, it will
be stored in the same unit within the \code{ctd} object, but
\code{x[["pressure"]]} will return a value that has been converted
to decibars.  (Users who need the pressure in PSI can
use \code{x@data$pressure}.)
Similarly, temperature is 
returned in the ITS-90 scale, with a conversion having been performed with
\code{\link{T90fromT68}}, if the object holds temperature in
IPTS-68.  Again, temperature on the IPTS-68
scale is returned with \code{x@data$temperature}.


This preference for computed over stored quantities is accomplished
by first checking for computed quantities, and then falling
back to the general \code{[[} method if no match is found.

Some quantities are optionally computed. For example, some data files
(e.g. the one upon which the \code{\link{section}} dataset is based)
store \code{nitrite} along with the sum of nitrite and nitrate, the
latter with name \code{`NO2+NO3`}. In this case, e.g. \code{x[["nitrate"]]}
will detect the setup, and subtract nitrite from the sum to yield
nitrate.

Below is a list of computed quantities, or at least quantites that are
typically not stored in data files.  (This is a vague statement because
Seabird software permits calculation of many of these and hence storage
within \code{.cnv} files.)

\itemize{

\item \code{CT} or \code{Conservative Temperature}: Conservative Temperature,
computed with \code{\link[gsw]{gsw_CT_from_t}} in the \code{gsw} package.

\item \code{depth}: Depth in metres below the surface, computed
with \code{\link{swDepth}(x)}.

\item \code{N2}: Square of Brunt-Vaisala frequency, computed  with
\code{\link{swN2}(x)}.

\item \code{potential temperature}: Potential temperature in the
UNESCO formulation, computed with \code{\link{swTheta}(x)}.
This is a synonym for \code{theta}.

\item \code{Rrho}: Density ratio, computed with \code{\link{swRrho}(x)}.

\item \code{SA} or \code{Absolute Salinity}: Absolute Salinity,
computed with \code{\link[gsw]{gsw_SA_from_SP}} in the \code{gsw} package.

\item \code{sigmaTheta}: A form of potential density anomaly, computed with
\code{\link{swSigmaTheta}(x)}.

\item \code{sigma0} Equal to \code{sigmaTheta}, i.e. potential density anomaly
referenced to a pressure of 0dbar, computed with \code{\link{swSigma0}(x)}.

\item \code{sigma1}: Potential density anomaly
referenced to a pressure of 1000dbar, computed with \code{\link{swSigma1}(x)}.

\item \code{sigma2}: Potential density anomaly
referenced to a pressure of 2000dbar, computed with \code{\link{swSigma2}(x)}.

\item \code{sigma3}: Potential density anomaly
referenced to a pressure of 3000dbar, computed with \code{\link{swSigma3}(x)}.

\item \code{sigma4}: potential density anomaly
referenced to a pressure of 4000dbar, computed with \code{\link{swSigma4}(x)}.
                                       
\item \code{SP}: Salinity on the Practical Salinity Scale, which is
\code{salinity} in the \code{data} slot.

\item \code{spice}: a variable that is in some sense orthogonal to density,
calculated with \code{\link{swSpice}(x)}.

\item \code{SR}: Reference Salinity computed with
\code{\link[gsw]{gsw_SR_from_SP}} in the \code{gsw} package.

\item \code{Sstar}: Preformed Salinity computed with
\code{\link[gsw]{gsw_SR_from_SP}} in the \code{gsw} package.

\item \code{theta}: potential temperature in the UNESCO formulation,
computed with \code{\link{swTheta}(x)}. This is a synonym for
\code{potential temperature}.

\item \code{z}: Vertical coordinate in metres above the surface, computed with
\code{\link{swZ}(x)}.

}
}
\examples{
data(ctd)
head(ctd[["temperature"]])

}
\seealso{
Other functions that extract parts of \code{oce} objects: \code{\link{[[,adp-method}},
  \code{\link{[[,adv-method}},
  \code{\link{[[,amsr-method}},
  \code{\link{[[,argo-method}},
  \code{\link{[[,bremen-method}},
  \code{\link{[[,cm-method}},
  \code{\link{[[,coastline-method}},
  \code{\link{[[,echosounder-method}},
  \code{\link{[[,g1sst-method}},
  \code{\link{[[,gps-method}},
  \code{\link{[[,ladp-method}},
  \code{\link{[[,lisst-method}},
  \code{\link{[[,lobo-method}},
  \code{\link{[[,met-method}}, \code{\link{[[,odf-method}},
  \code{\link{[[,rsk-method}},
  \code{\link{[[,sealevel-method}},
  \code{\link{[[,section-method}},
  \code{\link{[[,tidem-method}},
  \code{\link{[[,topo-method}},
  \code{\link{[[,windrose-method}},
  \code{\link{[[<-,adv-method}}

Other things related to \code{ctd} data: \code{\link{[[<-,ctd-method}},
  \code{\link{as.ctd}}, \code{\link{cnvName2oceName}},
  \code{\link{ctd-class}}, \code{\link{ctdAddColumn}},
  \code{\link{ctdDecimate}}, \code{\link{ctdFindProfiles}},
  \code{\link{ctdRaw}}, \code{\link{ctdTrim}},
  \code{\link{ctdUpdateHeader}}, \code{\link{ctd}},
  \code{\link{gps-class}},
  \code{\link{handleFlags,ctd-method}},
  \code{\link{plot,ctd-method}}, \code{\link{plotProfile}},
  \code{\link{plotScan}}, \code{\link{plotTS}},
  \code{\link{read.ctd.itp}}, \code{\link{read.ctd.odf}},
  \code{\link{read.ctd.sbe}},
  \code{\link{read.ctd.woce.other}},
  \code{\link{read.ctd.woce}}, \code{\link{read.ctd}},
  \code{\link{subset,ctd-method}},
  \code{\link{summary,ctd-method}},
  \code{\link{woceNames2oceNames}}, \code{\link{write.ctd}}
}

