% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gradient_score.R
\name{scoreEstimation}
\alias{scoreEstimation}
\title{Gradient score function for the Brown--Resnick model.}
\usage{
scoreEstimation(obs, loc, vario, weightFun = NULL, dWeightFun = NULL,
  nCores = 1L, cl = NULL, ...)
}
\arguments{
\item{obs}{List of vectors exceeding an R-threshold, see de Fondeville and Davison (2016) for more details.}

\item{loc}{Matrix of coordinates as given by \code{expand.grid()}.}

\item{vario}{Semi-variogram function taking a vector of coordinates as input.}

\item{weightFun}{Function of weights.}

\item{dWeightFun}{Partial derivative function of \code{weightFun}.}

\item{nCores}{Number of cores used for the computation}

\item{cl}{Cluster instance as created by \code{makeCluster} of the \code{parallel} package.}

\item{...}{Parameters for \code{weightFun} and \code{dWeightFun}.}
}
\value{
Evaluation of the gradient score function for the set of observations \code{obs} and semi-variogram \code{vario}.
}
\description{
Compute the peaks-over-threshold gradient score function for the Brown--Resnick model.
}
\details{
The function computes the gradient score based on the representation developed by Wadsworth et al. (2014).
Margins must have been standardized. The weighting function must be differentiable and verify some properties
for consistency, see de Fondeville and Davison (2016) for more details.
}
\examples{
#Define variogram function
vario <- function(h){
   1 / 2 * norm(h,type = "2")^1.5
}

#Define locations
loc <- expand.grid(1:4, 1:4)

#Simulate data
obs <- simulPareto(1000, loc, vario)

#Evaluate risk functional
sums <- sapply(obs, sum)

#Define weighting function
weightFun <- function(x, u){
 x * (1 - exp(-(sum(x / u) - 1)))
}

#Define partial derivative of weighting function
dWeightFun <- function(x, u){
(1 - exp(-(sum(x / u) - 1))) + (x / u) * exp( - (sum(x / u) - 1))
}

#Select exceedances
threshold <- quantile(sums, 0.9)
exceedances <- obs[sums > threshold]

#Evaluate gradient score function
scoreEstimation(exceedances, loc, vario, weightFun = weightFun, dWeightFun, u = threshold)
}
\references{
de Fondeville, R. and A. C. Davison (2017). High-dimensional Peaks-over-threshold Inference for Brown-Resnick Processes. Submitted.
}
\author{
Raphael de Fondeville
}
