#' Randomize input CDM according to defined null models
#'
#' Given a prepared nulls.input object, will randomize a community data matrix according
#' to all nulls defined in defineNulls, and return a list of randomized CDMs.
#'
#' @param nulls.input A prepared nulls.input object, as generated by prepNulls.
#' @param nulls Optional list of named null model functions to use. If invoked, this 
#' option will likely be used to run a subset of the defined null models.
#' 
#' @details Currently we are running 9 null models.
#' This function first confirms that the input is of class nulls.input and, if so, then
#' confirms that the nulls to be calculated are in a named list (via checkNulls),
#' then lapplies all null model functions to the input CDM.
#'
#' @return A list of matrices. Each matrix is a product of a randomization of the input
#' CDM and one of the nulls from defineNulls.
#'
#' @export
#'
#' @references Miller, E. T., D. R. Farine, and C. H. Trisos. 2015. Phylogenetic community
#' structure metrics and null models: a review with new methods and software.
#' bioRxiv 025726.
#'
#' @examples
#' #simulate tree with birth-death process
#' tree <- geiger::sim.bdtree(b=0.1, d=0, stop="taxa", n=50)
#'
#' sim.abundances <- round(rlnorm(5000, meanlog=2, sdlog=1)) + 1
#'
#' cdm <- simulateComm(tree, richness.vector=10:25, abundances=sim.abundances)
#'
#' prepped <- prepNulls(tree, cdm)
#'
#' results <- runNulls(prepped)

runNulls <- function(nulls.input, nulls)
{
	if(!inherits(nulls.input, "nulls.input"))
	{
		stop("Input needs to be of class 'nulls.input'")
	}
	
	#if a list of named nulls functions is not passed in, assign nulls to be NULL, in
	#which case all nulls will be run
	if(missing(nulls))
	{
		nulls <- NULL
	}	
	
	nulls <- checkNulls(nulls)

	#finally, if no distances.among input was provided, the dispersalNull cannot be run.
	#remove it here. if other nulls are defined later, either add them to this list, or
	#do something more elegant so they are not run and errors are not thrown. this used to
	#check whether distances.among == ignore, but that was throwing lots of warnings if it
	#did not equal ignore. now it checks whether it is a matrix. if its not it removes
	#the dispersal null. it does not have to say ignore specifically anymore.
	
	if(!is.matrix(nulls.input$distances.among))
	{
		nulls$dispersal <- NULL
	}

	lapply(nulls, function(x) x(nulls.input))
}
