% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/allow.R
\name{allow}
\alias{allow}
\title{Implement decision rules for land use change}
\usage{
allow(x, categories, cd, rules, hist = NULL, ...)
}
\arguments{
\item{x}{numeric vector containing the land use pattern for the current
timestep}

\item{categories}{numeric vector containing land use categories in the study
region}

\item{cd}{numeric vector indicating the direction of change for each
land use category. A value of 1 means demand is increasing (i.e. the number
of cells belonging to the category must increase), -1 means decreasing
demand and 0 means demand is static}

\item{rules}{matrix. See details}

\item{hist}{numeric vector containing land use history (values represent the
number of timesteps the cell has contained the current land use category).
Only required for rules 2 and 3}

\item{\dots}{additional arguments (none)}
}
\value{
A matrix.
}
\description{
Identify legitimate transitions based on land use history and specific
transition rules.
}
\details{
Decision rules are based on those described by Verburg et al. (2002). The
\code{rules} input argument is a square matrix with dimensions equal to the
number of land use categories in the study region where rows represent the
current land use and columns represent future transitions. The value of each
element should represent a rule from the following list:

\enumerate{
  \item rule == 0 | rule == 1: this rule concerns specific land use
    transitions that are allowed (1) or not (0)
  \item rule > 100 & rule < 1000: this rule imposes a time limit (rule - 100)
    on land use transitions, after which land use change is not allowed. Time
    is taken from \code{hist}
  \item rule > 1000: this rule imposes a minimum period of time (rule-1000)
    before land use is allowed to change
}

\code{allow} should be called from \code{\link{allocate}} methods. The output
is a matrix with the same dimensions as the matrix used internally by
allocation functions to store land use suitability. Thus, by multiplying the
two matrices together, disallowed transitions are removed from the allocation
procedure.
}
\examples{
## Plum Island Ecosystems

## load observed land use data
obs <- ObsLulcRasterStack(x=pie,
                   pattern="lu",
                   categories=c(1,2,3),
                   labels=c("forest","built","other"),
                   t=c(0,6,14))

## get land use values
x <- getValues(obs[[1]])
x <- x[!is.na(x)]

## create vector of arbitrary land use history values
hist <- sample(1:10, length(x), replace=TRUE)

## calculate demand and get change direction for first timestep
dmd <- approxExtrapDemand(obs=obs, tout=0:14)
cd <- dmd[2,] - dmd[1,]

## create rules matrix, only allowing forest to change if the cell has
## belonged to forest for more than 8 years
rules <- matrix(data=c(1,1008,1008,
                        1,1,1,
                        1,1,1), nrow=3, ncol=3, byrow=TRUE)

allow <- allow(x=x,
               hist=hist,
               categories=obs@categories,
               cd=cd,
               rules=rules)

## create raster showing cells that are allowed to change from forest to built
r <- obs[[1]]
r[!is.na(r)] <- allow[,2]
r[obs[[1]] != 1] <- NA
plot(r)

## NB output is only useful when used within allocation routine
}
\references{
Verburg, P.H., Soepboer, W., Veldkamp, A., Limpiada, R., Espaldon,
V., Mastura, S.S. (2002). Modeling the spatial dynamics of regional land use:
the CLUE-S model. Environmental management, 30(3):391-405.
}
\seealso{
\code{\link{allowNeighb}}
}

