
anovaRMClass <- R6::R6Class(
    "anovaRMClass",
    inherit=anovaRMBase,
    private=list(
        .model=NA,
        .postHocRows=NA,
        .init=function() {

            private$.initRMTable()
            private$.initBSTable()
            private$.initSpericityTable()
            private$.initLeveneTable()
            private$.initPostHocTables()
            private$.initDescPlots()

        },
        .run=function() {

            dataSelected <- ! sapply(lapply(self$options$rmCells, function(x) return(x$measure)), is.null)

            ready <- sum(dataSelected) == length(self$options$rmCells) && length(self$options$rmTerms) > 0

            if (ready) {

                private$.dataCheck()
                data <- private$.wideToLong()
                modelFormula <- private$.modelFormula()

                suppressMessages({
                    suppressWarnings({

                    result <- try(afex::aov_car(modelFormula, data, type=self$options$ss, factorize = FALSE), silent=TRUE)

                    }) # suppressWarnings
                }) # suppressMessages

                if (isError(result))
                    jmvcore::reject(extractErrorMessage(result), code='error')

                private$.populateEffectsTables(result)
                private$.populateSpericityTable(result)
                private$.populateLeveneTable()
                private$.populatePostHocTables(result)
                private$.prepareDescPlots(data)
            }
        },
        # Initings tables/plots functions
        #####################################
        .initRMTable=function() {
            rmTable <- self$results$get('rmTable')
            rmTable$setNote('Note', jmvcore::format("Type {} Sums of Squares", self$options$ss))

            rm <- private$.rmTerms()
            rmTerms <- rm$terms
            rmSpacing <- rm$spacing

            if (length(rmTerms) > 0) {
                for (i in seq_along(rmTerms)) {
                    name <- stringifyTerm(rmTerms[i])
                    values <- list(
                        `name[none]`=name,
                        `name[GG]`=name,
                        `name[HF]`=name)
                    if (rmTerms[i] == 'Residual')
                        key <- unlist(c(rmTerms[i-1],'.RES'))
                    else
                        key <- unlist(rmTerms[i])
                    rmTable$addRow(rowKey=key, values)
                }
            } else {
                name <- '.'
                values <- list(
                    `name[none]`=name,
                    `name[GG]`=name,
                    `name[HF]`=name)
                rmTable$addRow(rowKey='.', values)
                rmTable$addRow(rowKey='', list(name='Residual'))
            }

            for (i in seq_along(rmSpacing)) {
                if ( ! is.null(rmSpacing[[i]])) {
                    if (rmSpacing[[i]] == 'both')
                        rmTable$addFormat(rowNo=i, col=1, Cell.BEGIN_END_GROUP)
                    else if (rmSpacing[[i]] == 'above')
                        rmTable$addFormat(rowNo=i, col=1, Cell.BEGIN_GROUP)
                    else if (rmSpacing[[i]] == 'below')
                        rmTable$addFormat(rowNo=i, col=1, Cell.END_GROUP)
                }
            }
        },
        .initBSTable=function() {
            bsTable <- self$results$get('bsTable')
            bsTable$setNote('Note', jmvcore::format("Type {} Sums of Squares", self$options$ss))

            bsTerms <- private$.bsTerms()
            if (length(bsTerms) > 0) {
                for (term in bsTerms)
                    bsTable$addRow(rowKey=unlist(term), list(name=stringifyTerm(term)))
            } else {
                bsTable$addRow(rowKey='', list(name='Residual'))
            }
        },
        .initSpericityTable=function() {
            spherTable <- self$results$get('assump')$get('spherTable')
            for (term in self$options$rmTerms)
                spherTable$addRow(rowKey=term, list(name=stringifyTerm(term)))
        },
        .initLeveneTable=function() {
            leveneTable <- self$results$get('assump')$get('leveneTable')
            rmVars <- sapply(self$options$rmCells, function(x) return(x$measure))
            for (var in rmVars)
                leveneTable$addRow(rowKey=var, list(name=var))
        },
        .initPostHocTables=function() {

            bs <- self$options$bs
            rm <- self$options$rm
            phTerms <- self$options$postHoc

            bsLevels <- list()
            for (i in seq_along(bs))
                bsLevels[[bs[i]]] <- levels(self$data[[bs[i]]])

            rmVars <- sapply(rm, function(x) return(x$label))
            rmLevels <- list()
            for (i in seq_along(rmVars))
                rmLevels[[rmVars[i]]] <- rm[[i]]$levels

            allLevels <- c(bsLevels, rmLevels)
            tables <- self$results$postHoc

            postHocRows <- list()

            for (ph in phTerms) {

                table <- tables$get(key=ph)

                table$setTitle(paste0('Post Hoc Comparisons - ', stringifyTerm(ph)))

                for (i in seq_along(ph))
                    table$addColumn(name=paste0(ph[i],'1'), title=ph[i], type='text', superTitle='Comparison', combineBelow=TRUE)

                table$addColumn(name='sep', title='', type='text', content='-', superTitle='Comparison', format='narrow')

                for (i in seq_along(ph))
                    table$addColumn(name=paste0(ph[i],'2'), title=ph[i], type='text', superTitle='Comparison')

                table$addColumn(name='md', title='Mean Difference', type='number')
                table$addColumn(name='se', title='SE', type='number')
                table$addColumn(name='df', title='df', type='number')
                table$addColumn(name='t', title='t', type='number')

                table$addColumn(name='pnone', title='p', type='number', format='zto,pvalue', visible="(postHocCorr:none)")
                table$addColumn(name='ptukey', title='p<sub>tukey</sub>', type='number', format='zto,pvalue', visible="(postHocCorr:tukey)")
                table$addColumn(name='pscheffe', title='p<sub>sheffe</sub>', type='number', format='zto,pvalue', visible="(postHocCorr:scheffe)")
                table$addColumn(name='pbonferroni', title='p<sub>bonferroni</sub>', type='number', format='zto,pvalue', visible="(postHocCorr:bonf)")
                table$addColumn(name='pholm', title='p<sub>holm</sub>', type='number', format='zto,pvalue', visible="(postHocCorr:holm)")

                combin <- expand.grid(allLevels[rev(ph)])
                combin <- sapply(combin, as.character, simplify = 'matrix')
                if (length(ph) > 1)
                    combin <- combin[,rev(1:length(combin[1,]))]

                comp <- list()
                iter <- 1
                for (i in 1:(length(combin[,1]) - 1)) {
                    for (j in (i+1):length(combin[,1])) {
                        comp[[iter]] <- list()
                        comp[[iter]][[1]] <- combin[i,]
                        comp[[iter]][[2]] <- combin[j,]

                        if (j == length(combin[,1]))
                            comp[[iter]][[3]] <- TRUE
                        else
                            comp[[iter]][[3]] <- FALSE

                        iter <- iter + 1
                    }
                }

                postHocRows[[composeTerm(ph)]] <- comp

                for (i in seq_along(comp)) {
                    row <- list()
                    for (c in seq_along(comp[[i]][[1]]))
                        row[[paste0(names(comp[[i]][[1]][c]),'1')]] <- as.character(comp[[i]][[1]][c])
                    for (c in seq_along(comp[[i]][[2]]))
                        row[[paste0(names(comp[[i]][[2]][c]),'2')]] <- as.character(comp[[i]][[2]][c])

                    table$addRow(rowKey=i, row)
                    if (comp[[i]][[3]] == TRUE)
                        table$addFormat(rowNo=i, col=1, Cell.END_GROUP)
                }
            }
            private$.postHocRows <- postHocRows
        },
        .initDescPlots=function() {
            isAxis <- ! is.null(self$options$plotHAxis)
            isMulti <- ! is.null(self$options$plotSepPlots)

            self$results$get('descPlot')$setVisible( ! isMulti && isAxis)
            self$results$get('descPlots')$setVisible(isMulti)

            if (isMulti) {

                sepPlotsName <- self$options$plotSepPlots

                if (sepPlotsName %in% self$options$bs) {
                    sepPlotsVar <- self$data[[sepPlotsName]]
                    sepPlotsLevels <- levels(sepPlotsVar)
                } else  {
                    rmLabels <- sapply(self$options$rm, function(x) return(x$label))
                    rmLevels <- lapply(self$options$rm, function(x) return(x$levels))
                    sepPlotsLevels <- rmLevels[[which(rmLabels %in% sepPlotsName)]]
                }

                array <- self$results$descPlots

                for (level in sepPlotsLevels)
                    array$addItem(level)
            }
        },
        # Populating tables/plots functions
        #####################################
        .populateEffectsTables=function(result) {

            rmTable <- self$results$get('rmTable')
            bsTable <- self$results$get('bsTable')

            suppressWarnings({
                summaryResult <- summary(result)
            })
            model <- summaryResult$univariate.tests
            epsilon <- summaryResult$pval.adjustments

            rmRows <- rmTable$rowKeys
            bsRows <- bsTable$rowKeys
            modelRows <- jmvcore::decomposeTerms(as.list(rownames(model)))
            epsilonRows <- jmvcore::decomposeTerms(as.list(rownames(epsilon)))
            resIndices <- which(sapply(rmRows, function(x) '.RES' %in% x))

            # Populate RM table
            for (i in seq_along(rmRows)) {

                if (! '.RES' %in% rmRows[[i]]) { # if the row is not a residual

                    index <- which(sapply(modelRows, function(x) setequal(toB64(rmRows[[i]]), x)))

                    row <- list()
                    row[['ss[none]']] <- row[['ss[GG]']] <- row[['ss[HF]']] <- model[index,'SS']
                    row[['F[none]']] <- row[['F[GG]']] <- row[['F[HF]']] <- model[index,'F']

                    row[['df[none]']] <- model[index,'num Df']
                    row[['ms[none]']] <- row[['ss[none]']] / row[['df[none]']]
                    row[['p[none]']] <- model[index,'Pr(>F)']

                    # Add sphericity corrected values
                    indexEps <- which(sapply(epsilonRows, function(x) setequal(toB64(rmRows[[i]]), x)))
                    dfRes <- model[index,'den Df']

                    if (length(indexEps) == 0) {
                        GG <- 1
                        HF <- 1
                    } else {
                        GG <- if (is.na(epsilon[indexEps,'GG eps'])) 1 else epsilon[indexEps,'GG eps']
                        HF <- if (is.na(epsilon[indexEps,'HF eps'])) 1 else epsilon[indexEps,'HF eps']
                    }

                    row[['df[GG]']] <- row[['df[none]']] * GG
                    row[['ms[GG]']] <- row[['ss[GG]']] / row[['df[GG]']]
                    dfResGG <- dfRes * GG
                    row[['p[GG]']] <- pf(row[['F[GG]']], row[['df[GG]']], dfResGG, lower.tail=FALSE)

                    row[['df[HF]']] <- row[['df[none]']] * HF
                    row[['ms[HF]']] <- row[['ss[HF]']] / row[['df[HF]']]
                    dfResHF <- dfRes * HF
                    row[['p[HF]']] <- pf(row[['F[HF]']], row[['df[HF]']], dfResHF, lower.tail=FALSE)

                    # Add effect sizes
                    termsTotal <- private$.getTermsTotal(i, rmRows, resIndices, model)
                    SSr <- model[index,'Error SS']
                    SSt <- termsTotal + SSr
                    MSr <- SSr/dfRes
                    row[['eta[none]']] <- row[['eta[GG]']] <- row[['eta[HF]']] <- row[['ss[none]']] / SSt
                    row[['partEta[none]']] <- row[['partEta[GG]']] <- row[['partEta[HF]']] <- row[['ss[none]']] / (row[['ss[none]']] + SSr)

                    omega <- (row[['ss[none]']] - (row[['df[none]']] * MSr)) / (SSt + MSr)

                    row[['omega[none]']] <- row[['omega[GG]']] <- row[['omega[HF]']] <- if ( ! is.na(omega) && omega < 0) 0 else omega

                    rmTable$setRow(rowNo=i, values=row)

                } else { # if the row is a residual

                    term <- rmRows[[i]][-length(rmRows[[i]])]
                    index <- which(sapply(modelRows, function(x) setequal(toB64(term), x)))

                    row <- list()
                    row[['ss[none]']] <- row[['ss[GG]']] <- row[['ss[HF]']] <- model[index,'Error SS']
                    row[['df[none]']] <- model[index,'den Df']
                    row[['ms[none]']] <- row[['ss[none]']] / row[['df[none]']]
                    row[['F[none]']] <- row[['F[GG]']]  <- row[['F[HF]']] <- ''
                    row[['p[none]']] <- row[['p[GG]']] <- row[['p[HF]']] <- ''
                    row[['eta[none]']] <- row[['eta[GG]']] <- row[['eta[HF]']] <- ''
                    row[['partEta[none]']] <- row[['partEta[GG]']] <- row[['partEta[HF]']] <- ''
                    row[['omega[none]']] <- row[['omega[GG]']] <- row[['omega[HF]']] <- ''

                    # Add sphericity corrected values
                    indexEps <- which(sapply(epsilonRows, function(x) setequal(toB64(term), x)))
                    dfRes <- model[index,'den Df']

                    if (length(indexEps) == 0) {
                        GG <- 1
                        HF <- 1
                    } else {
                        GG <- if (is.na(epsilon[indexEps,'GG eps'])) 1 else epsilon[indexEps,'GG eps']
                        HF <- if (is.na(epsilon[indexEps,'HF eps'])) 1 else epsilon[indexEps,'HF eps']
                    }

                    row[['df[GG]']] <- row[['df[none]']] * GG
                    row[['ms[GG]']] <- row[['ss[GG]']] / row[['df[GG]']]
                    row[['df[HF]']] <- row[['df[none]']] * HF
                    row[['ms[HF]']] <- row[['ss[HF]']] / row[['df[HF]']]

                    rmTable$setRow(rowNo=i, values=row)
                }
            }

            # Populate BS table
            bsTerms <- lapply(bsRows[which( ! sapply(bsRows, function(x) x[1] == 'Residual'))], toB64)
            if (length(bsTerms) > 0)
                bsIndices <- sapply(bsTerms, function(x) which(sapply(modelRows, function(y) setequal(x,y))))

            for (i in seq_along(bsRows)) {

                if (! bsRows[[i]][1] == 'Residual') { # if the row is not a residual

                    index <- which(sapply(modelRows, function(x) setequal(toB64(bsRows[[i]]), x)))

                    row <- list()
                    row[['ss']] <- model[index,'SS']
                    row[['df']] <- model[index,'num Df']
                    row[['ms']] <- row[['ss']] / row[['df']]
                    row[['F']] <- model[index,'F']
                    row[['p']] <- model[index,'Pr(>F)']

                    # Add effect sizes
                    SSr <- model[index,'Error SS']
                    SSt <- sum(model[bsIndices,'SS']) + SSr
                    MSr <- SSr/model[index,'den Df']
                    row[['eta']] <- row[['ss']] / SSt
                    row[['partEta']] <- row[['ss']] / (row[['ss']] + SSr)
                    omega <- (row[['ss']] - (row[['df']] * MSr)) / (SSt + MSr)
                    row[['omega']] <- if ( ! is.na(omega) && omega < 0) 0 else omega

                    bsTable$setRow(rowNo=i, values=row)

                } else { # if the row is a residual

                    row <- list()
                    row[['ss']] <- model['(Intercept)','Error SS']
                    row[['df']] <- model['(Intercept)','den Df']
                    row[['ms']] <- row[['ss']] / row[['df']]
                    row[['F']] <- row[['p']] <- row[['eta']] <- row[['partEta']] <- row[['omega']] <-''

                    bsTable$setRow(rowNo=i, values=row)
                }
            }
        },
        .populateSpericityTable=function(result) {

            spherTable <- self$results$get('assump')$get('spherTable')
            summaryResult <- suppressWarnings({summary(result)})
            epsilon <- summaryResult$pval.adjustments
            mauchly <- summaryResult$sphericity.tests

            nLevels <- sapply(self$options$rm, function(x) return(length(x$levels)))
            resultRows <- decomposeTerms(rownames(mauchly))

            if (any(nLevels > 2) && length(resultRows) > 0) {

                for (term in self$options$rmTerms) {

                    index <- which(sapply(as.list(resultRows), function(x) setequal(x, toB64(term))))

                    if (length(index) == 0) {

                        spherTable$setRow(rowKey=term, values=list('mauch'=1, 'p'=NaN, 'gg'=1, 'hf'=1))
                        if (length(spherTable$getRow(rowKey=term)$name$footnotes) == 0)
                            spherTable$addFootnote(rowKey=term, 'p', 'The repeated measures has only two levels. The assumption of sphericity is always met when the repeated measures has only two levels')

                    } else {

                        row <- list()
                        row[['mauch']] <- mauchly[index,'Test statistic']
                        row[['p']] <- mauchly[index,'p-value']
                        row[['gg']] <- epsilon[index, 'GG eps']
                        row[['hf']] <- if (epsilon[index, 'HF eps'] > 1) 1 else epsilon[index, 'HF eps']

                        spherTable$setRow(rowKey=term, values=row)
                    }
                }
            } else {

                for (term in self$options$rmTerms) {

                    spherTable$setRow(rowKey=term, values=list('mauch'=1, 'p'=NaN, 'gg'=1, 'hf'=1))
                    if (length(spherTable$getRow(rowKey=term)$name$footnotes) == 0)
                        spherTable$addFootnote(rowKey=term, 'p', 'The repeated measures has only two levels. The assumption of sphericity is always met when the repeated measures has only two levels')
                }
            }
        },
        .populateLeveneTable=function () {

            if (length(self$options$rmCells) == 0)
                return()

            leveneTable <- self$results$get('assump')$get('leveneTable')

            rmVars <- sapply(self$options$rmCells, function(x) return(x$measure))
            covVars <- self$options$cov
            bsVars <- self$options$bs

            if (length(bsVars) == 0) {
                for (var in rmVars) {
                    leveneTable$setRow(rowKey=var, values=list('F'=NaN, 'df1'='', 'df2'='', 'p'=''))
                    leveneTable$addFootnote(rowKey=var, 'F', 'As there are no between subjects factors specified this assumption is always met.')
                }
                return()
            }

            data <- list()
            for (rm in c(rmVars,covVars))
                data[[rm]] <- jmvcore::toNumeric(self$data[[rm]])

            for (bs in bsVars)
                data[[bs]] <- factor(self$data[[bs]])

            attr(data, 'row.names') <- seq_len(length(data[[1]]))
            attr(data, 'class') <- 'data.frame'
            data <- jmvcore::naOmit(data)

            group <- interaction(data[bsVars])
            data <- cbind(data, .GROUP=group)

            for (var in rmVars) {

                if (length(covVars) > 0)
                    formula <- as.formula(paste0(composeTerm(var),'~ .GROUP +', paste0(composeTerm(covVars), collapse='+')))
                else
                    formula <- as.formula(paste0(composeTerm(var),'~ .GROUP'))

                res <- abs(aov(formula, data=data)$residuals)
                r <- summary(aov(res ~ group))[[1]]

                row <- list(F=r[1,'F value'], df1=r[1,'Df'], df2=r[2,'Df'], p=r[1,'Pr(>F)'])
                leveneTable$setRow(rowKey=var, values=row)
            }
        },
        .populatePostHocTables=function (result) {

            terms <- self$options$postHoc

            if (length(terms) == 0)
                return()

            tables <- self$results$postHoc

            postHocRows <- list()

            for (ph in terms) {

                table <- tables$get(key=ph)

                term <- jmvcore::composeTerm(ph)
                termB64 <- jmvcore::composeTerm(toB64(ph))

                formula <- as.formula(paste('~', termB64))

                suppressWarnings({

                    table$setStatus('running')

                    referenceGrid <- lsmeans::lsmeans(result, formula)
                    none <- summary(pairs(referenceGrid, adjust='none'))
                    tukey <- summary(pairs(referenceGrid, adjust='tukey'))
                    scheffe <- summary(pairs(referenceGrid, adjust='scheffe'))
                    bonferroni <- summary(pairs(referenceGrid, adjust='bonferroni'))
                    holm <- summary(pairs(referenceGrid, adjust='holm'))

                }) # suppressWarnings

                resultRows <- lapply(strsplit(as.character(tukey$contrast), ' - '), function(x) strsplit(x, ','))
                tableRows <- private$.postHocRows[[term]]

                for (i in seq_along(tableRows)) {
                    location <- lapply(resultRows, function(x) {

                        c1 <- identical(x[[1]], toB64(as.character(tableRows[[i]][[1]])))
                        c2 <- identical(x[[1]], toB64(as.character(tableRows[[i]][[2]])))
                        c3 <- identical(x[[2]], toB64(as.character(tableRows[[i]][[1]])))
                        c4 <- identical(x[[2]], toB64(as.character(tableRows[[i]][[2]])))

                        if (c1 && c4)
                            return(list(TRUE,FALSE))
                        else if (c2 && c3)
                            return(list(TRUE,TRUE))
                        else
                            return(list(FALSE,FALSE))
                    })

                    index <- which(sapply(location, function(x) return(x[[1]])))
                    reverse <- location[[index]][[2]]

                    row <- list()
                    row[['md']] <- if(reverse) -tukey[index,'estimate'] else tukey[index,'estimate']
                    row[['se']] <- tukey[index,'SE']
                    row[['df']] <- tukey[index,'df']
                    row[['t']] <- if(reverse) -tukey[index,'t.ratio'] else tukey[index,'t.ratio']

                    row[['pnone']] <- none[index,'p.value']
                    row[['ptukey']] <- tukey[index,'p.value']
                    row[['pscheffe']] <- scheffe[index,'p.value']
                    row[['pbonferroni']] <- bonferroni[index,'p.value']
                    row[['pholm']] <- holm[index,'p.value']

                    table$setRow(rowNo=i, values=row)
                    private$.checkpoint()
                }
                table$setStatus('complete')
            }
        },
        .prepareDescPlots=function(data) {

            depName <- '.DEPENDENT'
            groupName <- self$options$plotHAxis
            linesName <- self$options$plotSepLines
            plotsName <- self$options$plotSepPlots

            ciWidth   <- self$options$ciWidth
            errorBarType <- self$options$plotError

            if (length(depName) == 0 || length(groupName) == 0)
                return()

            by <- list()
            by[['group']] <- data[[toB64(groupName)]]
            levels(by[['group']]) <- fromB64(levels(by[['group']]))

            if ( ! is.null(linesName)) {
                by[['lines']] <- data[[toB64(linesName)]]
                levels(by[['lines']]) <- fromB64(levels(by[['lines']]))
            }

            if ( ! is.null(plotsName)) {
                by[['plots']] <- data[[toB64(plotsName)]]
                levels(by[['plots']]) <- fromB64(levels(by[['plots']]))
            }


            dep <- data[[toB64(depName)]]

            ciMult <- qt(ciWidth / 200 + .5, nrow(data)-1)

            means <- aggregate(dep, by=by, mean, simplify=FALSE)
            ses   <- aggregate(dep, by=by, function(x) { sd(x) / sqrt(length(x)) }, simplify=FALSE)
            cis   <- aggregate(dep, by=by, function(x) { sd(x) / sqrt(length(x)) * ciMult }, simplify=FALSE)

            plotData <- data.frame(group=means$group)
            if ( ! is.null(linesName))
                plotData <- cbind(plotData, lines=means$lines)
            if ( ! is.null(plotsName))
                plotData <- cbind(plotData, plots=means$plots)

            plotData <- cbind(plotData, mean=unlist(means$x))

            if (errorBarType == 'ci')
                plotData <- cbind(plotData, err=unlist(cis$x))
            else
                plotData <- cbind(plotData, err=unlist(ses$x))

            plotData <- cbind(plotData, lower=plotData$mean-plotData$err, upper=plotData$mean+plotData$err)

            if (self$options$plotError != 'none') {
                yAxisRange <- pretty(c(plotData$lower, plotData$upper))
            } else {
                yAxisRange <- plotData$mean
            }

            if (is.null(plotsName)) {

                image <- self$results$get('descPlot')
                image$setState(list(data=plotData, range=yAxisRange))

            } else {

                images <- self$results$descPlots
                for (level in images$itemKeys) {
                    image <- images$get(key=level)
                    image$setState(list(data=subset(plotData, plots == level), range=yAxisRange))
                }
            }
        },
        .descPlot=function(image, ...) {

            if (is.null(image$state))
                return(FALSE)

            groupName <- self$options$plotHAxis
            linesName <- self$options$plotSepLines
            plotsName <- self$options$plotSepPlots

            the <- theme(
                text=element_text(size=16, colour='#333333'),
                plot.background=element_rect(fill='transparent', color=NA),
                legend.background=element_rect(fill='transparent', colour=NA),
                panel.background=element_rect(fill='#E8E8E8'),
                plot.margin=margin(15, 15, 15, 15),
                axis.text.x=element_text(margin=margin(5,0,0,0)),
                axis.text.y=element_text(margin=margin(0,5,0,0)),
                axis.title.x=element_text(margin=margin(10,0,0,0)),
                axis.title.y=element_text(margin=margin(0,10,0,0)))

            if (self$options$plotError != 'none')
                dodge <- position_dodge(0.2)
            else
                dodge <- position_dodge(0)


            errorType <- ''
            if (self$options$plotError != 'none') {
                if (self$options$plotError == 'ci') {
                    ciWidth <- self$options$ciWidth
                    errorType <- paste0('(', ciWidth, '% CI)')
                } else {
                    errorType <- '(SE)'
                }
            }

            if ( ! is.null(linesName)) {

                p <- ggplot(data=image$state$data, aes(x=group, y=mean, group=lines, colour=lines)) +
                    geom_line(size=.8, position=dodge) +
                    labs(x=groupName, y="", colour=paste(linesName, errorType)) +
                    scale_y_continuous(limits=c(min(image$state$range), max(image$state$range))) +
                    the

                if (self$options$plotError != 'none')
                    p <- p + geom_errorbar(aes(x=group, ymin=lower, ymax=upper, width=.1, group=lines), size=.8, position=dodge)

                p <- p + geom_point(shape=21, fill='white', size=3, position=dodge)

                print(p)

            } else {

                p <- ggplot(data=image$state$data) +
                    labs(x=groupName, y="", colour=paste("", errorType)) +
                    scale_colour_manual(name=paste("", errorType), values=c(colour='#333333'), labels='') +
                    scale_y_continuous(limits=c(min(image$state$range), max(image$state$range))) +
                    the

                if (self$options$plotError != 'none')
                    p <- p + geom_errorbar(aes(x=group, ymin=lower, ymax=upper, colour='colour', width=.1), size=.8)

                p <- p + geom_point(aes(x=group, y=mean, colour='colour'), shape=21, fill='white', size=3)

                print(p)
            }

            TRUE
        },
        # Helper functions
        #####################################
        .dataCheck=function() {

            data <- self$data

            rm <- sapply(self$options$rmCells, function(x) return(x$measure))
            bs <- unlist(self$options$bs)
            cov <- unlist(self$opions$cov)

            varsNumeric <- c(rm, cov)

            dataFactors <- list()
            for (i in seq_along(bs))
                dataFactors[[bs[i]]] <- data[[bs[i]]]

            dataNumeric <- list()
            for (i in seq_along(varsNumeric))
                dataNumeric[[varsNumeric[i]]] <- data[[varsNumeric[i]]]

            # Check all values
            allNAItems <- sapply(c(dataFactors, dataNumeric), function(x) all(is.na(x)))
            if (any(allNAItems))
                jmvcore::reject("Item '{}' contains only missing values", code='error', c(bs,varsNumeric)[allNAItems])

            # Check factor values
            singleLevelItems <- sapply(dataFactors, function(x) length(levels(x)) == 1)
            if (any(singleLevelItems))
                jmvcore::reject("Item '{}' consists of one level only", code='error', bs[singleLevelItems])

            # Check numeric values
            factorItems <- sapply(dataNumeric, function(x) class(jmvcore::toNumeric(x)) == "factor")
            infItems <- sapply(dataNumeric, function(x) any(is.infinite(x)))
            noVarItems <- sapply(dataNumeric, function(x) var(x, na.rm = TRUE) == 0)
            if (any(factorItems))
                jmvcore::reject("Item '{}' needs to be numeric", code='error', varsNumeric[factorItems])
            if (any(infItems))
                jmvcore::reject("Item '{}' contains infinite values", code='error', varsNumeric[infItems])
            if (any(noVarItems))
                jmvcore::reject("Item '{}' has no variance", code='error', varsNumeric[noVarItems])
        },
        .rmTerms=function() {

            if (length(self$options$rmTerms) == 0) { # if no specific model is specified

                rmFactors <- self$options$rm
                bsFactors <- self$options$bs
                covariates <- self$options$cov

                if (length(rmFactors) == 0)
                    rmFactors <- list(list(label='RM Factor 1'))

                bsNames <- c(bsFactors, covariates)

                rmNames <- sapply(rmFactors, function(x) x$label, simplify=TRUE)
                rmFormula <- as.formula(paste('~', paste(paste0('`', rmNames, '`'), collapse='*')))
                rmTerms <- attr(stats::terms(rmFormula), 'term.labels')
                rmTerms <- sapply(rmTerms, function(x) as.list(strsplit(x, ':')), USE.NAMES=FALSE)

                if (length(bsFactors) > 0) {
                    bsFormula <- as.formula(paste('~', paste(paste0('`', bsFactors, '`'), collapse='*')))
                    bsTerms <- attr(stats::terms(bsFormula), 'term.labels')
                    bsTerms <- sapply(bsTerms, function(x) as.list(strsplit(x, ':')), USE.NAMES=FALSE)
                } else {
                    bsTerms <- NULL
                }

                terms <- list()
                spacing <- list()

                for (i in seq_along(rmTerms)) {

                    rmTerm <- rmTerms[[i]]
                    terms[[length(terms)+1]] <- rmTerm

                    for (j in seq_along(bsTerms))
                        terms[[length(terms)+1]] <- c(rmTerm, bsTerms[[j]])

                    terms[[length(terms)+1]] <- 'Residual'
                    spacing[[length(terms)]] <- 'below'
                }

            } else { # if the user specifies a model

                rmTerms <- self$options$rmTerms
                bsTerms <- self$options$bsTerms

                terms <- list()
                spacing <- list()

                for (i in seq_along(rmTerms)) {

                    rmTerm <- rmTerms[[i]]
                    terms[[length(terms) + 1]] <- rmTerm
                    spacing[[length(terms)]] <- 'above'

                    for (j in seq_along(bsTerms))
                        terms[[length(terms) + 1]] <- c(rmTerm, bsTerms[[j]])

                    terms[[length(terms) + 1]] <- 'Residual'
                    spacing[[length(terms)]] <- 'below'
                }
            }

            return(list(terms = terms, spacing = spacing))
        },
        .bsTerms=function() {

            if (length(self$options$bsTerms) == 0 && length(self$options$rmTerms) == 0) { # if no specific model is specified

                bsFactors <- self$options$bs
                covariates <- self$options$cov

                if (length(bsFactors) > 0) {
                    bsFormula <- as.formula(paste('~', paste(paste0('`', bsFactors, '`'), collapse='*')))
                    bsTerms <- attr(stats::terms(bsFormula), 'term.labels')
                    bsTerms <- sapply(bsTerms, function(x) as.list(strsplit(x, ':')), USE.NAMES=FALSE)
                } else {
                    bsTerms <- list()
                }

                terms <- bsTerms

                for (i  in seq_along(covariates))
                    terms[[length(terms) + 1]] <- covariates[i]

                terms[[length(terms) + 1]] <- 'Residual'

            } else { # if the user specifies a model

                bsTerms <- self$options$bsTerms
                bsFactors <- self$options$bs
                covariates <- self$options$cov

                terms <- list()

                # terms that include covariates:
                covTerms <- c()
                if (length(covariates) > 0)
                    covTerms <- apply(sapply(as.list(covariates), function (y) sapply(bsTerms, function(x) y %in% x)), 1, any)

                if (sum(covTerms) != length(covariates) || length(covTerms) == 0) {

                    terms <- c(terms, bsTerms)
                    terms[[length(terms) + 1]] <- 'Residual'

                } else {

                    terms <- c(terms, bsTerms[ ! covTerms])

                    for (i in seq_along(covariates))
                        terms[[length(terms) + 1]] <- covariates[[i]]

                    terms[[length(terms) + 1]] <- 'Residual'

                }
            }

            return(terms)
        },
        .wideToLong=function() {

            rmVars <- sapply(self$options$rmCells, function(x) return(x$measure))
            bsVars <- self$options$bs
            covVars <- self$options$cov

            labels <- sapply(self$options$rm, function(x) return(x$label))
            levels <- lapply(self$options$rm, function(x) return(x$levels))
            rmCells <- lapply(self$options$rmCells, function(x) return(x$cell))

            data <- list()
            for (var in c(rmVars, covVars))
                data[[var]] <- jmvcore::toNumeric(self$data[[var]])

            for (var in bsVars)
                data[[var]] <- self$data[[var]]

            attr(data, 'row.names') <- seq_len(length(data[[1]]))
            attr(data, 'class') <- 'data.frame'
            data <- jmvcore::naOmit(data)

            data <- cbind(data, '.SUBJECT'=1:nrow(data))

            data_long <- as.list(reshape2::melt(data, id.vars=c(bsVars, covVars, '.SUBJECT'), measure.vars=rmVars, value.name='.DEPENDENT'))

            col <- data_long[['variable']]
            temp <- numeric(length(col))
            for (j in seq_along(col))
                temp[j] <- which(rmVars %in% col[j])

            for (i in seq_along(labels))
                data_long[[labels[[i]]]] <- factor(sapply(rmCells[temp], function(x) x[i]), levels[[i]])

            data_long[['variable']] <- NULL

            data_long <- lapply(data_long, function(x) {
                if (is.factor(x))
                    levels(x) <- toB64(levels(x))
                return(x)
            })

            attr(data_long, 'row.names') <- seq_len(length(data_long[[1]]))
            attr(data_long, 'names') <- toB64(names(data_long))
            attr(data_long, 'class') <- 'data.frame'
            data_long <- jmvcore::naOmit(data_long)

            return(data_long)
        },
        .modelFormula=function() {

            if (is.null(self$options$rmTerms)) {



            } else {

                bsTerms <- lapply(self$options$bsTerms, function(x) toB64(x))
                rmTerms <- lapply(self$options$rmTerms, function(x) toB64(x))

                bsItems <- composeTerms(bsTerms)
                bsTerm <- paste0('(', paste0(bsItems, collapse = ' + '), ')')

                rmItems <- composeTerms(rmTerms)
                rmTerm <- paste0('Error(', paste0(toB64('.SUBJECT'),'/(', rmItems, ')', collapse=' + '),')')

                allTerms <- c(bsTerms, rmTerms)
                for (term1 in rmTerms) {
                    for (term2 in bsTerms) {
                        allTerms[[length(allTerms) + 1]] <- unlist(c(term1, term2))
                    }
                }

                allItems <- composeTerms(allTerms)
                mainTerm <- paste0('(', paste0(allItems, collapse = ' + '), ')')

                if (length(self$options$bsTerms) == 0) {
                    formula <- as.formula(paste(toB64('.DEPENDENT'), '~', paste(mainTerm, rmTerm, sep=' + ')))
                } else {
                    formula <- as.formula(paste(toB64('.DEPENDENT'), '~', paste(mainTerm, rmTerm, bsTerm, sep=' + ')))
                }

                return(formula)

            }
        },
        .summarySE=function(data=NULL, measurevar, groupvars=NULL, na.rm=FALSE, conf.interval=.95, .drop=TRUE, errorBarType='confidenceInterval') {

            # New version of length which can handle NA's: if na.rm==T, don't count them
            length2 <- function (x, na.rm=FALSE) {
                if (na.rm) {
                    sum(!is.na(x))
                } else {
                    length(x)
                }
            }

            # This does the summary. For each group's data frame, return a vector with
            # N, mean, and sd
            datac <- plyr::ddply(data, groupvars, .drop=.drop,
                                 .fun = function(xx, col) {
                                     c(N    = length2(xx[[col]], na.rm=na.rm),
                                       mean = mean   (xx[[col]], na.rm=na.rm),
                                       sd   = sd     (xx[[col]], na.rm=na.rm)
                                     )
                                 },
                                 measurevar
            )

            # Rename the 'mean' column
            datac <- plyr::rename(datac, c('mean' = measurevar))

            datac$se <- datac$sd / sqrt(datac$N)  # Calculate standard error of the mean

            # Confidence interval multiplier for standard error
            # Calculate t-statistic for confidence interval:
            # e.g., if conf.interval is .95, use .975 (above/below), and use df=N-1
            ciMult <- qt(conf.interval/2 + .5, datac$N-1)
            datac$ci <- datac$se * ciMult

            if (errorBarType == 'confidenceInterval') {

                datac$ciLower <- datac[,measurevar] - datac[,'ci']
                datac$ciUpper <- datac[,measurevar] + datac[,'ci']

            } else {

                datac$ciLower <- datac[,measurevar] - datac[,'se']
                datac$ciUpper <- datac[,measurevar] + datac[,'se']

            }

            return(datac)
        },
        .normDataWithin=function(data=NULL, idvar, measurevar, betweenvars=NULL, na.rm=FALSE, .drop=TRUE) {

            # Measure var on left, idvar + between vars on right of formula.
            data.subjMean <- plyr::ddply(data, c(idvar, betweenvars), .drop=.drop,
                                         .fun = function(xx, col, na.rm) {
                                             c(subjMean = mean(xx[,col], na.rm=na.rm))
                                         },
                                         measurevar,
                                         na.rm
            )



            # Put the subject means with original data
            data <- base::merge(data, data.subjMean)

            # Get the normalized data in a new column
            measureNormedVar <- paste(measurevar, '_norm', sep='')
            data[,measureNormedVar] <- data[,measurevar] - data[,'subjMean'] +
                mean(data[,measurevar], na.rm=na.rm)

            # Remove this subject mean column
            data$subjMean <- NULL

            return(data)
        },
        .summarySEwithin=function(data=NULL, measurevar, betweenvars=NULL, withinvars=NULL, idvar=NULL, na.rm=FALSE, conf.interval=.95, .drop=TRUE, errorBarType='confidenceInterval') {

            # Get the means from the un-normed data
            datac <- .summarySE(data, measurevar, groupvars=c(betweenvars, withinvars), na.rm=na.rm, conf.interval=conf.interval, .drop=.drop, errorBarType=errorBarType)

            # Drop all the unused columns (these will be calculated with normed data)
            datac$sd <- NULL
            datac$se <- NULL
            datac$ci <- NULL
            datac$ciLower <- NULL
            datac$ciUpper <- NULL

            # Norm each subject's data
            ndata <- .normDataWithin(data, idvar, measurevar, betweenvars, na.rm, .drop=.drop)

            # This is the name of the new column
            measurevar_n <- paste(measurevar, '_norm', sep='')

            # Collapse the normed data - now we can treat between and within vars the same
            ndatac <- .summarySE(ndata, measurevar_n, groupvars=c(betweenvars, withinvars), na.rm=na.rm, conf.interval=conf.interval, .drop=.drop, errorBarType=errorBarType)

            # Apply correction from Morey (2008) to the standard error and confidence interval
            # Get the product of the number of conditions of within-S variables
            nWithinGroups    <- prod(vapply(ndatac[,withinvars, drop=FALSE], FUN=nlevels, FUN.VALUE=numeric(1)))
            correctionFactor <- sqrt( nWithinGroups / (nWithinGroups-1) )

            # Apply the correction factor
            ndatac$sd <- ndatac$sd * correctionFactor
            ndatac$se <- ndatac$se * correctionFactor
            ndatac$ci <- ndatac$ci * correctionFactor

            if (errorBarType == 'confidenceInterval') {

                ndatac$ciLower <- datac[,measurevar] - ndatac[,'ci']
                ndatac$ciUpper <- datac[,measurevar] + ndatac[,'ci']

            } else {

                ndatac$ciLower <- datac[,measurevar] - ndatac[,'se']
                ndatac$ciUpper <- datac[,measurevar] + ndatac[,'se']

            }

            # Combine the un-normed means with the normed results
            merge(datac, ndatac)
        },
        .getTermsTotal=function (x, y, z, model) {
            for(i in x:1) {
                if (i %in% z) {
                    lower <- i + 1
                    break
                } else if (i == 1) {
                    lower <- i
                    break
                }
            }
            for(i in x:length(y)) {
                if (i %in% z) {
                    upper <- i - 1
                    break
                } else if (i == length(y)) {
                    upper <- i
                    break
                }
            }
            termsIndex <- lapply(y[lower:upper], toB64)
            modelRows <- jmvcore::decomposeTerms(as.list(rownames(model)))
            indicesTotal <- sapply(termsIndex, function(x) which(sapply(modelRows, function(y) setequal(y,x))))

            termsTotal <- sum(model[indicesTotal,'SS'])

            return(termsTotal)
        })
)
