% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/solve_dermal_pbtk.R
\name{solve_dermal_pbtk}
\alias{solve_dermal_pbtk}
\title{Solve_dermal_PBTK}
\usage{
solve_dermal_pbtk(
  chem.name = NULL,
  chem.cas = NULL,
  dtxsid = NULL,
  model.type = "dermal_1subcomp",
  method.permeability = "UK-Surrey",
  Kvehicle2water = NULL,
  times = NULL,
  parameters = NULL,
  days = 10,
  tsteps = 4,
  plots = FALSE,
  monitor.vars = NULL,
  suppress.messages = F,
  species = "Human",
  method = NULL,
  rtol = 1e-06,
  atol = 1e-06,
  recalc.blood2plasma = FALSE,
  recalc.clearance = FALSE,
  adjusted.Funbound.plasma = TRUE,
  minimum.Funbound.plasma = 1e-04,
  parameterize.arg.list = list(default.to.human = FALSE, clint.pvalue.threshold = 0.05,
    restrictive.clearance = TRUE, regression = TRUE),
  route = NULL,
  Vvehicle = NULL,
  initial.dose = NULL,
  daily.dose = NULL,
  doses.per.day = NULL,
  input.units = NULL,
  dose.duration = NULL,
  dose.duration.units = NULL,
  dosing.dermal = NULL,
  dosing.matrix = NULL,
  washoff = FALSE,
  InfiniteDose = FALSE,
  ...
)
}
\arguments{
\item{chem.name}{Either the chemical name, CAS number, or the parameters
must be specified.}

\item{chem.cas}{Either the chemical name, CAS number, or the parameters must
be specified.}

\item{dtxsid}{EPA's DSSTox Structure ID (\url{https://comptox.epa.gov/dashboard})
the chemical must be identified by either CAS, name, or DTXSIDs.}

\item{model.type}{Choice of dermal model, either the default "dermal_1subcomp" for
the model with 1 compartment for the skin; or "dermal" for the 
model with 2 sub compartments for skin: the stratum corneum (SC) and the combined
viable epidermis and dermis (ED).}

\item{method.permeability}{For "dermal_1subcomp" model, method of calculating 
the permeability coefficient, P, either "Potts-Guy" or "UK-Surrey". Default
is "UK-Surrey" (Sawyer et al., 2016 and Chen et al., 2015), which uses Fick's
law of diffusion to calculate P. For "dermal" model, this parameter is ignored.}

\item{Kvehicle2water}{Partition coefficient for the vehicle (sometimes called the 
media) carrying the chemical to water. Default is "water", which assumes the vehicle is water.
Other optional inputs are "octanol" and "olive oil".}

\item{times}{Optional time sequence for specified number of days.  Dosing
sequence begins at the beginning of times.}

\item{parameters}{Chemical parameters from parameterize_dermal_pbtk function,
overrides chem.name and chem.cas.}

\item{days}{Length of the simulation.If "times" input is used, this is ignored.}

\item{tsteps}{The number time steps per hour.}

\item{plots}{Plots all outputs if true.}

\item{monitor.vars}{Which variables are returned as a function of time. 
Default values of NULL looks up variables specified in modelinfo_MODEL.R}

\item{suppress.messages}{Whether or not the output message is suppressed.}

\item{species}{Species desired (either "Rat", "Rabbit", "Dog", "Mouse", or
default "Human").}

\item{method}{Method used by integrator (\code{\link[deSolve]{ode}}).}

\item{rtol}{Argument passed to integrator (\code{\link[deSolve]{ode}}).}

\item{atol}{Argument passed to integrator (\code{\link[deSolve]{ode}}).}

\item{recalc.blood2plasma}{Recalculates the ratio of the amount of chemical
in the blood to plasma using the input parameters, calculated with
hematocrit, Funbound.plasma, and Krbc2pu.}

\item{recalc.clearance}{Recalculates the the hepatic clearance
(Clmetabolism) with new million.cells.per.gliver parameter.}

\item{adjusted.Funbound.plasma}{Uses adjusted Funbound.plasma when set to
TRUE along with partition coefficients calculated with this value.}

\item{minimum.Funbound.plasma}{Monte Carlo draws less than this value are set 
equal to this value (default is 0.0001 -- half the lowest measured Fup in our
dataset).}

\item{parameterize.arg.list}{Additional parameterized passed to the model 
parameterization function, "parameterize_dermal_pbtk". The inputs "model.type",
"method.permeability", and "Kvehicle2water" are not passed through this.}

\item{route}{Route of exposure, can be "oral" OR "iv" OR "dermal" (default).}

\item{Vvehicle}{Volume of vehicle applied to skin in L, defaults to 0.1 L. If 
InfiniteDose=TRUE, this parameter is ignored and set = 1.}

\item{initial.dose}{Initial exposure dose. If InfiniteDose=TRUE, this is a concentration, 
otherwise, this is an amount.}

\item{daily.dose}{Total daily dose, defaults to mg/kg BW.}

\item{doses.per.day}{Number of doses per day.}

\item{input.units}{Exposure units applied to initial.dose and/or dosing.dermal.
If InfiniteDose=TRUE, must be a concentration, e.g., "mg/kg/L" (default), otherwise,
must be an amount, e.g., "mg/kg" (default).}

\item{dose.duration}{Amount of time dermal dose is on skin before being washed off.
Note that when dose.duration is used, washoff=TRUE.}

\item{dose.duration.units}{Units for dose.duration, can be "minutes" OR "hours" 
OR "days" (default).}

\item{dosing.dermal}{Matrix consisting of three columns named
"Cvehicle", "Vvehicle", and "time" containing the dosing times, days,
with the applied amount in the vehicle, and the volume of the applied
vehicle, L. Note that the units of Cvehicle are controlled by input.units. **If
InfiniteDose=TRUE, the Vvehicle column of dosing.dermal is ignored.**}

\item{dosing.matrix}{Vector of dosing times or a matrix consisting of two
columns or rows named "dose" and "time" containing the time and amount, in
mg/kg BW, of each dose.}

\item{washoff}{If TRUE, any chemical left on the skin is assumed to be replaced 
by new dose  (i.e., wash-off occurs before new dose is administered). If FALSE 
(default), any chemical left on the skin is added to the new dose.}

\item{InfiniteDose}{Is there so much compound in the vehicle that it does not deplete?}

\item{...}{Additional arguments passed to the integrator (\code{\link[deSolve]{ode}}).}
}
\value{
A matrix of class deSolve with a column for time (in days), each
compartment, the area under the curve, and plasma concentration and a row
for each time point.
}
\description{
This function solves for the amounts or concentrations in uM of a chemical in
different tissues as functions of time after dermal exposure.
}
\details{
The user can input            
dermal doses via one of three options:
\describe{
  \item{"dose.duration"}{ User can input the length of exposure time for one 
dermal dose before wash-off occurs. Note that initial.dose can be used to 
change the initial dose used along with this option.}
  \item{"dosing.dermal"}{ With this option, users can input multiple doses 
over time as a matrix with columns for time, the volume of vehicle 
administered, and the concentration of the vehicle administered. Note that 
the the parameter washoff can be used to specify whether chemical is washed 
off in between doses.}
  \item{"dosing.matrix"}{ This option is also used to describe multiple 
exposure doses over time, and is described in the help file of solve_model. 
Note that unlike dosing.dermal, Vvehicle cannot be changed with this option.}
}

Model units are the same as vehicle concentration, units/L or units when
use.amounts=TRUE.

New doses replace rather than add to previous doses. A concentration of 0 in
dosing.matrix switches off the dosing/diffusion between the vehicle and
exposed skin.

Note that the model parameters have units of hours while the model output is
in days.

The compartments used in this model are the gutlumen, gut, liver, kidneys,
veins, arteries, lungs, unexposed skin, exposed skin, vehicle, and the rest of the body. When model.type 
= "dermal", a 2-compartment model is used where skin
is divided into the stratum corneum, SC, and the combined viable epidermis and
dermis, ED.

The extra compartments include the amounts or concentrations metabolized by
the liver and excreted by the kidneys through the tubules.

AUC is the area under the curve of the plasma concentration.

When species is specified as rabbit, dog, or mouse, the function uses the
appropriate physiological data(volumes and flows) but substitutes human
fraction unbound, partition coefficients, and intrinsic hepatic clearance.
}
\examples{

# Dermal exposure to default dose
out <- solve_dermal_pbtk(chem.name="bisphenola")

\donttest{
# Dermal exposure to 20 mg/L in 0.01 L of octanol with wash-off after 8 hours
# Since skin permeability happens quickly for bisphenol A, let's only look at 3 days.
dose.conc <- 2 #mg/L
Vvehicle <- 0.01 #L
initial.dose <- dose.conc*Vvehicle
out <- solve_dermal_pbtk(chem.name="bisphenola", initial.dose=initial.dose, 
                         input.units="mg", Vvehicle=0.01, 
                         Kskin2vehicle="octanol", dose.duration=8, 
                         dose.duration.units="hr", days=3)

# Now, try this again with an infinite dose.
out <- solve_dermal_pbtk(chem.name="bisphenola", initial.dose=dose.conc, 
                         input.units="mg/L", Vvehicle=0.01, 
                         Kskin2vehicle="octanol", dose.duration=8, 
                         dose.duration.units="hr", days=3, 
                         InfiniteDose=TRUE)

# Now, try a scenario where 2 mg of chemical in 1 mL of water is applied 
# and washed off 8 hours later every day for 5 days
num.days <- 5;
time <- c(0:(num.days-1),(0:(num.days-1)) + 8/24); time <- sort(time) #in days
Vvehicle <- rep(1e-3,length(time)) #convert mL to L
Cvehicle <- rep(c(2,0),num.days)/Vvehicle # convert 2 mg to mg/L
dosing.dermal <- cbind(time,Cvehicle,Vvehicle)
out <- solve_dermal_pbtk(chem.name='bisphenola',
                         dosing.dermal=dosing.dermal)

parameters <- parameterize_dermal_pbtk(chem.name='bisphenola',skin_depth=1)
parameters$Fskin_exposed <- 0.25
parameters$Vvehicle <- 1
out <- solve_dermal_pbtk(parameters=parameters)

head(solve_dermal_pbtk(chem.name="propylparaben"))
head(solve_dermal_pbtk(chem.cas="94-13-3"))


p <- parameterize_dermal_pbtk(chem.name="propylparaben")
p <- p[sort(names(p))]
# Try to standardize order of variable names
for (this.param in 
     names(p)[order(toupper(names(p)))]) cat(
     paste(this.param,": ",p[[this.param]],"\r\n",sep=""))
head(solve_dermal_pbtk(parameters=p))
       
# Dermal is the default route:
head(solve_dermal_pbtk(chem.name="bisphenola"))
head(solve_dermal_pbtk(chem.name="bisphenola", route="dermal"))
# But we can also do intravenous (iv):
head(solve_dermal_pbtk(chem.name="bisphenola", route="iv"))
# And oral:
head(solve_dermal_pbtk(chem.name="bisphenola", route="oral"))
}

}
\author{
Annabel Meade, John Wambaugh, Celia Schacht, and Robert Pearce
}
\keyword{Solve}
