#' Run one or more Markov Model
#' 
#' Runs one or more unevaluated Markov Models. When more 
#' than one model is provided, all models should have the 
#' same states and state value names.
#' 
#' 
#' A usual situation where more than one model needs to be 
#' run is when comparing different care startegies.
#' 
#' In order to compute comparisons Markov Models must be 
#' similar (same states and state value names). Thus models 
#' should only differ through parameters, transition matrix 
#' cell values and values attached to states (but not state 
#' value names).
#' 
#' The initial number of individuals in each state and the 
#' number of cycle will be the same for all models.
#' 
#' Internally this function does 2 operations: first 
#' evaluating parameters, transition matrix, state values 
#' and computing individual counts through 
#' \code{\link{eval_model}}; and then using individual 
#' counts and evaluated state values to compute values at 
#' each cycle through \code{compute_values}.
#' 
#' @param ... One or more \code{uneval_model} object.
#' @param init numeric vector, same length as number of 
#'   model states. Number of individuals in each model state
#'   at the beginning.
#' @param cycles positive integer. Number of Markov Cycles 
#'   to compute.
#' @param cost Names or expression to compute cost on the
#'   cost-effectiveness plane.
#' @param effect Names or expression to compute effect on
#'   the cost-effectiveness plane.
#' @param base_model Name of base model used as reference.
#'   By default the model with the lowest effectiveness.
#' @param method Counting method.
#' @param list_models List of models, only used by 
#'   \code{run_models_} to avoid using \code{...}.
#'   
#' @return A list of evaluated models with computed values.
#' @export
#' 
#' @example inst/examples/example_run_models.R
#'   
run_models <- function(...,
                       init = c(1000L, rep(0L, get_state_number(get_states(list(...)[[1]])) - 1)),
                       cycles = 1,
                       method = c("beginning", "end", "cycle-tree",
                                  "half-cycle", "life-table", "spread-half-cycle"),
                       cost, effect, base_model = NULL) {
  list_models <- list(...)
  
  method <- match.arg(method)
  
  run_models_(
    list_models = list_models,
    init = init,
    cycles = cycles,
    method = method,
    cost = lazyeval::lazy(cost),
    effect = lazyeval::lazy(effect),
    base_model = base_model
  )
}

#' @export
#' @rdname run_models
run_models_ <- function(list_models,
                        init,
                        cycles,
                        method,
                        cost, effect, base_model) {
  
  stopifnot(
    all(unlist(lapply(list_models,
                      function(x) "uneval_model" %in% class(x)))),
    ! missing(cost),
    ! missing(effect)
  )
  
  list_ce <- list(
    cost,
    effect
  )
  names(list_ce) <- c(".cost", ".effect")
  ce <- c(
    lazyeval::lazy_dots(),
    list_ce
  )
  
  model_names <- names(list_models)
  
  if (is.null(model_names)) {
    message("No named model -> generating names.")
    model_names <- as.character(utils::as.roman(seq_along(list_models)))
    names(list_models) <- model_names
  }
  
  if (any(model_names == "")) {
    warning("Not all models are named -> generating names.")
    model_names <- as.character(utils::as.roman(seq_along(list_models)))
    names(list_models) <- model_names
  }
  
  stopifnot(
    all("uneval_model" %in% unlist(lapply(list_models, class))),
    list_all_same(lapply(list_models,
                         function(x) sort(get_state_names(x)))),
    list_all_same(lapply(list_models,
                         function(x) sort(get_state_value_names(x))))
  )
  
  stopifnot(
    length(init) == get_state_number(list_models[[1]]),
    all(init >= 0)
  )
  
  if (is.null(names(init)))
    names(init) <- get_state_names(list_models[[1]])
  
  stopifnot(
    all(sort(names(init)) == sort(get_state_names(list_models[[1]])))
  )
  
  eval_model_list <- lapply(list_models, eval_model, 
                            init = init, 
                            cycles = cycles,
                            method = method)
  
  list_res <- lapply(eval_model_list, get_total_state_values)
  
  for (n in model_names){
    list_res[[n]]$.model_names <- n
  }
  
  res <- Reduce(dplyr::bind_rows, list_res)
  
  res <- dplyr::mutate_(res, .dots = ce)
  
  if (is.null(base_model)) {
    base_model <- get_base_model(res)
  }
  
  structure(
    res,
    eval_model_list = eval_model_list,
    uneval_model_list = list_models,
    class = c("eval_model_list", class(res)),
    init = init,
    cycles = cycles,
    method = method,
    ce = ce,
    base_model = base_model
  )
}

#' @export
print.eval_model_list <- function(x, ...) {
  print(summary(x, ...))
}

get_total_state_values <- function(x) {
  # faster than as.data.frame or dplyr::as_data_frame
  res <- as.list(colSums((x$values)[- 1]))
  class(res) <- "data.frame"
  attr(res, "row.names") <- c(NA, -1)
  res
}

get_base_model <- function(x, ...) {
  UseMethod("get_base_model")
}

get_base_model.default <- function(x, ...) {
  x$.model_names[which(x$.effect == min(x$.effect))[1]]
}
get_base_model.eval_model_list <- function(x, ...) {
  attr(x, "base_model")
}
get_base_model.probabilistic <- function(x, ...) {
  get_base_model(attr(x, "model"))
}

#' Summarise Markov Model Results
#' 
#' @param object Output from \code{\link{run_models}}.
#' @param ... additional arguments affecting the summary
#'   produced.
#'   
#' @return A \code{summary_eval_model_list} object.
#' @export
#' 
summary.eval_model_list <- function(object, ...) {
  res <- as.data.frame(compute_icer(normalize_ce(object)))
  
  res <- dplyr::select(res, - .model_names)

  rownames(res) <- object$.model_names
  
  res_comp <- res[c(".cost", ".effect", ".icer")]
  is.na(res_comp$.icer) <- ! is.finite(res_comp$.icer)
  res_comp$.icer <- format(res_comp$.icer)
  res_comp$.icer[res_comp$.icer == "NA"] <- "-"
  res_comp$.cost <- res_comp$.cost / sum(attr(object, "init"))
  res_comp$.effect <- res_comp$.effect / sum(attr(object, "init"))
  names(res_comp) <- c("Cost", "Effect", "ICER")

  structure(
    list(
      res = dplyr::select(res, - .cost, - .effect, - .icer),
      res_comp = res_comp[-1, ],
      cycles = attr(object, "cycles"),
      init = attr(object, "init"),
      count_args = attr(object, "count_args"),
      frontier = get_frontier(object)
    ),
    class = "summary_eval_model_list"
  )
}
if(getRversion() >= "2.15.1")
  utils::globalVariables(c(".model_names", ".cost", ".effect", ".icer"))

#' Normalize Cost and Effect
#' 
#' Normalize cost and effect values taking base model as a 
#' reference.
#' 
#' @param x Result of \code{\link{run_models}} or
#'   \code{\link{run_probabilistic}}.
#'   
#' @return Input with normalized \code{.cost} and 
#'   \code{.effect}, ordered by \code{.effect}.
normalize_ce <- function(x) {
  UseMethod("normalize_ce")
}
normalize_ce.eval_model_list <- function(x) {
  bm <- get_base_model(x)
  x$.cost <- x$.cost - x$.cost[x$.model_names == bm]
  x$.effect <- x$.effect - x$.effect[x$.model_names == bm]
  x[order(x$.effect), ]
}

#' Compute ICER
#' 
#' Compute ICER for Markov models.
#' 
#' Models are ordered by effectiveness and ICER are computed sequencially.
#' 
#' @param x Result of \code{\link{run_models}}.
#'   
#' @return A \code{data.frame} with computed ICER.
#' @export
#' 
compute_icer <- function(x) {
  tab <- x[order(x$.effect), ]
  
  for (i in seq_len(nrow(tab))) {
    if ( i == 1) {
      tab$.icer[i] <- -Inf
    } else {
      tab$.icer[i] <- (tab$.cost[i] - tab$.cost[i-1]) /
        (tab$.effect[i] - tab$.effect[i-1])
    }
  }
  tab
}

#' @export
print.summary_eval_model_list <- function(x, ...) {
  cat(sprintf(
    "%i Markov model%s run for %i cycle%s.\n\n",
    nrow(x$res),
    plur(nrow(x$res)),
    x$cycles,
    plur(x$cycles)
  ))
  cat("Initial states:\n\n")
  print(matrix(
    x$init,
    dimnames = list(
      names(x$init),
      "N"
    )
  ))
  print(x$res)
  
  if (nrow(x$res) > 1) {
    cat("\nEfficiency frontier:\n\n")
    cat(x$frontier)
    cat("\n\nModel difference:\n\n")
    print(x$res_comp)
  }
}

#' @export
print.eval_model <- function(x, width = Inf, ...) {
  cat(sprintf("A Markov model, run for %i cycle%s.\n\n",
              attr(x, "cycles"),
              plur(attr(x, "cycles"))
  ))
  
  cat("Initial individual counts:\n\n")
  print(data.frame(
    "State names" = names(get_counts(x)),
    "Initial counts" = attr(x, "init"),
    check.names = FALSE
  ))
  
  cat("Individual counts per cycle:\n\n")
  print(get_counts(x), width = width, ...)
  
  cat("State values per cycle:\n\n")
  print(get_state_values(x), width = width, ...)
}


