% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/corrected_lasso.R
\name{corrected_lasso}
\alias{corrected_lasso}
\title{Corrected Lasso}
\usage{
corrected_lasso(
  W,
  y,
  sigmaUU,
  family = c("gaussian", "binomial", "poisson"),
  radii = NULL,
  no_radii = NULL,
  alpha = 0.1,
  maxits = 5000
)
}
\arguments{
\item{W}{Design matrix, measured with error. Must be a numeric matrix.}

\item{y}{Vector of responses.}

\item{sigmaUU}{Covariance matrix of the measurement error.}

\item{family}{Response type. Character string of length 1. Possible values are
"gaussian", "binomial" and "poisson".}

\item{radii}{Vector containing the set of radii of the l1-ball onto which the
solution is projected. If not provided, the algorithm will select an evenly
spaced vector of 20 radii.}

\item{no_radii}{Length of vector radii, i.e., the number of regularization
parameters to fit the corrected lasso for.}

\item{alpha}{Step size of the projected gradient descent algorithm. Default is
0.1.}

\item{maxits}{Maximum number of iterations of the project gradient descent
algorithm for each radius. Default is 5000.}
}
\value{
An object of class "corrected_lasso".
}
\description{
Lasso (L1-regularization) for generalized linear models with
 measurement error.
}
\details{
Corrected version of the lasso for generalized linear models. The
 method does require an estimate of the measurement error covariance matrix.
}
\examples{
# Example with linear regression
# Number of samples
n <- 100
# Number of covariates
p <- 50
# True (latent) variables
X <- matrix(rnorm(n * p), nrow = n)
# Measurement error covariance matrix
# (typically estimated by replicate measurements)
sigmaUU <- diag(x = 0.2, nrow = p, ncol = p)
# Measurement matrix (this is the one we observe)
W <- X + rnorm(n, sd = diag(sigmaUU))
# Coefficient
beta <- c(seq(from = 0.1, to = 1, length.out = 5), rep(0, p-5))
# Response
y <- X \%*\% beta + rnorm(n, sd = 1)
# Run the corrected lasso
fit <- corrected_lasso(W, y, sigmaUU, family = "gaussian")
coef(fit)
plot(fit)
plot(fit, type = "path")

# Binomial, logistic regression
# Number of samples
n <- 1000
# Number of covariates
p <- 50
# True (latent) variables
X <- matrix(rnorm(n * p), nrow = n)
# Measurement error covariance matrix
sigmaUU <- diag(x = 0.2, nrow = p, ncol = p)
# Measurement matrix (this is the one we observe)
W <- X + rnorm(n, sd = diag(sigmaUU))
logit <- function(x) (1+exp(-x))^(-1)
# Response
y <- rbinom(n, size = 1, prob = logit(X \%*\% c(rep(5, 5), rep(0, p-5))))
fit <- corrected_lasso(W, y, sigmaUU, family = "binomial")
plot(fit)
coef(fit)

}
\references{
\insertRef{loh2012}{hdme}

\insertRef{sorensen2015}{hdme}
}
