makeFitFunction <- function(fitfun)
{
	# prefer marginal TODO
  if(fitfun == "WLS")        mxFitFunctionWLS(allContinuousMethod= "marginals")
  else if(fitfun == "ML")  mxFitFunctionML()
  else stop(paste("Unknown fitfun", omxQuotes(fitfun)))
}

calcMinVar <- function(minMAF) 2*minMAF*(1-minMAF)

#' Return a suitable compute plan for a genome-wide association study
#'
#' Instead of using OpenMx's default model processing sequence (i.e.,
#' \link[OpenMx]{omxDefaultComputePlan}), it is more efficient and
#' convienient to assemble a compute plan tailored for a genome-wide
#' association study.  This function returns a compute plan that loads
#' SNP data into model \code{modelName}, fits the model, outputs the
#' results to \code{out}, and repeats this procedure for all SNPs.
#'
#' @details
#'
#' You can request a specific list of SNPs using the \code{SNP}
#' argument. The numbers provided in \code{SNP} refer to offsets in
#' the \code{snpData} file. For example, \code{SNP=c(100,200)} will
#' process the 100th and 200th SNP. The first SNP in the
#' \code{snpData} file is at offset 1. When \code{SNP} is omitted then
#' all available SNPs are processed.
#' 
#' The suffix of \code{snpData} filename is interpreted to signal the
#' format of how the SNP data is stored on disk. Suffixes
#' \sQuote{pgen}, \sQuote{bed}, and \sQuote{bgen} are supported.
#' Per-SNP descriptions are found in different places depending on the
#' suffix. For \sQuote{bgen}, both the SNP data and description are
#' built into the same file. In the case of \sQuote{pgen}, an
#' associated file with suffix \sQuote{pvar} is expected to exist (see
#' the
#' \href{https://www.cog-genomics.org/plink/2.0/formats#pvar}{spec}
#' for details). In the case of \sQuote{bed}, an associated
#' \sQuote{bim} file is expected to exist (see the
#' \href{https://www.cog-genomics.org/plink2/formats#bim}{spec} for
#' details). The chromosome, base-pair coordinate, and variant ID are
#' added to each line of \code{out}.
#'
#' The code to implement method='pgen' is based on plink 2.0
#' alpha. plink's \sQuote{bed} file format is supported in addition
#' to \sQuote{pgen}. Data are coerced appropriately depending on the
#' type of the destination column. For a numeric column, data are
#' recorded as the values NA, 0, 1, or 2. An ordinal column must have
#' exactly 3 levels.
#'
#' For \code{method='bgen'}, the file \code{path+".bgi"} must also
#' exist. If not available, generate this index file with the
#' \href{https://bitbucket.org/gavinband/bgen/wiki/bgenix}{bgenix}
#' tool.
#'
#' For \sQuote{bgen} and \sQuote{pgen} formats, the numeric column can be
#' populated with a dosage (sum of probabilities multiplied by genotypes)
#' if these data are available.
#'
#' A compute plan does not do anything by itself. You'll need to combine
#' the compute plan with a model (such as returned by \link{buildOneFac})
#' to perform a GWAS.
#'
#' @template args-model
#' @template args-snpData
#' @template args-snp
#' @template args-out
#' @template args-dots-barrier
#' @template args-startfrom
#' @return
#' The given model with an appropriate compute plan.
#'
#' @export
#' @importFrom methods is
#' @seealso \link{GWAS}
#' @examples
#' m1 <- mxModel("test", mxFitFunctionWLS())
#' dir <- system.file("extdata", package = "gwsem")
#' m1 <- prepareComputePlan(m1, file.path(dir,"example.pgen"))
#' m1$compute
prepareComputePlan <- function(model, snpData, out="out.log", ...,
			       SNP=NULL, startFrom=1L)
{
  if (length(list(...)) > 0) stop("Rejected are any values passed in the '...' argument")
  modelName <- model$name
  pieces <- strsplit(snpData, ".", fixed=TRUE)[[1]]
  if (length(pieces) < 2) {
    stop(paste("Please rename snpData",omxQuotes(snpData),
               "to the form file.ext where ext reflects the format of the data"))
  }
  snpFileExt <- pieces[length(pieces)]
  stem <- paste(pieces[-length(pieces)], collapse=".")

  if (snpFileExt == 'pgen' || snpFileExt == 'bed') method <- 'pgen'
  else if (snpFileExt == 'bgen') method <- 'bgen'
  else stop(paste("Unrecognized file extension", omxQuotes(snpFileExt),
                  "inferred from snpData", omxQuotes(snpData)))

  onesnp <- list(
    ST=mxComputeSetOriginalStarts(),
    LD=mxComputeLoadData(modelName, column='snp',
                         path=snpData, method=method))

  if (snpFileExt == "pgen") {
    # TODO doc column=1:3, sep='\t'
    onesnp <- c(
      onesnp,
      LC=mxComputeLoadContext(path=paste(stem, "pvar", sep = "."), column=1:3, sep='\t',
			      col.names=c("CHR", "BP", "SNP")))
  } else if (snpFileExt == "bed") {
    onesnp <- c(
      onesnp,
      LC=mxComputeLoadContext(path=paste(stem, "bim", sep = "."),
                              column=c(1,2,4), sep='\t', header=FALSE,
                              col.names=c("CHR", "SNP", "BP")))
  }

  opt <- list(GD=mxComputeGradientDescent())
  if (is(model$fitfunction, "MxFitFunctionWLS")) {
	  opt <- c(opt, SE=mxComputeStandardError())
  } else {
	  opt <- c(opt,
		   ND=mxComputeNumericDeriv(),
		   SE=mxComputeStandardError(),
		   HQ=mxComputeHessianQuality())
  }

  onesnp <- c(
    onesnp,
    TC=mxComputeTryCatch(mxComputeSequence(opt)),
    CK=mxComputeCheckpoint(path=out, standardErrors = TRUE))

  mxModel(model, mxComputeLoop(onesnp, i=SNP, startFrom=startFrom))
}

#' Run a genome-wide association study (GWAS) using the provided model
#'
#' The GWAS function is used to run a genome-wide association study based on the specified model. This function is design to take the output from \link{buildOneFac}, \link{buildOneFacRes}, and \link{buildTwoFac} as input, but can also take a similar user specified model. Users should be confident that the models they are running are statistically identified. It is advisable that the users empirically gauge time requirements by running a limited number of SNPs (e.g. 10) to ensure that all SNPs can be fit in a reasonable amount of time.
#' 
#' Adds a compute plan returned by \link{prepareComputePlan} to the
#' provided \code{model} and runs it. Once analyses are complete,
#' load your aggregated results with \link{loadResults}. 
#'
#' @template args-model
#' @template args-snpData
#' @template args-snp
#' @template args-out
#' @template args-dots-barrier
#' @template args-startfrom
#' @export
#' @return
#' The results for each SNP are recorded in the specified log file (\code{out}).
#' In addition, data and estimates for the last SNP run are returned
#' as an \link[OpenMx:MxModel-class]{MxModel} object
#' (similar to the return value of \link[OpenMx]{mxRun}).
#' In this way, the last SNP processed is available for close inspection.
#' @examples
#' dir <- system.file("extdata", package = "gwsem")
#' pheno <- data.frame(anxiety=rnorm(500))
#' m1 <- buildOneItem(pheno, 'anxiety')
#' GWAS(m1, file.path(dir,"example.pgen"),
#'      file.path(tempdir(),"out.log"))
GWAS <- function(model, snpData, out="out.log", ..., SNP=NULL, startFrom=1L)
{
	# verify model has a continuous 'snp' data column TODO
  if (length(list(...)) > 0) stop("Rejected are any values passed in the '...' argument")
  model <- prepareComputePlan(model, snpData, out=out,
			      SNP=SNP, startFrom=startFrom)
  model <- mxRun(model)
  message(paste("Done. See", omxQuotes(out), "for results"))
  invisible(model)
}

#' Set up thresholds for ordinal indicators
#'
#' Ordinal indicator thresholds are freely estimated with fixed means
#' and variance. This function adds thresholds to the given
#' \code{model}.  If no indicators are ordinal, the given \code{model}
#' is returned without changes.
#'
#' @details
#' 
#' Thresholds are added using \link[OpenMx]{mxThreshold}. Starting
#' values for thresholds use the defaults provided by this function
#' which assumes a mean of zero and variance of the square root of
#' two.  This variance is appropriate for \link{buildOneFac} where the
#' implied model variance of ordinal indicators is one plus the square
#' of the factor loading, and the loading's starting value is 1.0.
#'
#' @template args-model
#' @template detail-adv
#'
#' @return
#' The given \link[OpenMx:MxModel-class]{MxModel} with appropriate thresholds added.
#' @export
#' @examples
#' pheno <- data.frame(anxiety=cut(rnorm(500), c(-Inf, -.5, .5, Inf),
#'                     ordered_result = TRUE))
#' m1 <- buildOneItem(pheno, 'anxiety')
#' m1 <- setupThresholds(m1)
#' m1$Thresholds
setupThresholds <- function(model)
{
  phenoData <- model$data$observed
  manifestNames <- setdiff(model$manifestVars, 'snp')

  thr <- sapply(phenoData[,manifestNames,drop=FALSE], nlevels)-1
  fac <- thr >= 1
  thr[thr< 0] <- 0

  if (max(thr) == 0) return(model)

  thresh <- mxThreshold(manifestNames[fac], nThresh=thr[fac], free = T ,
                        labels = paste0(rep(manifestNames[fac], each = max(thr)), "_Thr_", 1:max(thr)),
			values=mxNormalQuantiles(thr[fac], sd=sqrt(2.0)))
  mxModel(model, thresh)
}

#' Set up exogenous model covariates
#'
#' In GWAS, including a number of the first principle components as
#' covariates helps reduce false positives caused by population
#' stratification. This function adds paths from covariates to
#' manifest indicators (\code{itemNames}). Covariates are always treated as continuous
#' variables (not ordinal).
#'
#' @details
#' This is not the only way to adjust a model for
#' covariates. For example, in a single factor model (e.g., \link{buildOneFac}),
#' it would be more
#' appropriate to adjust the latent factor instead of the manifest
#' indicators.
#' This is how endogenous covariates work.
#' However, exogenous covariate adjustments to latent variables are only
#' possible with a maximum likelihood fit function
#' (\link[OpenMx]{mxFitFunctionML}).  For
#' \link[OpenMx]{mxFitFunctionWLS}, only manifest indicators can be
#' adjusted for exogenous covariates.
#' This function always adjusts manifest indicators regardless of the fit function.
#' 
#' @template args-model
#' @param covariates a character vector naming covariates available in the model data
#' @param itemNames a character vector of item names
#' @template detail-adv
#' @return
#' The given \link[OpenMx:MxModel-class]{MxModel} with paths
#' added from covariates to manifest indicators.
#' @export
#' @examples
#' m1 <- mxModel("test", type="RAM",
#'              latentVars = "sex", manifestVars = "anxiety")
#' m1 <- setupExogenousCovariates(m1, 'sex', 'anxiety')
setupExogenousCovariates <- function(model, covariates, itemNames)
{
  if (length(covariates)==0) return(model)

  covMean   <- mxPath(from = "one", to = covariates, free=FALSE, labels = paste0('data.',covariates)) 
  cov2item  <- mxPath(from = covariates, to = itemNames, connect = "all.pairs",
                      labels = paste(rep(covariates, each = length(itemNames)), itemNames, sep = "_"))
  mxModel(model, covMean, cov2item)
}

# export? TODO
# just pass in model as argument? TODO
setupData <- function(phenoData, observed, covariates, exogenousCovariates, gxe, customMinMAF, minMAF, fitfun)
{
  if (customMinMAF && fitfun != "WLS") warning("minMAF is ignored when fitfun != 'WLS'")
  minVar <- calcMinVar(minMAF)
  result <- list()
  aname <- c()
  if (length(gxe)) for (v1 in gxe) {
	  alg <- mxAlgebraFromString(paste0("data.snp * data.",v1),
				     name=paste0('snp_',v1,"Alg"),
				     dimnames=list(NULL,paste0('snp_',v1)))
	  result <- c(result, alg)
	  phenoData[[ paste0('snp_',v1) ]] <- 0.0  # placeholder
	  aname <- c(aname, paste0('snp_',v1))
  }
  exoFree <- matrix(TRUE, length(observed), length(exogenousCovariates),
		    dimnames=list(observed, exogenousCovariates))
  exoFree[c('snp',covariates),] <- FALSE
  c(mxData(observed=phenoData, type="raw", minVariance=minVar, warnNPDacov=FALSE,
	   exoFree=exoFree, algebra=aname), result)
}

#' @importFrom stats rbinom
addPlaceholderSNP <- function(phenoData) {
	# We use as.numeric because we currently only support dosages.
	phenoData$snp <- as.numeric(rbinom(dim(phenoData)[1], 2, .5))
	phenoData
}

setupPaths <- function(covariates, depVar)
{
	paths <- list(mxPath(from = "one", to = "snp" , labels = "snpMean"),
		      mxPath(from = "snp", to = depVar, values = 0,
			     labels=paste("snp", depVar, sep = "2")),
		      mxPath(from = "snp", arrows=2, values=1, labels = paste("snp", "res", sep = "_")))

	if (length(covariates)) {
		paths <- c(paths, list(mxPath(from='one',covariates),
				       mxPath(from=covariates, arrows=2, values=1),
				       mxPath(from=covariates, to=depVar)))
	}
	paths
}

#' Build a model suitable for a single item genome-wide association study
#'
#' @template detail-build
#'
#' @template args-phenoData
#' @param depVar the name of the single item to predict
#' @template args-covariates
#' @template args-exogenouscovariates
#' @template args-dots-barrier
#' @template args-fitfun
#' @template args-minmaf
#' @template args-modeltype
#' @template args-gxe
#' @family model builder
#' @export
#' @return
#' A \link[OpenMx:MxModel-class]{MxModel}
#' @examples
#' pheno <- data.frame(anxiety=cut(rnorm(500), c(-Inf, -.5, .5, Inf),
#'                     ordered_result = TRUE))
#' m1 <- buildOneItem(pheno, 'anxiety')
buildOneItem <- function(phenoData, depVar, covariates=NULL, ..., fitfun = c("WLS","ML"), minMAF=0.01,
			 modelType=c('RAM','LISREL'), gxe=NULL, exogenousCovariates=NULL)
{
  if (length(list(...)) > 0) stop("Rejected are any values passed in the '...' argument")
  fitfun <- match.arg(fitfun)
  modelType <- match.arg(modelType)

  if (length(depVar) != 1) {
    stop(paste("buildOneItem provided with", length(depVar), "dependent variables",
	       "instead of 1. Did you intend to use buildOneFac instead?"))
  }
  fac <- is.factor(phenoData[[depVar]])

  phenoData <- addPlaceholderSNP(phenoData)
  paths <- setupPaths(covariates, depVar)
  paths <- c(paths,
	     mxPath(from = c(depVar), arrows=2, values=1, free = !fac, labels = paste(c(depVar), "res", sep = "_")),
	     mxPath(from = 'one', to = depVar, free= !fac, values = 0, labels = paste0(depVar, "Mean")))

  manifest <- c("snp", depVar, covariates)
  dat       <- setupData(phenoData, manifest, covariates, exogenousCovariates, gxe, force(!missing(minMAF)), minMAF, fitfun)

  modelName <- "OneItem"
  oneFacPre <- mxModel(model=modelName, type=modelType,
                       manifestVars = manifest,
                       latentVars = c(exogenousCovariates),
                       paths, dat, makeFitFunction(fitfun))

  oneFacPre <- setupThresholds(oneFacPre)
  setupExogenousCovariates(oneFacPre, exogenousCovariates, depVar)
}

#' Build a model suitable for a single factor genome-wide association study
#'
#' The \code{buildOneFac} function is used to specify a single factor latent variable model where the latent variable is predicted by a genomic variant such as a single nucleotide polymorphism, as well as range of covariates. \figure{singleFactor.jpg}{Single Factor Model}
#'
#' @template detail-build
#'
#' @template args-phenoData
#' @param itemNames A vector of phenotypic item names (from \code{phenoData}) that load on the latent factor.
#' @template args-covariates
#' @template args-exogenouscovariates
#' @template args-dots-barrier
#' @template args-fitfun
#' @template args-minmaf
#' @template args-modeltype
#' @template args-gxe
#' @family model builder
#' @export
#' @return
#' \code{buildOneFac} returns an \link[OpenMx:MxModel-class]{MxModel} object that can serve as input for the \link{GWAS} function.
#' @examples
#' pheno <- list()
#' for (i in 1:5) pheno[[paste0('i',i)]] <- rnorm(500)
#' pheno <- as.data.frame(pheno)
#' buildOneFac(pheno, colnames(pheno))
buildOneFac <- function(phenoData, itemNames, covariates=NULL, ..., fitfun = c("WLS","ML"), minMAF=0.01,
			modelType=c('RAM','LISREL'), gxe=NULL, exogenousCovariates=NULL)
{
  if (length(list(...)) > 0) stop("Rejected are any values passed in the '...' argument")
  fitfun <- match.arg(fitfun)
  modelType <- match.arg(modelType)

  fac <- sapply(phenoData[,itemNames,drop=FALSE], is.factor)

  phenoData <- addPlaceholderSNP(phenoData)
  latents   <- c("F")
  paths <- setupPaths(covariates, "F")
  paths <- c(paths,
	     mxPath(from=latents, to=itemNames,values=1, free = T,
		    labels = paste("lambda", itemNames, sep = "_")  ),
	     mxPath(from = c(itemNames), arrows=2, values=1, free = !fac,
		    labels = paste(c(itemNames), "res", sep = "_")),
	     mxPath(from=latents, arrows=2,free=F, values=1.0, labels = "facRes"),
	     mxPath(from = 'one', to = itemNames, free= !fac, values = 0,
		    labels = paste0(itemNames, "Mean")))

  manifest <- c("snp", itemNames, covariates)
  dat       <- setupData(phenoData, manifest, covariates, exogenousCovariates, gxe, force(!missing(minMAF)), minMAF, fitfun)

  modelName <- "OneFac"
  oneFacPre <- mxModel(model=modelName, type=modelType,
                       manifestVars = manifest,
                       latentVars = c(latents, exogenousCovariates),
                       paths, dat, makeFitFunction(fitfun))

  oneFacPre <- setupThresholds(oneFacPre)
  setupExogenousCovariates(oneFacPre, exogenousCovariates, itemNames)
}

#' Build a model suitable for a single factor residual genome-wide association study
#'
#' The \code{buildOneFacRes} function is used to specify a single factor latent variable model where a combination of items as well as the latent variable may be predicted by a genomic variant such as a single nucleotide polymorphism, as well as range of covariates. \figure{resid.jpg}{Single Factor Model with a Focus on Residuals}
#'
#' Be aware that a latent variable model is not identified if all of the residuals as well as the latent variable are simultaneously predicted by the SNP.  Specifically, if users wish to use the SNP to predict the latent variable, they much choose at least one (and preferably more that one) item to not be predicted by the SNP.
#' 
#' @template detail-build
#' 
#' @param itemNames A vector of phenotypic item names (from phenoData) that load on the latent factor.
#' @param factor A logical expression (\code{FALSE} or \code{TRUE}) indicating whether to estimate a regression pathway from the SNP to the latent factor (default FALSE).
#' @param res A character vector of phenotypic item names that indicate which specific items the user wishes to regress on the SNP. The default is to regress all of the items on the SNP.
#' @template args-phenoData
#' @template args-covariates
#' @template args-exogenouscovariates
#' @template args-fitfun
#' @template args-minmaf
#' @template args-dots-barrier
#' @template args-modeltype
#' @template args-gxe
#' 
#' @family model builder
#' @export
#' @return
#' \code{buildOneFacRes} returns an \link[OpenMx:MxModel-class]{MxModel} object that can serve as input for the \link{GWAS} function.
#' @examples
#' pheno <- list()
#' for (i in 1:5) pheno[[paste0('i',i)]] <- rnorm(500)
#' pheno <- as.data.frame(pheno)
#' buildOneFacRes(pheno, colnames(pheno))
buildOneFacRes <- function(phenoData, itemNames, factor = F, res = itemNames, covariates = NULL,
			   ..., fitfun = c("WLS","ML"), minMAF = .01, modelType=c('RAM','LISREL'), gxe=NULL,
			 exogenousCovariates=NULL)
{
  if (length(list(...)) > 0) stop("Rejected are any values passed in the '...' argument")
  fitfun <- match.arg(fitfun)
  modelType <- match.arg(modelType)
  
  fac <- sapply(phenoData[,itemNames,drop=FALSE], is.factor)

  phenoData <- addPlaceholderSNP(phenoData)
  latents   <- c("F")

  paths <- setupPaths(covariates, c("F", res))
  paths <- c(paths,
	     mxPath(from=latents, to=itemNames,values=1, free = T,
		    labels = paste("lambda", itemNames, sep = "_")  ),
	     mxPath(from = c(itemNames), arrows=2, values=1, free = c(fac==0),
		    labels = paste(c(itemNames), "res", sep = "_")),
	     mxPath(from=latents, arrows=2,free=F, values=1.0, labels = "facRes"),
	     mxPath(from = 'one', to = itemNames, free= c(fac==0), values = 0,
		    labels = paste0(itemNames, "Mean")))

  manifest <- c("snp", itemNames, covariates)
  dat       <- setupData(phenoData, manifest, covariates, exogenousCovariates, gxe, force(!missing(minMAF)), minMAF, fitfun)

  modelName <- "OneFacRes"
  oneFacPre <- mxModel(model=modelName, type=modelType,
                       manifestVars = manifest,
                       latentVars = c(latents, exogenousCovariates),
                       paths, dat, makeFitFunction(fitfun))

  oneFacPre <- setupThresholds(oneFacPre)
  setupExogenousCovariates(oneFacPre, exogenousCovariates, itemNames)
}

#' Build a model suitable for a two factor genome-wide association study
#'
#' The buildTwoFac function is used to specify a model with two latent variables where each latent variable is simultaneously predicted by a genomic variant such as a single nucleotide polymorphism, as well as range of covariates. The model allows the latent variables to correlate to accomodate comorbidity between latent traits. \figure{twoFactor.jpg}{Two Factor Model}
#'
#' @template detail-build
#' 
#' @param F1itemNames A vector of phenotypic item names (from phenoData) that load on the first latent factor.
#' @param F2itemNames a vector of phenotypic item names (from phenoData) that load on the second latent factor.
#' 
#' @template args-phenoData
#' @template args-covariates
#' @template args-exogenouscovariates
#' @template args-fitfun
#' @template args-minmaf
#' @template args-dots-barrier
#' @template args-modeltype
#' @template args-gxe
#' @export
#' @family model builder
#' @return
#' \code{buildTwoFac} returns an \link[OpenMx:MxModel-class]{MxModel} object that can serve as input for the \link{GWAS} function.
#' @examples
#' pheno <- list()
#' for (i in 1:10) pheno[[paste0('i',i)]] <- rnorm(500)
#' pheno <- as.data.frame(pheno)
#' buildTwoFac(pheno, paste0('i',1:6), paste0('i',5:10))
buildTwoFac <- function(phenoData, F1itemNames, F2itemNames, covariates = NULL, ...,
			fitfun = c("WLS","ML"), minMAF = .01, modelType=c('RAM','LISREL'), gxe=NULL,
			exogenousCovariates=NULL)
{
  if (length(list(...)) > 0) stop("Rejected are any values passed in the '...' argument")
  fitfun <- match.arg(fitfun)
  modelType <- match.arg(modelType)

  itemNames <- union(F1itemNames, F2itemNames)

  fac <- sapply(phenoData[,itemNames,drop=FALSE], is.factor)

  phenoData <- addPlaceholderSNP(phenoData)
  latents   <- c("F1", "F2")
  paths <- setupPaths(covariates, latents)
  paths <- c(paths,
	     mxPath(from="F1", to=F1itemNames,values=1, labels = paste("lambda", F1itemNames, sep = "_")  ),
	     mxPath(from="F2", to=F2itemNames,values=1, labels = paste("lambda", F2itemNames, sep = "_")  ),
	     mxPath(from="F1", to= "F2", arrows=2,free=T, values=.3),
	     mxPath(from = itemNames, arrows=2, values=1, free = c(fac==0),
		    labels = paste(c(itemNames), "res", sep = "_")),
	     mxPath(from=latents, arrows=2,free=F, values=1.0, labels = "facRes"),
	     mxPath(from = 'one', to = itemNames, free= c(fac==0), values = 0, labels = paste0(itemNames, "Mean")))

  manifest <- c("snp", itemNames, covariates)
  dat       <- setupData(phenoData, manifest, covariates, exogenousCovariates, gxe, force(!missing(minMAF)), minMAF, fitfun)

  modelName <- "TwoFac"
  twoFacPre <- mxModel(model=modelName, type=modelType,
                       manifestVars = manifest,
                       latentVars = c(latents, exogenousCovariates),
                       paths, dat, makeFitFunction(fitfun))

  twoFacPre <- setupThresholds(twoFacPre)
  setupExogenousCovariates(twoFacPre, exogenousCovariates, itemNames)
}
