% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mcmc.r
\name{MCMC}
\alias{MCMC}
\alias{Metropolis-Hastings}
\alias{MCMC.mcmc}
\alias{MCMC.mcmc.list}
\alias{MCMC.default}
\title{Markov Chain Monte Carlo}
\usage{
MCMC(initial, fun, nsteps, ..., nchains = 1L, burnin = 0L, thin = 1L,
  kernel = kernel_normal(), multicore = FALSE, conv_checker = NULL,
  cl = NULL, progress = interactive())

\method{MCMC}{mcmc}(initial, fun, nsteps, ..., nchains = 1L,
  burnin = 0L, thin = 1L, kernel = kernel_normal(),
  multicore = FALSE, conv_checker = NULL, cl = NULL,
  progress = interactive() && !multicore)

\method{MCMC}{mcmc.list}(initial, fun, nsteps, ..., nchains = 1L,
  burnin = 0L, thin = 1L, kernel = kernel_normal(),
  multicore = FALSE, conv_checker = NULL, cl = NULL,
  progress = interactive() && !multicore)

\method{MCMC}{default}(initial, fun, nsteps, ..., nchains = 1L,
  burnin = 0L, thin = 1L, kernel = kernel_normal(),
  multicore = FALSE, conv_checker = NULL, cl = NULL,
  progress = interactive() && !multicore)
}
\arguments{
\item{initial}{Either a numeric matrix or vector, or an object of class \link[coda:mcmc]{coda::mcmc}
or \link[coda:mcmc.list]{coda::mcmc.list} (see details).
initial values of the parameters for each chain (See details).}

\item{fun}{A function. Returns the log-likelihood.}

\item{nsteps}{Integer scalar. Length of each chain.}

\item{...}{Further arguments passed to \code{fun}.}

\item{nchains}{Integer scalar. Number of chains to run (in parallel).}

\item{burnin}{Integer scalar. Length of burn-in. Passed to
\link[coda:mcmc]{coda::mcmc} as \code{start}.}

\item{thin}{Integer scalar. Passed to \link[coda:mcmc]{coda::mcmc}.}

\item{kernel}{An object of class \link{fmcmc_kernel}.}

\item{multicore}{Logical. If \code{FALSE} then chains will be executed in serial.}

\item{conv_checker}{A function that receives an object of class \link[coda:mcmc.list]{coda::mcmc.list},
and returns a logical value with \code{TRUE} indicating convergence. See the
"Automatic stop" section and the \link{convergence-checker} manual.}

\item{cl}{A \code{cluster} object passed to \link[parallel:clusterApply]{parallel::clusterApply}.}

\item{progress}{Logical scalar (currently ignored).}
}
\value{
An object of class \link[coda:mcmc]{coda::mcmc} from the \CRANpkg{coda}
package. The \code{mcmc} object is a matrix with one column per parameter,
and \code{nsteps} rows. If \code{nchains > 1}, then it returns a \link[coda:mcmc.list]{coda::mcmc.list}.
}
\description{
A flexible implementation of the Metropolis-Hastings MCMC algorithm.
}
\details{
This function implements MCMC using the Metropolis-Hastings ratio with
flexible transition kernels. Users can specify either one of the available
transition kernels or define one of their own (see \link{kernels}). Furthermore,
it allows easy parallel implementation running multiple chains in parallel. In
addition, we incorporate a variety of convergence diagnostics, alternatively
the user can specify their own (see \link{convergence-checker}).

We now give details of the various options included in the function.
}
\section{Starting point}{


By default, if \code{initial} is of class \code{mcmc}, \code{MCMC} will take the last \code{nchains}
points from the chain as starting point for the new sequence. If \code{initial} is
of class \code{mcmc.list}, the number of chains in \code{initial} must match the \code{nchains}
parameter.

If \code{initial} is a vector, then it must be of length equal to the number of
parameters used in the model. When using multiple chains, if \code{initial} is not
an object of class \code{mcmc} or \code{mcmc.list}, then it must be a numeric matrix
with as many rows as chains, and as many columns as parameters in the model.
}

\section{Multiple chains}{


When \code{nchains > 1}, the function will run multiple chains. Furthermore,
if \code{cl} is not passed, \code{MCMC} will create a \code{PSOCK} cluster
using \link[parallel:makePSOCKcluster]{parallel::makePSOCKcluster} with
\link[parallel:detectCores]{parallel::detectCores}
clusters and attempt to execute using multiple cores. Internally, the function does
the following:

\preformatted{
  # Creating the cluster
  ncores <- parallel::detectCores()
  ncores <- ifelse(nchains < ncores, nchains, ncores)
  cl     <- parallel::makePSOCKcluster(ncores)
  
  # Loading the package and setting the seed using clusterRNGStream
  invisible(parallel::clusterEvalQ(cl, library(fmcmc)))
  parallel::clusterSetRNGStream(cl, .Random.seed)
}

When running in parallel, objects that are
used within \code{fun} must be passed through \code{...}, otherwise the cluster
will return with an error.

The user controls the initial value of the parameters of the MCMC algorithm
using the argument \code{initial}. When using multiple chains, i.e., \code{nchains > 1},
the user can specify multiple starting points, which is recommended. In such a
case, each row of \code{initial} is use as a starting point for each of the
chains. If \code{initial} is a vector and \code{nchains > 1}, the value is recycled, so
all chains start from the same point (not recommended, the function throws a
warning message).
}

\section{Automatic stop}{


By default, no automatic stop is implemented. If one of the functions in
\link{convergence-checker} is used, then the MCMC is done by bulks as specified
by the convergence checker function, and thus the algorithm will stop if,
the \code{conv_checker} returns \code{TRUE}. For more information see \link{convergence-checker}.
}

\examples{
# Univariate distributed data with multiple parameters ----------------------
# Parameters
set.seed(1231)
n <- 1e3
pars <- c(mean = 2.6, sd = 3)

# Generating data and writing the log likelihood function
D <- rnorm(n, pars[1], pars[2])
fun <- function(x) {
  x <- log(dnorm(D, x[1], x[2]))
  sum(x)
}

# Calling MCMC, but first, loading the coda R package for
# diagnostics
library(coda)
ans <- MCMC(
  fun, initial = c(mu=1, sigma=1), nsteps = 2e3,
  kernel = kernel_normal_reflective(scale = .1, ub = 10, lb = 0)
  )

# Ploting the output
oldpar <- par(no.readonly = TRUE)
par(mfrow = c(1,2))
boxplot(as.matrix(ans), 
        main = expression("Posterior distribution of"~mu~and~sigma),
        names =  expression(mu, sigma), horizontal = TRUE,
        col  = blues9[c(4,9)],
        sub = bquote(mu == .(pars[1])~", and"~sigma == .(pars[2]))
)
abline(v = pars, col  = blues9[c(4,9)], lwd = 2, lty = 2)

plot(apply(as.matrix(ans), 1, fun), type = "l",
     main = "LogLikelihood",
     ylab = expression(L("{"~mu,sigma~"}"~"|"~D)) 
)
par(oldpar)


# In this example we estimate the parameter for a dataset with ----------------
# With 5,000 draws from a MVN() with parameters M and S.
\donttest{
# Loading the required packages
library(mvtnorm)
library(coda)

# Parameters and data simulation
S <- cbind(c(.8, .2), c(.2, 1))
M <- c(0, 1)

set.seed(123)
D <- rmvnorm(5e3, mean = M, sigma = S)

# Function to pass to MCMC
fun <- function(pars) {
  # Putting the parameters in a sensible way
  m <- pars[1:2]
  s <- cbind( c(pars[3], pars[4]), c(pars[4], pars[5]) )
  
  # Computing the unnormalized log likelihood
  sum(log(dmvnorm(D, m, s)))
}

# Calling MCMC
ans <- MCMC(
  initial = c(mu0=5, mu1=5, s0=5, s01=0, s2=5), 
  fun,
  kernel  = kernel_normal_reflective(
    lb    = c(-10, -10, .01, -5, .01),
    ub    = 5,
    scale = 0.01
  ),
  nsteps  = 1e4,
  thin    = 20,
  burnin  = 5e3
)

# Checking out the outcomes
plot(ans)
summary(ans)

# Multiple chains -----------------------------------------------------------

# As we want to run -fun- in multiple cores, we have to
# pass -D- explicitly (unless using Fork Clusters)
# just like specifying that we are calling a function from the
# -mvtnorm- package.
  
fun <- function(pars, D) {
  # Putting the parameters in a sensible way
  m <- pars[1:2]
  s <- cbind( c(pars[3], pars[4]), c(pars[4], pars[5]) )
  
  # Computing the unnormalized log likelihood
  sum(log(mvtnorm::dmvnorm(D, m, s)))
}

# Two chains
ans <- MCMC(
  initial = c(mu0=5, mu1=5, s0=5, s01=0, s2=5), 
  fun,
  nchains = 2,
  kernel  = kernel_normal_reflective(
    lb    = c(-10, -10, .01, -5, .01),
    ub    = 5,
    scale = 0.01
  ),
  nsteps  = 1e4,
  thin    = 20,
  burnin  = 5e3,
  D       = D
)

summary(ans)
}

}
\references{
Brooks, S., Gelman, A., Jones, G. L., & Meng, X. L. (2011). Handbook of
Markov Chain Monte Carlo. Handbook of Markov Chain Monte Carlo.
}
