% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/finalfit.R
\name{finalfit}
\alias{finalfit}
\alias{finalfit.lm}
\alias{finalfit.glm}
\alias{finalfit.coxph}
\title{Final output tables for common regression models}
\usage{
finalfit(.data, dependent, explanatory, explanatory_multi = NULL,
  random_effect = NULL, keep_models = FALSE, metrics = FALSE,
  add_dependent_label = TRUE, dependent_label_prefix = "Dependent: ",
  dependent_label_suffix = "", ...)

finalfit.lm(.data, dependent, explanatory, explanatory_multi = NULL,
  random_effect = NULL, keep_models = FALSE, metrics = FALSE,
  add_dependent_label = TRUE, dependent_label_prefix = "Dependent: ",
  dependent_label_suffix = "", ...)

finalfit.glm(.data, dependent, explanatory, explanatory_multi = NULL,
  random_effect = NULL, keep_models = FALSE, metrics = FALSE,
  add_dependent_label = TRUE, dependent_label_prefix = "Dependent: ",
  dependent_label_suffix = "", ...)

finalfit.coxph(.data, dependent, explanatory, explanatory_multi = NULL,
  random_effect = NULL, keep_models = FALSE, metrics = FALSE,
  add_dependent_label = TRUE, dependent_label_prefix = "Dependent: ",
  dependent_label_suffix = "", ...)
}
\arguments{
\item{.data}{Dataframe.}

\item{dependent}{Character vector of length 1:  quoted name of depdendent
variable. Can be continuous, a binary factor, or a survival object of form
\code{Surv(time, status)}}

\item{explanatory}{Character vector of any length: quoted name(s) of
explanatory variables.}

\item{explanatory_multi}{Character vector of any length: quoted name(s) of a
subset of explanatory variables to generate reduced multivariable model
(must only contain variables contained in \code{explanatory}).}

\item{random_effect}{Character vector of length 1: quoted name of random
effects variable. When included mixed effects model generated
(\code{lme4::glmer lme4::lmer}).}

\item{keep_models}{Logical: include full multivariable model in output when
working with reduced multivariable model (\code{explanatory_multi}) and/or
mixed effect models (\code{random_effect}).}

\item{metrics}{Logical: include useful model metrics in output in
publication format.}

\item{add_dependent_label}{Add the name of the dependent label to the top
left of table.}

\item{dependent_label_prefix}{Add text before dependent label.}

\item{dependent_label_suffix}{Add text after dependent label.}

\item{...}{Other arguments to pass to \code{\link{fit2df}}:
\code{estimate_name, digits, confint_type, confint_level,
confint_sep}.}
}
\value{
Returns a dataframe with the final model table.
}
\description{
An "all-in-one" function that takes a single dependent variable with a vector
of explanatory variable names (continuous or categorical variables) to
produce a final table for publication including summary statistics. The
appropriate model is selected on the basis of dependent variable and whether
a random effect is specified.

\code{finalfit.lm} method (not called directly)

\code{finalfit.glm} method (not called directly)

\code{finalfit.coxph} method (not called directly)
}
\examples{
library(finalfit)
library(dplyr)

# Summary, univariable and multivariable analyses of the form:
# glm(depdendent ~ explanatory, family="binomial")
# lmuni(), lmmulti(), lmmixed(), glmuni(), glmmulti(), glmmixed(), glmmultiboot(),
#   coxphuni(), coxphmulti()

data(colon_s) # Modified from survival::colon
explanatory = c("age.factor", "sex.factor", "obstruct.factor", "perfor.factor")
dependent = 'mort_5yr'
colon_s \%>\%
  finalfit(dependent, explanatory)

# Multivariable analysis with subset of explanatory
#   variable set used in univariable analysis
explanatory = c("age.factor", "sex.factor", "obstruct.factor", "perfor.factor")
explanatory_multi = c("age.factor", "obstruct.factor")
dependent = "mort_5yr"
colon_s \%>\%
  finalfit(dependent, explanatory, explanatory_multi)

# Summary, univariable and multivariable analyses of the form:
# lme4::glmer(dependent ~ explanatory + (1 | random_effect), family="binomial")

explanatory = c("age.factor", "sex.factor", "obstruct.factor", "perfor.factor")
explanatory_multi = c("age.factor", "obstruct.factor")
random_effect = "hospital"
dependent = "mort_5yr"
colon_s \%>\%
  finalfit(dependent, explanatory, explanatory_multi, random_effect)

# Include model metrics:
colon_s \%>\%
  finalfit(dependent, explanatory, explanatory_multi,  metrics=TRUE)

# Summary, univariable and multivariable analyses of the form:
# survival::coxph(dependent ~ explanatory)

explanatory = c("age.factor", "sex.factor", "obstruct.factor", "perfor.factor")
dependent = "Surv(time, status)"

colon_s \%>\%
  finalfit(dependent, explanatory)

# Rather than going all-in-one, any number of subset models can
# be manually added on to a summary_factorlist() table using finalfit.merge().
# This is particularly useful when models take a long-time to run or are complicated.

# Note requirement for fit_id=TRUE.
# `fit2df` is a subfunction extracting most common models to a dataframe.

explanatory = c("age.factor", "sex.factor", "obstruct.factor", "perfor.factor")
dependent = 'mort_5yr'
colon_s \%>\%
  finalfit(dependent, explanatory, metrics=TRUE)

explanatory = c("age.factor", "sex.factor", "obstruct.factor", "perfor.factor")
explanatory_multi = c("age.factor", "obstruct.factor")
random_effect = "hospital"
dependent = 'mort_5yr'

# Separate tables
colon_s \%>\%
  summary_factorlist(dependent, explanatory, fit_id=TRUE) -> example.summary

colon_s \%>\%
  glmuni(dependent, explanatory) \%>\%
  fit2df(estimate_suffix=" (univariable)") -> example.univariable

colon_s \%>\%
  glmmulti(dependent, explanatory) \%>\%
  fit2df(estimate_suffix=" (multivariable)") -> example.multivariable

colon_s \%>\%
  glmmixed(dependent, explanatory, random_effect) \%>\%
  fit2df(estimate_suffix=" (multilevel") -> example.multilevel

# Pipe together
example.summary \%>\%
  finalfit_merge(example.univariable) \%>\%
  finalfit_merge(example.multivariable) \%>\%
  finalfit_merge(example.multilevel) \%>\%
  select(-c(fit_id, index)) \%>\%
  dependent_label(colon_s, dependent) -> example.final
example.final

}
\concept{\code{finalfit} all-in-one functions}
