% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sub_dann.R
\name{sub_dann_df}
\alias{sub_dann_df}
\title{Discriminant Adaptive Nearest Neighbor With Subspace Reduction}
\usage{
sub_dann_df(
  formula,
  train,
  test,
  k = 5,
  neighborhood_size = max(floor(nrow(train)/5), 50),
  epsilon = 1,
  probability = FALSE,
  weighted = FALSE,
  sphere = "mcd",
  numDim = ceiling(ncol(train)/2)
)
}
\arguments{
\item{formula}{An object of class formula. (Y ~ X1 + X2)}

\item{train}{A data frame or tibble containing training data.}

\item{test}{A data frame or tibble containing test data.}

\item{k}{The number of data points used for final classification.}

\item{neighborhood_size}{The number of data points used to calculate between and within class covariance.}

\item{epsilon}{Diagonal elements of a diagonal matrix. 1 is the identity matrix.}

\item{probability}{Should probabilities instead of classes be returned?}

\item{weighted}{weighted argument to ncoord. See \code{\link[fpc]{ncoord}} for details.}

\item{sphere}{One of "mcd", "mve", "classical", or "none" See \code{\link[fpc]{ncoord}} for details.}

\item{numDim}{Dimension of subspace used by dann. See \code{\link[fpc]{ncoord}} for details.}
}
\value{
A numeric vector containing predicted class or a numeric matrix containing class probabilities.
}
\description{
Discriminant Adaptive Nearest Neighbor With Subspace Reduction
}
\details{
An implementation of Hastie and Tibshirani's sub-dann in section 4.1 of
\href{https://web.stanford.edu/~hastie/Papers/dann_IEEE.pdf}{Discriminant Adaptive Nearest
Neighbor Classification publication.}.

dann's performance suffers when noise variables are included in the model. Simulations show sub_dann
will generally be more performant in this scenario.
}
\examples{
library(dann)
library(mlbench)
library(magrittr)
library(dplyr)
library(ggplot2)

######################
# Circle data with unrelated variables
######################
set.seed(1)
train <- mlbench.circle(300, 2) \%>\%
  tibble::as_tibble()
colnames(train)[1:3] <- c("X1", "X2", "Y")
train <- train \%>\%
  mutate(Y = as.numeric(Y))

# Add 5 unrelated variables
train <- train \%>\%
  mutate(
    U1 = runif(300, -1, 1),
    U2 = runif(300, -1, 1),
    U3 = runif(300, -1, 1),
    U4 = runif(300, -1, 1),
    U5 = runif(300, -1, 1)
  )

test <- mlbench.circle(100, 2) \%>\%
  tibble::as_tibble()
colnames(test)[1:3] <- c("X1", "X2", "Y")
test <- test \%>\%
  mutate(Y = as.numeric(Y))

# Add 5 unrelated variables
test <- test \%>\%
  mutate(
    U1 = runif(100, -1, 1),
    U2 = runif(100, -1, 1),
    U3 = runif(100, -1, 1),
    U4 = runif(100, -1, 1),
    U5 = runif(100, -1, 1)
  )

dannPreds <- dann_df(
  formula = Y ~ X1 + X2 + U1 + U2 + U3 + U4 + U5,
  train = train, test = test,
  k = 3, neighborhood_size = 50, epsilon = 1,
  probability = FALSE
)
mean(dannPreds == test$Y) # Not a good model

# Graph suggests a subspace with 2 dimensions. (The correct answer.)
graph_eigenvalues_df(
  formula = Y ~ X1 + X2 + U1 + U2 + U3 + U4 + U5, train = train,
  neighborhood_size = 50, weighted = FALSE, sphere = "mcd"
)

subDannPreds <- sub_dann_df(
  formula = Y ~ X1 + X2 + U1 + U2 + U3 + U4 + U5,
  train = train, test = test,
  k = 3, neighborhood_size = 50, epsilon = 1,
  probability = FALSE,
  weighted = FALSE, sphere = "classical", numDim = 2
)
# sub_dan does much better when unrelated variables are present.
mean(subDannPreds == test$Y)

rm(train, test)
rm(dannPreds, subDannPreds)
}
