% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/baseline_wrappers.R
\name{baseline_multinomial}
\alias{baseline_multinomial}
\title{Create baseline evaluations}
\usage{
baseline_multinomial(
  test_data,
  dependent_col,
  n = 100,
  metrics = list(),
  random_generator_fn = runif,
  parallel = FALSE
)
}
\arguments{
\item{test_data}{\code{data.frame}.}

\item{dependent_col}{Name of dependent variable in the supplied test and training sets.}

\item{n}{The number of sets of random predictions to evaluate. (Default is \code{100})}

\item{metrics}{\code{list} for enabling/disabling metrics.

  E.g. \code{list("F1" = FALSE)} would remove \code{F1} from the results,
  and \code{list("Accuracy" = TRUE)} would add the regular \code{Accuracy} metric
  to the results.
  Default values (\code{TRUE}/\code{FALSE}) will be used for the remaining available metrics.

  You can enable/disable all metrics at once by including
  \code{"all" = TRUE/FALSE} in the \code{list}. This is done prior to enabling/disabling
  individual metrics, why f.i. \code{list("all" = FALSE, "Accuracy" = TRUE)}
  would return only the \code{Accuracy} metric.

  The \code{list} can be created with
  \code{\link[cvms:multinomial_metrics]{multinomial_metrics()}}.

  Also accepts the string \code{"all"}.}

\item{random_generator_fn}{Function for generating random numbers.
 The \code{softmax} function is applied to the generated numbers to transform them to probabilities.

 The first argument must be the number of random numbers to generate,
 as no other arguments are supplied.

 To test the effect of using different functions,
 see \code{\link[cvms:multiclass_probability_tibble]{multiclass_probability_tibble()}}.}

\item{parallel}{Whether to run the \code{`n`} evaluations in parallel. (Logical)

 Remember to register a parallel backend first.
 E.g. with \code{doParallel::registerDoParallel}.}
}
\value{
\code{list} containing:

 \enumerate{
  \item a \code{tibble} with summarized results (called \code{summarized_metrics})
  \item a \code{tibble} with random evaluations (\code{random_evaluations})
  \item a \code{tibble} with the summarized class level results
        (\code{summarized_class_level_results})
 }

 ....................................................................

 \subsection{Macro metrics}{

 Based on the generated predictions,
 \emph{one-vs-all} (binomial) evaluations are performed and aggregated
 to get the following \strong{macro} metrics:

 \strong{\code{Balanced Accuracy}},
 \strong{\code{F1}},
 \strong{\code{Sensitivity}},
 \strong{\code{Specificity}},
 \strong{\code{Positive Predictive Value}},
 \strong{\code{Negative Predictive Value}},
 \strong{\code{Kappa}},
 \strong{\code{Detection Rate}},
 \strong{\code{Detection Prevalence}}, and
 \strong{\code{Prevalence}}.

 In general, the metrics mentioned in
 \code{\link[cvms:binomial_metrics]{binomial_metrics()}}
 can be enabled as macro metrics
 (excluding \code{MCC}, \code{AUC}, \code{Lower CI},
 \code{Upper CI}, and the \code{AIC/AICc/BIC} metrics).
 These metrics also has a weighted average
 version.

 \strong{N.B.} we also refer to the \emph{one-vs-all evaluations} as the \emph{class level results}.
 }

 \subsection{Multiclass metrics}{

 In addition, the \strong{\code{Overall Accuracy}} and \emph{multiclass}
 \strong{\code{MCC}} metrics are computed. \emph{Multiclass} \code{AUC} can be enabled but
 is slow to calculate with many classes.
 }
 ....................................................................

 The \strong{Summarized Results} \code{tibble} contains:

 Summary of the random evaluations.

 \strong{How}: The one-vs-all binomial evaluations are aggregated by repetition and summarized. Besides the
 metrics from the binomial evaluations, it
 also includes \strong{\code{Overall Accuracy}} and \emph{multiclass} \strong{\code{MCC}}.

 The \strong{Measure} column indicates the statistical descriptor used on the evaluations.
 The \strong{Mean}, \strong{Median}, \strong{SD}, \strong{IQR}, \strong{Max}, \strong{Min},
 \strong{NAs}, and \strong{INFs} measures describe the \emph{Random Evaluations} \code{tibble},
 while the \strong{CL_Max}, \strong{CL_Min}, \strong{CL_NAs}, and
 \strong{CL_INFs} describe the \strong{C}lass \strong{L}evel results.

 The rows where \code{Measure == All_<<class name>>} are the evaluations when all
 the observations are predicted to be in that class.

 ....................................................................

 The \strong{Summarized Class Level Results} \code{tibble} contains:

 The (nested) summarized results for each class, with the same metrics and descriptors as
 the \emph{Summarized Results} \code{tibble}. Use \code{\link[tidyr:unnest]{tidyr::unnest}}
 on the \code{tibble} to inspect the results.

 \strong{How}: The one-vs-all evaluations are summarized by class.

 The rows where \code{Measure == All_0} are the evaluations when none of the observations
 are predicted to be in that class, while the rows where \code{Measure == All_1} are the
 evaluations when all of the observations are predicted to be in that class.

 ....................................................................

 The \strong{Random Evaluations} \code{tibble} contains:

 The repetition results with the same metrics as the \emph{Summarized Results} \code{tibble}.

 \strong{How}: The one-vs-all evaluations are aggregated by repetition.
 If a metric contains one or more \code{NAs} in the one-vs-all evaluations, it
 will lead to an \code{NA} result for that repetition.

 Also includes:

 A nested \code{tibble} with the one-vs-all binomial evaluations (\strong{Class Level Results}),
 including nested \strong{Confusion Matrices} and the
 \strong{Support} column, which is a count of how many observations from the
 class is in the test set.

 A nested \code{tibble} with the \strong{predictions} and targets.

 A \code{list} of \strong{ROC} curve objects.

 A nested \code{tibble} with the multiclass \strong{confusion matrix}.

 A nested \strong{Process} information object with information
 about the evaluation.

 Name of \strong{dependent} variable.
}
\description{
\Sexpr[results=rd, stage=render]{lifecycle::badge("maturing")}

 Create a baseline evaluation of a test set.

 In modelling, a \emph{baseline} is a result that
 is meaningful to compare the results from our models to. For instance, in
 classification, we usually want our results to be better than \emph{random guessing}.
 E.g. if we have three classes, we can expect an accuracy of \code{33.33\%}, as for every
 observation we have \code{1/3} chance of guessing the correct class. So our model should achieve
 a higher accuracy than \code{33.33\%} before it is more useful to us than guessing.

 While this expected value is often fairly straightforward to find analytically, it
 only represents what we can expect on average. In reality, it's possible to get far better
 results than that by guessing.
 \strong{\code{baseline_multinomial()}}
 finds the range of likely values by evaluating multiple sets
 of random predictions and summarizing them with a set of useful descriptors.

 Technically, it creates \emph{one-vs-all} (binomial) baseline evaluations
 for the \code{`n`} sets of random predictions and summarizes them. Additionally,
 sets of "all class x,y,z,..." predictions are evaluated.
}
\details{
Packages used:

 Multiclass \code{ROC} curve and \code{AUC}:
 \code{\link[pROC:multiclass.roc]{pROC::multiclass.roc}}
}
\examples{
\donttest{
# Attach packages
library(cvms)
library(groupdata2) # partition()
library(dplyr) # \%>\% arrange()
library(tibble)

# Data is part of cvms
data <- participant.scores

# Set seed for reproducibility
set.seed(1)

# Partition data
partitions <- partition(data, p = 0.7, list_out = TRUE)
train_set <- partitions[[1]]
test_set <- partitions[[2]]

# Create baseline evaluations
# Note: usually n=100 is a good setting

# Create some data with multiple classes
multiclass_data <- tibble(
  "target" = rep(paste0("class_", 1:5), each = 10)
) \%>\%
  dplyr::sample_n(35)

baseline_multinomial(
  test_data = multiclass_data,
  dependent_col = "target",
  n = 4
)

# Parallelize evaluations

# Attach doParallel and register four cores
# Uncomment:
# library(doParallel)
# registerDoParallel(4)

# Make sure to uncomment the parallel argument
(mb <- baseline_multinomial(
  test_data = multiclass_data,
  dependent_col = "target",
  n = 6
  #, parallel = TRUE  # Uncomment
))

# Inspect the summarized class level results
# for class_2
mb$summarized_class_level_results \%>\%
  dplyr::filter(Class == "class_2") \%>\%
  tidyr::unnest(Results)

# Multinomial with custom random generator function
# that creates very "certain" predictions
# (once softmax is applied)

rcertain <- function(n) {
  (runif(n, min = 1, max = 100)^1.4) / 100
}

# Make sure to uncomment the parallel argument
baseline_multinomial(
  test_data = multiclass_data,
  dependent_col = "target",
  n = 6,
  random_generator_fn = rcertain
  #, parallel = TRUE  # Uncomment
)
}
}
\seealso{
Other baseline functions: 
\code{\link{baseline}()},
\code{\link{baseline_binomial}()},
\code{\link{baseline_gaussian}()}
}
\author{
Ludvig Renbo Olsen, \email{r-pkgs@ludvigolsen.dk}
}
\concept{baseline functions}
