% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Design-methods.R
\docType{methods}
\name{examine}
\alias{examine}
\alias{examine,Design-method}
\alias{examine,RuleDesign-method}
\title{Obtain hypothetical trial course table for a design}
\usage{
examine(object, ...)

\S4method{examine}{Design}(object, mcmcOptions = McmcOptions(), ...)

\S4method{examine}{RuleDesign}(object, ...)
}
\arguments{
\item{object}{the design (\code{\linkS4class{Design}} or
\code{\linkS4class{RuleDesign}} object) we want to examine}

\item{mcmcOptions}{object of class \code{\linkS4class{McmcOptions}},
giving the MCMC options for each evaluation in the trial. By default,
the standard options are used}

\item{\dots}{additional arguments (see methods)}
}
\value{
The data frame
}
\description{
This generic function takes a design and generates a dataframe
showing the beginning of several hypothetical trial courses under
the design. This means, from the generated dataframe one can read off:
- how many cohorts are required in the optimal case (no DLTs observed) in
  order to reach the highest dose of the specified dose grid
- assuming no DLTs are observed until a certain dose level, what the next
  recommended dose is for all possible number of DLTs observed
- the actual relative increments that will be used in these cases
- whether the trial would stop at a certain cohort
Examining the "single trial" behavior of a dose escalation design is
the first important step in evaluating a design, and cannot be replaced by
studying solely the operating characteristics in "many trials". The cohort
sizes are also taken from the design, assuming no DLTs occur until the dose
listed.
}
\section{Methods (by class)}{
\itemize{
\item \code{Design}: Examine a model-based CRM

\item \code{RuleDesign}: Examine a rule-based design
}}
\examples{

# Define the dose-grid
emptydata <- Data(doseGrid = c(1, 3, 5, 10, 15, 20, 25))

# Initialize the CRM model 
model <- LogisticLogNormal(mean=c(-0.85, 1),
                           cov=
                             matrix(c(1, -0.5, -0.5, 1),
                                    nrow=2),
                           refDose=56)

# Choose the rule for selecting the next dose 
myNextBest <- NextBestNCRM(target=c(0.2, 0.35),
                           overdose=c(0.35, 1),
                           maxOverdoseProb=0.25)

# Choose the rule for the cohort-size 
mySize1 <- CohortSizeRange(intervals=c(0, 30),
                           cohortSize=c(1, 3))
mySize2 <- CohortSizeDLT(DLTintervals=c(0, 1),
                         cohortSize=c(1, 3))
mySize <- maxSize(mySize1, mySize2)

# Choose the rule for stopping
myStopping1 <- StoppingMinCohorts(nCohorts=3)
myStopping2 <- StoppingTargetProb(target=c(0.2, 0.35),
                                  prob=0.5)
myStopping3 <- StoppingMinPatients(nPatients=20)
myStopping <- (myStopping1 & myStopping2) | myStopping3

# Choose the rule for dose increments
myIncrements <- IncrementsRelative(intervals=c(0, 20),
                                   increments=c(1, 0.33))

# Initialize the design
design <- Design(model=model,
                 nextBest=myNextBest,
                 stopping=myStopping,
                 increments=myIncrements,
                 cohortSize=mySize,
                 data=emptydata,
                 startingDose=3)

# Examine the design
set.seed(4235)
# MCMC parameters are set to small values only to show this example. They should be
# increased for a real case.
options <- McmcOptions(burnin=10,step=1,samples=20)
examine(design,options=options)
  

# Define the dose-grid
emptydata <- Data(doseGrid = c(5, 10, 15, 25, 35, 50, 80))

# inizialing a 3+3 design with constant cohort size of 3 and
# starting dose equal 5
myDesign <- RuleDesign(nextBest = NextBestThreePlusThree(),
                       cohortSize = CohortSizeConst(size=3L),
                       data = emptydata,
                       startingDose = 5)
  
# Examine the design
set.seed(4235)
examine(myDesign)
}
\keyword{methods}
\keyword{regression}

