% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/randwin.R
\name{randwin}
\alias{randwin}
\title{Climate window analysis for randomised data}
\usage{
randwin(exclude = NA, repeats = 5, window = "sliding", xvar, cdate, bdate,
  baseline, stat, range, func, type, refday, cmissing = FALSE,
  cinterval = "day", spatial = NULL, cohort = NULL, upper = NA,
  lower = NA, binary = FALSE, centre = list(NULL, "both"), k = 0,
  weightfunc = "W", par = c(3, 0.2, 0), control = list(ndeps = c(0.01,
  0.01, 0.01)), method = "L-BFGS-B", cutoff.day = NULL,
  cutoff.month = NULL, furthest = NULL, closest = NULL, thresh = NULL,
  cvk = NULL)
}
\arguments{
\item{exclude}{Two values (distance and duration) which allow users
to exclude short-duration long-lag climate windows from analysis (e.g., 
windows with a duration of 10 days which occur over a month ago).
These windows are often considered to be biologically implausible.}

\item{repeats}{The number of times that data will be randomised and analysed 
for climate windows.}

\item{window}{Whether randomisations are carried out for a sliding window ("sliding")
or weighted window ("weighted") approach.}

\item{xvar}{A list object containing all climate variables of interest. 
Please specify the parent environment and variable name (e.g. Climate$Temp).}

\item{cdate}{The climate date variable (dd/mm/yyyy). Please specify the parent
environment and variable name (e.g. Climate$Date).}

\item{bdate}{The biological date variable (dd/mm/yyyy). Please specify the 
parent environment and variable name (e.g. Biol$Date).}

\item{baseline}{The baseline model structure used for testing correlation. 
Currently known to support lm, glm, lmer and glmer objects.}

\item{stat}{If window = "sliding"; The aggregate statistic used to analyse the climate data. Can 
currently use basic R statistics (e.g. mean, min), as well as slope. 
Additional aggregate statistics can be created using the format function(x)
(...). See FUN in \code{\link{apply}} for more detail.}

\item{range}{Two values signifying respectively the furthest and closest number 
of time intervals (set by cinterval) back from the cutoff date or biological record to include 
in the climate window search.}

\item{func}{The functions used to fit the climate variable. Can be linear 
("lin"), quadratic ("quad"), cubic ("cub"), inverse ("inv") or log ("log").}

\item{type}{"absolute" or "relative", whether you wish the climate window to be relative
(e.g. the number of days before each biological record is measured) or absolute
(e.g. number of days before a set point in time).}

\item{refday}{If type is absolute, the day and month respectively of the 
year from which the absolute window analysis will start.}

\item{cmissing}{TRUE or FALSE, determines what should be done if there are 
missing climate data. If FALSE, the function will not run if missing 
climate data is encountered. If TRUE, any records affected by missing 
climate data will be removed from climate window analysis.}

\item{cinterval}{The resolution at which climate window analysis will be 
conducted. May be days ("day"), weeks ("week"), or months ("month"). Note the units
of parameter 'range' will differ depending on the choice
of cinterval.}

\item{spatial}{A list item containing:
1. A factor that defines which spatial group (i.e. population) each biological
record is taken from. The length of this factor should correspond to the length 
of the biological dataset.
2. A factor that defines which spatial group (i.e. population) climate data
corresponds to. This length of this factor should correspond to the length of
the climate dataset.}

\item{cohort}{A variable used to group biological records that occur in the same biological
season but cover multiple years (e.g. southern hemisphere breeding season). Only required
when type is "absolute". The cohort variable should be in the same dataset as the variable bdate.}

\item{upper}{Cut-off values used to determine growing degree days or positive 
climate thresholds (depending on parameter thresh). Note that when values
of lower and upper are both provided, climatewin will instead calculate an 
optimal climate zone.}

\item{lower}{Cut-off values used to determine chill days or negative 
climate thresholds (depending on parameter thresh). Note that when values
of lower and upper are both provided, climatewin will instead calculate an 
optimal climate zone.}

\item{binary}{TRUE or FALSE. Determines whether to use values of upper and
lower to calculate binary climate data (thresh = TRUE), or to use for
growing degree days (thresh = FALSE).}

\item{centre}{A list item containing:
1. The variable used for mean centring (e.g. Year, Site, Individual). 
Please specify the parent environment and variable name (e.g. Biol$Year).
2. Whether the model should include both within-group means and variance ("both"),
only within-group means ("mean"), or only within-group variance ("var").}

\item{k}{If window = "sliding"; the number of folds used for k-fold cross validation. By default
this value is set to 0, so no cross validation occurs. Value should be a
minimum of 2 for cross validation to occur.}

\item{weightfunc}{If window = "weighted"; 
the distribution to be used for optimisation. Can be 
either a Weibull ("W") or Generalised Extreme Value distribution ("G").}

\item{par}{If window = "weighted"; the shape, scale and location parameters 
of the Weibull or GEV weight function used as start weight function. 
For Weibull : Shape and scale parameters must be greater than 0, 
while location parameter must be less than or equal to 0. 
For GEV : Scale parameter must be greater than 0.}

\item{control}{If window = "weighted";
parameters used to determine step size for the optimisation 
function. Please see \code{\link{optim}} for more detail.}

\item{method}{If window = "weighted"; 
the method used for the optimisation function. Please see 
\code{\link{optim}} for more detail.}

\item{cutoff.day, cutoff.month}{Redundant parameters. Now replaced by refday.}

\item{furthest, closest}{Redundant parameters. Now replaced by range.}

\item{thresh}{Redundant parameter. Now replaced by binary.}

\item{cvk}{Redundant parameter. Now replaced by k.}
}
\value{
Returns a dataframe containing information on the best climate
 window from each randomisation. See \code{\link{MassRand}} as an example.
}
\description{
Randomises biological data and carries out a climate window analysis. Used
to help determine the chance of obtaining an observed result at random.
}
\examples{
\dontrun{

## EXAMPLE 1 ##

# Test climate windows in randomised data using a sliding window approach.

data(Mass)
data(MassClimate)

# Randomise data twice
# Note all other parameters are fitted in the same way as the climatewin function.

rand <- randwin(repeats = 2, window = "sliding", 
               xvar = list(Temp = MassClimate$Temp), 
               cdate = MassClimate$Date, bdate = Mass$Date,
               baseline = lm(Mass ~ 1, data = Mass), 
               range = c(100, 0),
               stat = "mean", func = "lin", type = "absolute", 
               refday = c(20, 5),
               cmissing = FALSE, cinterval = "day")
                
# View output #

head(rand)

## EXAMPLE 2 ##

# Test climate windows in randomised data using a weighted window approach.
  
data(Offspring)
data(OffspringClimate)

# Randomise data twice
# Note all other parameters are fitted in the same way as the weightwin function.

weightrand <- randwin(repeats = 2, window = "weighted", 
                     xvar = list(Temp = OffspringClimate$Temperature), 
                     cdate = OffspringClimate$Date,
                     bdate = Offspring$Date,
                     baseline = glm(Offspring ~ 1, family = poisson, data = Offspring),
                     range = c(365, 0), func = "quad",
                     type = "relative", weightfunc = "W", cinterval = "day",
                     par = c(3, 0.2, 0), control = list(ndeps = c(0.01, 0.01, 0.01)),
                     method = "L-BFGS-B")
                   
# View output

head(weightrand)
                          
       }

}
\author{
Liam D. Bailey and Martijn van de Pol
}

