% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/boot.cgaim.R
\name{boot.cgaim}
\alias{boot.cgaim}
\title{Boostrap CGAIM}
\usage{
boot.cgaim(object, boot.type = c("residuals", "wild", "pairs"),
  bsamples = NULL, B = 100, l = 1, nc = 1)
}
\arguments{
\item{object}{A \code{cgaim} object.}

\item{boot.type}{The type of bootstrap to perform. Currently
available type are \code{"residuals"}, \code{"wild"} and 
\code{"pairs"}. See details}

\item{bsamples}{A numerical matrix of observation indices specifying 
bootstrap samples.
Rows indicate observations and columns bootstrap samples.
If \code{NULL} (the default), samples are generated internally.}

\item{B}{Number of bootstrap samples to generate when \code{bsamples = NULL}.}

\item{l}{Block length for block-bootstrap. Samples are generated by
resampling block of observation of length \code{l}. The classical
bootstrap corresponds to \code{l = 1} (the default).}

\item{nc}{Positive integer. If \code{nc > 1}, the function is parallelized with \code{nc} indicating the number of cores to use.}
}
\value{
A \code{boot.cgaim} object with components
  \item{\code{boot}}{The bootstrap result. A list that includes all
    \code{B} replications of \code{alpha}, \code{beta}, \code{gfit} and
    \code{indexfit} organized in arrays.}
  \item{\code{obs}}{The original \code{object} passed to the function.}
  \item{\code{samples}}{The bootstrap samples. A matrix with indices 
    corresponding to original observations.}
  \item{\code{boot.type}}{The type of bootstrap performed.}
  \item{\code{B}}{The number of bootstrap replications.}
  \item{\code{l}}{The block length for block bootstrap.}
}
\description{
Generates bootstrap replicates of a \code{cgaim} object.
}
\details{
This function fits the \code{cgaim} on bootstrap samples.
It is called internally by the \code{\link{confint.cgaim}} function, but can also be
called directly to generate various statistics.

Three types of bootstrap are currently implemented. \code{"residuals"} 
(the default) resamples the residuals in \code{object} to then be added to fitted values, creating alternative response vectors. The \code{cgaim} is then fitted on these newly generated y values with the original x. \code{"wild"} is
similar except that residuals are multiplied by random draws from a
standard normal distribution before being added to fitted values. 
\code{"pairs"} resamples directly pairs of y and x to create 
bootstrap samples.

Bootstrap samples can either be prespecified by the user through
\code{bsamples} or generated internally. In the former case, 
the columns of \code{bsamples} indicate the number of replications \code{B} and
the rows should match the original number of observations. Internally
generated bootstrap samples are controlled by the number of replications
\code{B} and block length \code{l}, implementing block bootstrap.
The latter is particularly recommended for time series data.

As fitting a large number of \code{cgaim} models can be computationally
intensive, the function can be run in parallel, using the 
\code{\link{doParallel}} package. This can be done by setting the
argument \code{nc} to a value greater than 1, controlling the number
of cores used in parallelization.
}
\examples{
# A simple CGAIM
n <- 200
x1 <- rnorm(n)
x2 <- x1 + rnorm(n)
z <- x1 + x2
y <- z + rnorm(n)
df1 <- data.frame(y, x1, x2) 
ans <- cgaim(y ~ g(x1, x2, acons = list(monotone = 1)), data = df1)

# Use function to compute confidence intervals (B should be increased)
set.seed(1989) 
boot1 <- boot.cgaim(ans, B = 10)
ci1 <- confint(boot1)

# Produces the same result as
set.seed(1989)
ci2 <- confint(ans, type = "boot", B = 10)

# Create sampling beforehand
bsamp <- matrix(sample(1:n, n * 10, replace = TRUE), n)
boot2 <- boot.cgaim(ans, bsamples = bsamp)

# Parallel computing (two cores)
\donttest{
boot3 <- boot.cgaim(ans, nc = 2)
}

}
