% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/orsf_vi.R
\name{orsf_vi}
\alias{orsf_vi}
\alias{orsf_vi_negate}
\alias{orsf_vi_permute}
\alias{orsf_vi_anova}
\title{ORSF variable importance}
\usage{
orsf_vi(object, group_factors = TRUE, importance = NULL, oobag_fun = NULL, ...)

orsf_vi_negate(object, group_factors = TRUE, oobag_fun = NULL, ...)

orsf_vi_permute(object, group_factors = TRUE, oobag_fun = NULL, ...)

orsf_vi_anova(object, group_factors = TRUE, ...)
}
\arguments{
\item{object}{(\emph{orsf_fit}) a trained oblique random survival forest
(see \link{orsf}).}

\item{group_factors}{(\emph{logical}) if \code{TRUE}, the importance of factor variables will be reported overall by aggregating the importance of individual levels of the factor. If \code{FALSE}, the importance of individual factor levels will be returned.}

\item{importance}{(\emph{character}) Indicate method for variable importance:
\itemize{
\item 'anova': compute analysis of variance (ANOVA) importance
\item 'negate': compute negation importance
\item 'permute': compute permutation importance
}}

\item{oobag_fun}{(\emph{function}) to be used for evaluating out-of-bag prediction accuracy after negating coefficients
(if importance = 'negate') or permuting the values of a predictor
(if importance = 'permute')
\itemize{
\item When \code{oobag_fun = NULL} (the default), Harrell's C-statistic (1982) is used to evaluate accuracy.
\item if you use your own \code{oobag_fun} note the following:
\itemize{
\item \code{oobag_fun} should have two inputs: \code{y_mat} and \code{s_vec}
\item \code{y_mat} is a two column matrix with first column named 'time', second named 'status'
\item \code{s_vec} is a numeric vector containing predicted survival probabilities.
\item \code{oobag_fun} should return a numeric output of length 1
\item the same \code{oobag_fun} should have been used when you created \code{object}
so that the initial value of out-of-bag prediction accuracy is
consistent with the values that will be computed while variable
importance is estimated.
}
}

For more details, see the out-of-bag
\href{https://docs.ropensci.org/aorsf/articles/oobag.html}{vignette}.}

\item{...}{Further arguments passed to or from other methods (not currently used).}
}
\value{
\code{orsf_vi} functions return a named numeric vector.
\itemize{
\item Names of the vector are the predictor variables used by \code{object}
\item Values of the vector are the estimated importance of the given predictor.
}

The returned vector is sorted from highest to lowest value, with higher
values indicating higher importance.
}
\description{
Estimate the importance of individual variables using oblique random
survival forests.
}
\details{
When an \code{orsf_fit} object is fitted with importance = 'anova', 'negate', or
'permute', the output will have a vector of importance values based on
the requested type of importance. However, you may still want to call
\code{orsf_vi()} on this output if you want to group factor levels into one
overall importance value.

\code{orsf_vi()} is a general purpose function to extract or compute variable
importance estimates from an \code{'orsf_fit'} object (see \link{orsf}).
\code{orsf_vi_negate()}, \code{orsf_vi_permute()}, and \code{orsf_vi_anova()} are wrappers
for \code{orsf_vi()}. The way these functions work depends on whether the
\code{object} they are given already has variable importance estimates in it
or not (see examples).
}
\section{Variable importance methods}{


\strong{negation importance}: Each variable is assessed separately by multiplying the variable's coefficients by -1 and then determining how much the model's performance changes. The worse the model's performance after negating coefficients for a given variable, the more important the variable. This technique is promising b/c it does not require permutation and it emphasizes variables with larger coefficients in linear combinations, but it is also relatively new and hasn't been studied as much as permutation importance. See \href{https://arxiv.org/abs/2208.01129}{Jaeger, 2022} for more details on this technique.

\strong{permutation importance}: Each variable is assessed separately by randomly permuting the variable's values and then determining how much the model's performance changes. The worse the model's performance after permuting the values of a given variable, the more important the variable. This technique is flexible, intuitive, and frequently used. It also has several \href{https://christophm.github.io/interpretable-ml-book/feature-importance.html#disadvantages-9}{known limitations}

\strong{analysis of variance (ANOVA) importance}: A p-value is computed for each coefficient in each linear combination of variables in each decision tree. Importance for an individual predictor variable is the proportion of times a p-value for its coefficient is < 0.01. This technique is very efficient computationally, but may not be as effective as permutation or negation in terms of selecting signal over noise variables. See \href{https://link.springer.com/chapter/10.1007/978-3-642-23783-6_29}{Menze, 2011} for more details on this technique.
}

\section{Examples}{
\subsection{ANOVA importance}{

The default variable importance technique, ANOVA, is calculated while
you fit an ORSF ensemble.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fit <- orsf(pbc_orsf, Surv(time, status) ~ . - id)

fit
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## ---------- Oblique random survival forest
## 
##      Linear combinations: Accelerated
##           N observations: 276
##                 N events: 111
##                  N trees: 500
##       N predictors total: 17
##    N predictors per node: 5
##  Average leaves per tree: 25
## Min observations in leaf: 5
##       Min events in leaf: 1
##           OOB stat value: 0.84
##            OOB stat type: Harrell's C-statistic
##      Variable importance: anova
## 
## -----------------------------------------
}\if{html}{\out{</div>}}

ANOVA is the default because it is fast, but it may not be as decisive
as the permutation and negation techniques for variable selection.
}

\subsection{Raw VI values}{

the ‘raw’ variable importance values can be accessed from the fit object

\if{html}{\out{<div class="sourceCode r">}}\preformatted{attr(fit, 'importance_values')
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{##     edema_1   ascites_1        bili      copper         age     albumin 
##  0.41468531  0.34547820  0.27357335  0.19702602  0.17831563  0.17231851 
##   edema_0.5     protime        chol       stage       sex_f   spiders_1 
##  0.16100917  0.15265823  0.14529486  0.13818084  0.13186813  0.12881052 
##         ast    hepato_1    alk.phos        trig    platelet trt_placebo 
##  0.12509496  0.11370348  0.10024752  0.09878683  0.08006941  0.06398488
}\if{html}{\out{</div>}}

these are ‘raw’ because values for factors have not been aggregated into
a single value. Currently there is one value for k-1 levels of a k level
factor. For example, you can see edema_1 and edema_0.5 in the importance
values above because edema is a factor variable with levels of 0, 0.5,
and 1.
}

\subsection{Collapse VI across factor levels}{

To get aggregated values across all levels of each factor,
\itemize{
\item access the \code{importance} element from the \code{orsf} fit:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fit$importance
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{##    ascites       bili      edema     copper        age    albumin    protime 
## 0.34547820 0.27357335 0.26368761 0.19702602 0.17831563 0.17231851 0.15265823 
##       chol      stage        sex    spiders        ast     hepato   alk.phos 
## 0.14529486 0.13818084 0.13186813 0.12881052 0.12509496 0.11370348 0.10024752 
##       trig   platelet        trt 
## 0.09878683 0.08006941 0.06398488
}\if{html}{\out{</div>}}
\item use \code{orsf_vi()} with group_factors set to \code{TRUE} (the default)

\if{html}{\out{<div class="sourceCode r">}}\preformatted{orsf_vi(fit)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{##    ascites       bili      edema     copper        age    albumin    protime 
## 0.34547820 0.27357335 0.26368761 0.19702602 0.17831563 0.17231851 0.15265823 
##       chol      stage        sex    spiders        ast     hepato   alk.phos 
## 0.14529486 0.13818084 0.13186813 0.12881052 0.12509496 0.11370348 0.10024752 
##       trig   platelet        trt 
## 0.09878683 0.08006941 0.06398488
}\if{html}{\out{</div>}}
}

Note that you can make the default returned importance values ungrouped
by setting \code{group_factors} to \code{FALSE} in the \code{orsf_vi} functions or the
\code{orsf} function.
}

\subsection{Add VI to an ORSF}{

You can fit an ORSF without VI, then add VI later

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fit_no_vi <- orsf(pbc_orsf,
                  Surv(time, status) ~ . - id,
                  importance = 'none')

# Note: you can't call orsf_vi_anova() on fit_no_vi because anova
# VI can only be computed while the forest is being grown.

orsf_vi_negate(fit_no_vi)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{##          bili        copper           age       protime       albumin 
##  0.0717336945  0.0288601792  0.0253698687  0.0110960617  0.0100020838 
##          chol       ascites       spiders           ast         stage 
##  0.0075015628  0.0060950198  0.0045321942  0.0044280058  0.0025526151 
##         edema           sex        hepato      platelet      alk.phos 
##  0.0024856369  0.0015628256  0.0004688477  0.0003646593 -0.0007293186 
##          trig           trt 
## -0.0020316733 -0.0061471140
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{orsf_vi_permute(fit_no_vi)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{##           age          bili        copper       albumin          chol 
##  1.109606e-02  1.083559e-02  7.032715e-03  5.157324e-03  4.636383e-03 
##       protime       ascites       spiders           ast      platelet 
##  4.011252e-03  3.854970e-03  2.396333e-03  1.146072e-03  5.209419e-04 
##      alk.phos         edema           sex        hepato          trig 
##  2.083767e-04  1.959734e-04  5.209419e-05 -4.688477e-04 -1.719108e-03 
##           trt 
## -3.698687e-03
}\if{html}{\out{</div>}}
}

\subsection{ORSF and VI all at once}{

fit an ORSF and compute vi at the same time

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fit_permute_vi <- orsf(pbc_orsf,
                        Surv(time, status) ~ . - id,
                        importance = 'permute')

# get the vi instantly (i.e., it doesn't need to be computed again)
orsf_vi_permute(fit_permute_vi)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{##          bili           age        copper         stage       ascites 
##  0.0114086268  0.0094811419  0.0055219837  0.0043238175  0.0032298395 
##       albumin        hepato       protime           ast         edema 
##  0.0031256512  0.0030214628  0.0029172744  0.0021358616  0.0019051588 
##       spiders          chol      alk.phos      platelet           trt 
##  0.0017712023  0.0013023547  0.0008335070 -0.0009376954 -0.0016149198 
##           sex          trig 
## -0.0020837675 -0.0022921442
}\if{html}{\out{</div>}}

You can still get negation VI from this fit, but it needs to be computed

\if{html}{\out{<div class="sourceCode r">}}\preformatted{orsf_vi_negate(fit_permute_vi)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{##          bili        copper           age       protime       albumin 
##  0.0773598666  0.0272452594  0.0258387164  0.0115649094  0.0084392582 
##           sex          chol           ast       ascites         stage 
##  0.0081787872  0.0074494686  0.0060429256  0.0058866431  0.0043238175 
##        hepato         edema       spiders      platelet          trig 
##  0.0040112523  0.0027684339  0.0026047093  0.0005730360  0.0002083767 
##           trt      alk.phos 
## -0.0003125651 -0.0016149198
}\if{html}{\out{</div>}}
}
}

\references{
Harrell FE, Califf RM, Pryor DB, Lee KL, Rosati RA. Evaluating the Yield of Medical Tests. \emph{JAMA} 1982; 247(18):2543-2546. DOI: 10.1001/jama.1982.03320430047030

Breiman L. Random forests. \emph{Machine learning} 2001 Oct; 45(1):5-32. DOI: 10.1023/A:1010933404324

Menze BH, Kelm BM, Splitthoff DN, Koethe U, Hamprecht FA. On oblique random forests. \emph{Joint European Conference on Machine Learning and Knowledge Discovery in Databases} 2011 Sep 4; pp. 453-469. DOI: 10.1007/978-3-642-23783-6_29

Jaeger BC, Welden S, Lenoir K, Speiser JL, Segar MW, Pandey A, Pajewski NM. Accelerated and interpretable oblique random survival forests. \emph{arXiv e-prints} 2022 Aug; arXiv-2208. URL: https://arxiv.org/abs/2208.01129
}
