% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/xGR2nGenes.r
\name{xGR2nGenes}
\alias{xGR2nGenes}
\title{Function to define nearby genes given a list of genomic regions}
\usage{
xGR2nGenes(data, format = c("chr:start-end", "data.frame", "bed",
"GRanges"),
build.conversion = c(NA, "hg38.to.hg19", "hg18.to.hg19"),
distance.max = 50000, decay.kernel = c("rapid", "slow", "linear",
"constant"), decay.exponent = 2, GR.Gene = c("UCSC_knownGene",
"UCSC_knownCanonical"), scoring = F, scoring.scheme = c("max", "sum",
"sequential"), scoring.rescale = F, verbose = T,
RData.location = "http://galahad.well.ox.ac.uk/bigdata")
}
\arguments{
\item{data}{input genomic regions (GR). If formatted as "chr:start-end"
(see the next parameter 'format' below), GR should be provided as a
vector in the format of 'chrN:start-end', where N is either 1-22 or X,
start (or end) is genomic positional number; for example, 'chr1:13-20'.
If formatted as a 'data.frame', the first three columns correspond to
the chromosome (1st column), the starting chromosome position (2nd
column), and the ending chromosome position (3rd column). If the format
is indicated as 'bed' (browser extensible data), the same as
'data.frame' format but the position is 0-based offset from chromomose
position. If the genomic regions provided are not ranged but only the
single position, the ending chromosome position (3rd column) is allowed
not to be provided. The data could also be an object of 'GRanges' (in
this case, formatted as 'GRanges')}

\item{format}{the format of the input data. It can be one of
"data.frame", "chr:start-end", "bed" or "GRanges"}

\item{build.conversion}{the conversion from one genome build to
another. The conversions supported are "hg38.to.hg19" and
"hg18.to.hg19". By default it is NA (no need to do so)}

\item{distance.max}{the maximum distance between genes and GR. Only
those genes no far way from this distance will be considered as seed
genes. This parameter will influence the distance-component weights
calculated for nearby GR per gene}

\item{decay.kernel}{a character specifying a decay kernel function. It
can be one of 'slow' for slow decay, 'linear' for linear decay, and
'rapid' for rapid decay. If no distance weight is used, please select
'constant'}

\item{decay.exponent}{a numeric specifying a decay exponent. By
default, it sets to 2}

\item{GR.Gene}{the genomic regions of genes. By default, it is
'UCSC_knownGene', that is, UCSC known genes (together with genomic
locations) based on human genome assembly hg19. It can be
'UCSC_knownCanonical', that is, UCSC known canonical genes (together
with genomic locations) based on human genome assembly hg19.
Alternatively, the user can specify the customised input. To do so,
first save your RData file (containing an GR object) into your local
computer, and make sure the GR object content names refer to Gene
Symbols. Then, tell "GR.Gene" with your RData file name (with or
without extension), plus specify your file RData path in
"RData.location". Note: you can also load your customised GR object
directly}

\item{scoring}{logical to indicate whether gene-level scoring will be
further calculated. By default, it sets to false}

\item{scoring.scheme}{the method used to calculate seed gene scores
under a set of GR. It can be one of "sum" for adding up, "max" for the
maximum, and "sequential" for the sequential weighting. The sequential
weighting is done via: \eqn{\sum_{i=1}{\frac{R_{i}}{i}}}, where
\eqn{R_{i}} is the \eqn{i^{th}} rank (in a descreasing order)}

\item{scoring.rescale}{logical to indicate whether gene scores will be
further rescaled into the [0,1] range. By default, it sets to false}

\item{verbose}{logical to indicate whether the messages will be
displayed in the screen. By default, it sets to true for display}

\item{RData.location}{the characters to tell the location of built-in
RData files. See \code{\link{xRDataLoader}} for details}
}
\value{
If scoring sets to false, a data frame with following columns:
\itemize{
\item{\code{Gene}: nearby genes}
\item{\code{GR}: genomic regions}
\item{\code{Dist}: the genomic distance between the gene and the GR}
\item{\code{Weight}: the distance weight based on the genomic distance}
}
If scoring sets to true, a data frame with following columns:
\itemize{
\item{\code{Gene}: nearby genes}
\item{\code{Score}: gene score taking into account the distance weight
based on the genomic distance}
}
}
\description{
\code{xGR2nGenes} is supposed to define nearby genes given a list of
genomic regions (GR) within certain distance window. The distance
weight is calcualted as a decaying function of the gene-to-GR distance.
}
\note{
For details on the decay kernels, please refer to
\code{\link{xVisKernels}}
}
\examples{
\dontrun{
# Load the XGR package and specify the location of built-in data
library(XGR)
RData.location <- "http://galahad.well.ox.ac.uk/bigdata_dev"

# a) provide the genomic regions
## load ImmunoBase
ImmunoBase <- xRDataLoader(RData.customised='ImmunoBase',
RData.location=RData.location)
## get lead SNPs reported in AS GWAS and their significance info (p-values)
gr <- ImmunoBase$AS$variant
df <- as.data.frame(gr, row.names=NULL)
chr <- df$seqnames
start <- df$start
end <- df$end
data <- paste(chr,':',start,'-',end, sep='')

# b) define nearby genes taking into acount distance weight
# without gene scoring
df_nGenes <- xGR2nGenes(data=data, format="chr:start-end",
distance.max=10000, decay.kernel="slow", decay.exponent=2,
RData.location=RData.location)
# with their scores
df_nGenes <- xGR2nGenes(data=data, format="chr:start-end",
distance.max=10000, decay.kernel="slow", decay.exponent=2, scoring=T,
scoring.scheme="max", RData.location=RData.location)

# c) define nearby genes without taking into acount distance weight
# without gene scoring
df_nGenes <- xGR2nGenes(data=data, format="chr:start-end",
distance.max=10000, decay.kernel="constant",
RData.location=RData.location)
# with their scores
df_nGenes <- xGR2nGenes(data=data, format="chr:start-end",
distance.max=10000, decay.kernel="constant", scoring=T,
scoring.scheme="max", RData.location=RData.location)
}
}
\seealso{
\code{\link{xRDataLoader}}, \code{\link{xGR}}
}

