% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/xPrioritiserSNPs.r
\name{xPrioritiserSNPs}
\alias{xPrioritiserSNPs}
\title{Function to priorise genes given a list of seed SNPs together with the significance level (e.g. GWAS reported p-values)}
\usage{
xPrioritiserSNPs(data, include.LD = NA, LD.r2 = 0.8, include.eQTL = NA,
network = c("STRING_highest", "STRING_high", "STRING_medium",
"PCommonsUN_high", "PCommonsUN_medium", "PCommonsDN_high",
"PCommonsDN_medium", "PCommonsDN_Reactome", "PCommonsDN_KEGG",
"PCommonsDN_HumanCyc", "PCommonsDN_PID", "PCommonsDN_PANTHER",
"PCommonsDN_ReconX", "PCommonsDN_TRANSFAC", "PCommonsDN_PhosphoSite",
"PCommonsDN_CTD"), network.customised = NULL,
significance.threshold = 5e-05, distance.max = 2e+05,
normalise = c("laplacian", "row", "column", "none"), restart = 0.75,
normalise.affinity.matrix = c("none", "quantile"), parallel = TRUE,
multicores = NULL, verbose = T,
RData.location =
"https://github.com/hfang-bristol/RDataCentre/blob/master/XGR/1.0.0")
}
\arguments{
\item{data}{a named input vector containing the sinificance level for
nodes (dbSNP). For this named vector, the element names are dbSNP, the
element values for the significance level (measured as p-value or fdr).
Alternatively, it can be a matrix or data frame with two columns: 1st
column for dbSNP, 2nd column for the significance level}

\item{include.LD}{additional SNPs in LD with Lead SNPs are also
included. By default, it is 'NA' to disable this option. Otherwise, LD
SNPs will be included based on one or more of 26 populations and 5
super populations from 1000 Genomics Project data (phase 3). The
population can be one of 5 super populations ("AFR", "AMR", "EAS",
"EUR", "SAS"), or one of 26 populations ("ACB", "ASW", "BEB", "CDX",
"CEU", "CHB", "CHS", "CLM", "ESN", "FIN", "GBR", "GIH", "GWD", "IBS",
"ITU", "JPT", "KHV", "LWK", "MSL", "MXL", "PEL", "PJL", "PUR", "STU",
"TSI", "YRI"). Explanations for population code can be found at
\url{http://www.1000genomes.org/faq/which-populations-are-part-your-study}}

\item{LD.r2}{the LD r2 value. By default, it is 0.8, meaning that SNPs
in LD (r2>=0.8) with input SNPs will be considered as LD SNPs. It can
be any value from 0.8 to 1}

\item{include.eQTL}{genes modulated by eQTL (also Lead SNPs or in LD
with Lead SNPs) are also included. By default, it is 'NA' to disable
this option. Otherwise, those genes modulated either by cis-eQTLs
('JKscience_TS2B') or trans-eQTLs ('JKscience_TS3A') will be inlcuded
according to this work by Fairfax et al. Science 2014,
343(6175):1246949}

\item{network}{the built-in network. Currently two sources of network
information are supported: the STRING database (version 10) and the
Pathways Commons database (version 7). STRING is a meta-integration of
undirect interactions from the functional aspect, while Pathways
Commons mainly contains both undirect and direct interactions from the
physical/pathway aspect. Both have scores to control the confidence of
interactions. Therefore, the user can choose the different quality of
the interactions. In STRING, "STRING_highest" indicates interactions
with highest confidence (confidence scores>=900), "STRING_high" for
interactions with high confidence (confidence scores>=700), and
"STRING_medium" for interactions with medium confidence (confidence
scores>=400). For undirect/physical interactions from Pathways Commons,
"PCommonsUN_high" indicates undirect interactions with high confidence
(supported with the PubMed references plus at least 2 different
sources), "PCommonsUN_medium" for undirect interactions with medium
confidence (supported with the PubMed references). For direct
(pathway-merged) interactions from Pathways Commons, "PCommonsDN_high"
indicates direct interactions with high confidence (supported with the
PubMed references plus at least 2 different sources), and
"PCommonsUN_medium" for direct interactions with medium confidence
(supported with the PubMed references). In addtion to pooled version of
pathways from all data sources, the user can also choose the
pathway-merged network from individual sources, that is,
"PCommonsDN_Reactome" for those from Reactome, "PCommonsDN_KEGG" for
those from KEGG, "PCommonsDN_HumanCyc" for those from HumanCyc,
"PCommonsDN_PID" for those froom PID, "PCommonsDN_PANTHER" for those
from PANTHER, "PCommonsDN_ReconX" for those from ReconX,
"PCommonsDN_TRANSFAC" for those from TRANSFAC, "PCommonsDN_PhosphoSite"
for those from PhosphoSite, and "PCommonsDN_CTD" for those from CTD}

\item{network.customised}{an object of class "igraph". By default, it
is NULL. It is designed to allow the user analysing their customised
network data that are not listed in the above argument 'network'. This
customisation (if provided) has the high priority over built-in
network}

\item{significance.threshold}{the given significance threshold. By
default, it is set to NULL, meaning there is no constraint on the
significance level when transforming the significance level of SNPs
into the significance-component weights. If given, those SNPs below
this are considered significant and thus weighted positively. Instead,
those above this are considered insigificant and thus receive no
weight}

\item{distance.max}{the maximum distance between genes and SNPs. Only
those genes no far way from this distance will be considered as seed
genes. This parameter will influence the distance-component weights
calculated for nearby SNPs per gene}

\item{normalise}{the way to normalise the adjacency matrix of the input
graph. It can be 'laplacian' for laplacian normalisation, 'row' for
row-wise normalisation, 'column' for column-wise normalisation, or
'none'}

\item{restart}{the restart probability used for Random Walk with
Restart (RWR). The restart probability takes the value from 0 to 1,
controlling the range from the starting nodes/seeds that the walker
will explore. The higher the value, the more likely the walker is to
visit the nodes centered on the starting nodes. At the extreme when the
restart probability is zero, the walker moves freely to the neighbors
at each step without restarting from seeds, i.e., following a random
walk (RW)}

\item{normalise.affinity.matrix}{the way to normalise the output
affinity matrix. It can be 'none' for no normalisation, 'quantile' for
quantile normalisation to ensure that columns (if multiple) of the
output affinity matrix have the same quantiles}

\item{parallel}{logical to indicate whether parallel computation with
multicores is used. By default, it sets to true, but not necessarily
does so. Partly because parallel backends available will be
system-specific (now only Linux or Mac OS). Also, it will depend on
whether these two packages "foreach" and "doMC" have been installed. It
can be installed via:
\code{source("http://bioconductor.org/biocLite.R");
biocLite(c("foreach","doMC"))}. If not yet installed, this option will
be disabled}

\item{multicores}{an integer to specify how many cores will be
registered as the multicore parallel backend to the 'foreach' package.
If NULL, it will use a half of cores available in a user's computer.
This option only works when parallel computation is enabled}

\item{verbose}{logical to indicate whether the messages will be
displayed in the screen. By default, it sets to true for display}

\item{RData.location}{the characters to tell the location of built-in
RData files. See \code{\link{xRDataLoader}} for details}
}
\value{
an object of class "pNode", a list with following components:
\itemize{
\item{\code{priority}: a matrix of nNode X 4 containing node priority
information, where nNode is the number of nodes in the input graph, and
the 4 columns are "name" (node names), "seed" (1 for seeds, 0 for
non-seeds), "weight" (weight values for seeds), "priority" (the
priority scores that are rescaled to the range [0,1]), "rank" (ranks of
the priority scores)}
\item{\code{g}: an input "igraph" object}
\item{\code{SNP}: a data frame of nSNP X 3 containing input SNPs and/or
LD SNPs info, where nSNP is the number of input SNPs and/or LD SNPs,
and the 3 columns are "SNP" (dbSNP), "Pval" (the SNP p-value), "Score"
(the SNP score)}
\item{\code{call}: the call that produced this result}
}
}
\description{
\code{xPrioritiserSNPs} is supposed to priorise genes given a list of
seed SNPs together with the significance level. To priorise genes, it
first defines seed genes and their weights that take into account the
distance to and the significance of seed SNPs. With seed genes and
weights, it then uses Random Walk with Restart (RWR) to calculate the
affinity score of all nodes in the input graph to the seed genes. The
priority score is the affinity score. Parallel computing is also
supported for Linux or Mac operating systems. It returns an object of
class "pNode".
}
\note{
The search procedure is heuristic to find the subgraph with the maximum
score:
\itemize{
\item{i) transform the significance level of SNPs into the
significance-component weights (noded as 'wS'). If the intolerable
significance threshold is given, those SNPs below this are considered
significant and thus weighted positively. Instead, those above this are
considered insigificant and thus receive no weight.}
\item{ii) find genes located away from seed SNPs within the certain
range (by default 500kb) and, for nearby SNPs per gene, calculate the
distance-component weights (noded as 'wD').}
\item{iii) define seed genes as those found in ii) and their weights as
the maximum of 'wS * wD'.}
\item{iv) \code{\link{xPrioritiserGenes}} used to prioritise genes
using an input graph and a list of seed genes weighted from iii). The
priority score is the affinity score estimated by Random Walk with
Restart (RWR), measured as the affinity of all nodes in the graph to
the seeds.}
}
}
\examples{
\dontrun{
# Load the library
library(XGR)
library(igraph)
library(dnet)
library(ggbio)

# a) provide the SNPs with the significance info
## get lead SNPs reported in AS GWAS and their significance info (p-values)
AS <- read.delim(file.path(path.package("XGR"),"AS.txt"),
stringsAsFactors=FALSE)

# b) perform priority analysis
pNode <- xPrioritiserSNPs(data=AS,
network="PCommonsUN_medium",restart=0.7)

# c) save to the file called 'SNPs_priority.txt'
write.table(pNode$priority, file="SNPs_priority.txt", sep="\\t",
row.names=FALSE)

# d) manhattan plot
mp <- xPrioritiserManhattan(pNode, highlight.top=10)
#pdf(file="Gene_manhattan.pdf", height=6, width=12, compress=TRUE)
print(mp)
#dev.off()
}
}
\seealso{
\code{\link{xRDataLoader}}, \code{\link{xPrioritiser}},
\code{\link{xPrioritiserGenes}}, \code{\link{xPrioritiserPathways}}
}

