% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/updist_sites.R
\name{updist_sites}
\alias{updist_sites}
\title{Get upstream distance for sites in a Landscape Network (LSN)}
\usage{
updist_sites(
  sites,
  edges,
  length_col,
  lsn_path,
  save_local = TRUE,
  overwrite = TRUE
)
}
\arguments{
\item{sites}{A named list of one or more \code{sf} objects with
POINT geometry that have been snapped to the LSN using
\code{\link[SSNbler]{sites_to_lsn}}.}

\item{edges}{An \code{sf} object with LINESTING geometry created
using \code{\link{lines_to_lsn}} and
\code{link[SSNbler]{updist_edges}}.}

\item{length_col}{The name of the column in \code{edges} that
contains the length of each edge feature.}

\item{lsn_path}{Local pathname to a directory in
character format specifying where the LSN resides, which is
created using \code{link[SSNbler]{lines_to_lsn}}. Must be
specified if \code{save_local = TRUE}.}

\item{save_local}{Logical indicating whether the updated
\code{sites} should be saved to \code{lsn_path} in GeoPackage
format. File basenames are taken from the names assigned to the
\code{sites} list. Default is \code{TRUE}.}

\item{overwrite}{A logical indicating whether results should be
overwritten if the upDist column already exists in \code{sites}
or sites.gpkg already exists in \code{lsn_path} and
\code{save_local = TRUE}. Default = TRUE.

#' @details \code{updist_sites()} calculates the total hydrologic distance from
each observed or prediction point feature to
the stream outlet (i.e. the most downstream location in the
stream network), when movement is restricted to the stream
network. We refer to this as the upstream distance.

Upstream distances are measured in the map projection units for the
\code{sf} object containing the point features and stored in a
new column named \code{upDist}.

The upstream distances stored in \code{upDist} are used to
calculate the pairwise hydrologic distances used to fit spatial
stream network models in the 'SSN2' package. Do not modify the name
of the column in any way or the values the \code{upDist} column
contains.}
}
\value{
One or more \code{sf} object(s) with all the original
data from \code{sites}, along with a new \code{upDist} column in
each sites \code{sf} object. A named list is returned. If
\code{save_local = TRUE}, a GeoPackage for each \code{sf} object
is saved in \code{lsn_path}. Output file names are assigned based
on the input \code{sites} attribute \code{names}.
}
\description{
Get upstream distance for sites in a Landscape Network (LSN)
}
\examples{
# Get temporary directory, where the example LSN will be stored
# locally.
temp_dir <- tempdir()
# Build the LSN. When working with your own data, lsn_path will be
# a local folder of your choice rather than a temporary directory.
edges<- lines_to_lsn(
   streams = MF_streams,
   lsn_path = temp_dir,
   snap_tolerance = 1,
   check_topology = FALSE,
   overwrite = TRUE,
   verbose = FALSE
)

# Incorporate observed sites, MF_obs, into LSN
obs<- sites_to_lsn(
   sites = MF_obs,
   edges = edges,
   save_local = FALSE,
   snap_tolerance = 100,
   overwrite = TRUE,
   verbose = FALSE
)

# Incorporate prediction dataset, MF_preds, into LSN
preds<- sites_to_lsn(
   sites = MF_preds,
   edges = edges,
   save_local = FALSE,
   snap_tolerance = 1,
   overwrite = TRUE,
   verbose = FALSE
)

# Calculate upstream distance for edges
edges<- updist_edges(
   edges = edges,
   lsn_path = temp_dir,
   calc_length = TRUE,
   length_col = "Length",
   overwrite = TRUE,
   save_local = FALSE,
   verbose = FALSE
)

# Calculate upstream distance for observed sites (obs) and one
# prediction dataset (preds)
site.list<- updist_sites(
   sites = list(obs = obs,
                preds = preds),
   edges = edges,
   length_col= "Length",
   lsn_path = temp_dir,
   save_local = FALSE,
   overwrite = TRUE
)

# Summarize the new column upDist in obs
summary(site.list$obs$upDist)

# Summarize the new column upDist in preds
summary(site.list$preds$upDist)

}
