% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/perturb_matrix.R
\name{perturb_matrix}
\alias{perturb_matrix}
\title{Perturbation analysis of a matrix population model}
\usage{
perturb_matrix(
  matA,
  pert = 1e-06,
  type = "sensitivity",
  demog_stat = "lambda",
  ...
)
}
\arguments{
\item{matA}{A matrix population model (i.e., a square projection matrix).}

\item{pert}{Magnitude of the perturbation. Defaults to \code{1e-6}.}

\item{type}{Whether to return \code{sensitivity} or \code{elasticity} values.}

\item{demog_stat}{The demographic statistic to be used, as in "the
sensitivity/elasticity of \code{demog_stat} to matrix element 
perturbations." Defaults to the per-capita population growth rate at 
equilibrium (\eqn{\lambda}). Also accepts a user-supplied function that 
performs a calculation on a projection matrix and returns a single numeric 
value.}

\item{...}{Additional arguments passed to the function \code{demog_stat}}
}
\value{
A sensitivity or elasticity matrix.
}
\description{
Perturbs elements within a matrix population model and measures the response 
(sensitivity or elasticity) of the per-capita population growth rate at 
equilibrium (\eqn{\lambda}), or, with a user-supplied function, any other 
demographic statistic.
}
\examples{
matA <- rbind(c(0.1,   0, 1.5, 4.6),
              c(0.5, 0.2, 0.1,   0),
              c(  0, 0.3, 0.3, 0.1),
              c(  0,   0, 0.5, 0.6))

perturb_matrix(matA)

# use a larger perturbation than the default
perturb_matrix(matA, pert = 0.01)

# calculate the sensitivity/elasticity of the damping ratio to perturbations
damping <- function(matA) {  # define function for damping ratio
  eig <- eigen(matA)$values
  dm <- rle(Mod(eig))$values
  return(dm[1] / dm[2])
}

perturb_matrix(matA, demog_stat = "damping")

}
\references{
Caswell, H. 2001. Matrix Population Models: Construction,
  Analysis, and Interpretation. Sinauer Associates; 2nd edition. ISBN:
  978-0878930968
}
\seealso{
Other perturbation analysis: 
\code{\link{perturb_stochastic}()},
\code{\link{perturb_trans}()},
\code{\link{perturb_vr}()},
\code{\link{pop_vectors}()}
}
\author{
Rob Salguero-Gomez <rob.salguero@zoo.ox.ac.uk>
}
\concept{perturbation analysis}
