#' Plotting output and parameters of inferential interest for IMIFA and related models
#'
#' @param x An object of class "\code{Results_IMIFA}" generated by \code{\link{get_IMIFA_results}}.
#' @param plot.meth The type of plot to be produced for the \code{param} of interest, where \code{correlation} refers to ACF/PACF plots, \code{means} refers to posterior means, \code{density}, \code{trace} and \code{parallel.coords} are self-explanatory. "\code{all}" in this case, the default, refers to {\code{trace, density, means, correlation}}. \code{parallel.coords} is only available when \code{param} is one of \code{means}, \code{loadings} or \code{uniquenesses} - note that this method applies a small amount of horizontal jitter to avoid overplotting. Special types of plots which don't require a \code{param} are \code{GQ}, for plotting the posterior summaries of the numbers of groups/factors, if available, \code{zlabels} for plotting clustering uncertainties if clustering has taken place (with or without the clustering labels being supplied via the \code{zlabels} argument), and \code{errors} for visualing the difference between the estimated and empirical covariance matrix/matrices.
#' @param param The parameter of interest for any of the following \code{plot.meth} options: {\code{trace, density, means, correlation}}. The \code{param} must have been stored when \code{\link{mcmc_IMIFA}} was initially ran. Includes \code{pis} for methods where clustering takes place, and allows posterior inference on \code{alpha} for the "\code{IMFA}" and "\code{IMIFA}" methods.
#' @param zlabels The true labels can be supplied if they are known. Only relevant when \code{plot.meth = "zlabels"}. If this is not supplied, the function uses the labels that were supplied, if any, to \code{\link{get_IMIFA_results}}.
#' @param load.meth Switch which allows plotting posterior mean loadings as a heatmap (the default), or as something akin to \code{link{plot}} with \code{type = "h"}. ONly relevant if \code{param = "loadings"}. Heatmaps are produced with the aid of \code{\link{mat2cols}} and \code{\link[gclus]{plotcolors}}.
#' @param palette An optional colour palette to be supplied if overwriting the default palette set inside the function by \code{\link[viridis]{viridis}} is desired.
#' @param g Optional argument that allows specification of exactly which cluster the plot of interest is to be produced for. If not supplied, the user will be prompted to cycle through plots for all clusters.
#' @param mat Logical indicating whether a \code{\link{matplot}} is produced (defaults to TRUE). If given as FALSE, \code{ind} is invoked.
#' @param ind Either a single number indicating which variable to plot when \code{param} is one of \code{means} or \code{uniquenesses}, or which cluster to plot if \code{param} is \code{pis}. If \code{scores} are plotted, a vector of length two giving which observation and factor to plot; If \code{loadings} are plotted, a vector of length two giving which variable and factor to plot. Only relevant when \code{mat} is FALSE.
#' @param fac Optional argument that provides an alternative way to specify \code{ind[2]} when \code{mat} is FALSE and \code{param} is one of \code{scores} or \code{loadings}.
#' @param by.fac Optionally allows (mat)plotting of scores and loadings by factor - i.e. observation(s) (scores) or variable(s) (loadings) for a given factor, respectively, controlled by \code{ind} or \code{fac}) when set to TRUE (the default). Otherwise factor(s) are plotted for a given observation or variable, again controlled by \code{ind} or \code{fac}.
#' @param type The manner in which the plot is to be drawn, as per the \code{type} argument to \code{\link{plot}}.
#' @param intervals Logical indicating whether credible intervals around the posterior mean(s) are to be plotted when \code{is.element(plot.meth, c("all", "means"))}. Defaults to TRUE.
#' @param partial Logical indicating whether plots of type "\code{correlation}" use the PACF. The default, FALSE, ensures the ACF is used. Only relevant when \code{plot.meth = "all"}, otherwise both plots are produced when \code{plot.meth = "correlation"}.
#' @param titles Logical indicating whether default plot titles are to be used (TRUE), or suppressed (FALSE).
#' @param transparency A factor in [0, 1] modifying the opacity for overplotted lines. Defaults to 0.75.
#' @param ... Other arguments typically passed to \code{\link{plot}}.
#'
#' @return The desired plot with appropriate output and summary statistics printed to the console screen.
#' @export
#' @import graphics
#' @importFrom grDevices "palette" "adjustcolor"
#' @importFrom Rfast "Order" "med" "colMedians"
#' @importFrom plotrix "plotCI"
#' @importFrom gclus "plotcolors"
#' @importFrom e1071 "classAgreement"
#' @importFrom mclust "classError"
#' @importFrom utils "tail"
#' @importFrom viridis "viridis"
#' @seealso \code{\link{mcmc_IMIFA}}, \code{\link{get_IMIFA_results}}, \code{\link{mat2cols}}, \code{\link[gclus]{plotcolors}}
#' @references Murphy, K., Gormley, I.C. and Viroli, C. (2017) Infinite Mixtures of Infinite Factor Analysers: Nonparametric Model-Based Clustering via Latent Gaussian Models, \code{https://arxiv.org/abs/1701.07010}
#'
#' @examples
#' # See the vignette associated with the package for more graphical examples.
#'
#' # data(olive)
#' # area     <- olive$area
#' # simIMIFA <- mcmc_IMIFA(olive, method="IMIFA")
#' # resIMIFA <- get_IMIFA_results(simIMIFA)
#'
#' # plot(resIMIFA, "GQ")
#' # plot(resIMIFA, "GQ", g=2)
#'
#' # plot(resIMIFA, "zlabels", zlabels=area)
#'
#' # plot(resIMIFA, "errors")
#'
#' # plot(resIMIFA, "all", "means", g=1)
#' # plot(resIMIFA, "all", "means", g=1, ind=2)
#' # plot(resIMIFA, "all", "scores")
#' # plot(resIMIFA, "all", "scores", by.fac=FALSE)
#' # plot(resIMIFA, "all", "loadings", g=1)
#' # plot(resIMIFA, "all", "loadings", g=1, load.meth="raw")
#' # plot(resIMIFA, "parallel.coords", "uniquenesses")
#' # plot(resIMIFA, "all", "pis", intervals=FALSE, partial=TRUE)
#' # plot(resIMIFA, "all", "alpha")
plot.Results_IMIFA  <- function(x = NULL, plot.meth = c("all", "correlation", "density", "errors", "GQ", "means", "parallel.coords", "trace", "zlabels"),
                                param = c("means", "scores", "loadings", "uniquenesses", "pis", "alpha"), zlabels = NULL, load.meth = c("heatmap", "raw"), palette = NULL, g = NULL,
                                mat = TRUE, ind = NULL, fac = NULL, by.fac = TRUE, type = c("h", "n", "p", "l"), intervals = TRUE, partial = FALSE, titles = TRUE, transparency = 0.75, ...) {

  if(missing(x))                      stop("'x' must be supplied")
  if(!exists(deparse(substitute(x)),
             envir=.GlobalEnv))       stop(paste0("Object ", match.call()$x, " not found\n"))
  if(class(x) != "Results_IMIFA")     stop(paste0("Results object of class 'Results_IMIFA' must be supplied"))
  GQ.res  <- x$GQ.results
  G       <- GQ.res$G
  Gseq    <- seq_len(G)
  Qs      <- GQ.res$Q
  Q.max   <- max(Qs)
  defpar  <- suppressWarnings(graphics::par(no.readonly=TRUE))
  defpar$new        <- FALSE
  if(missing(palette))   palette <- viridis::viridis(min(10, max(G, Q.max, 5)))
  if(!all(.are_cols(cols=palette)))   stop("Supplied colour palette contains invalid colours")
  if(length(palette) < 5)             stop("Palette must contain 5 or more colours")
  if(length(transparency) != 1 &&
     any(!is.numeric(transparency),
         (transparency     < 0 ||
          transparency     > 1)))     stop("'transparency' must be a single number in [0, 1]")
  tmp.pal <- palette
  palette <- grDevices::adjustcolor(palette, alpha.f=transparency)
  grDevices::palette(palette)
  grey    <- grDevices::adjustcolor("#999999", alpha.f=0.3)
  defopt  <- options()
  options(warn=1)
  suppressWarnings(graphics::par(cex.axis=0.8, new=FALSE))
  on.exit(suppressWarnings(graphics::par(defpar)))
  on.exit(do.call("clip", as.list(defpar$usr)), add=TRUE)
  on.exit(grDevices::palette("default"), add=TRUE)
  on.exit(suppressWarnings(options(defopt)), add=TRUE)
  n.grp   <- attr(GQ.res, "Groups")
  n.fac   <- attr(GQ.res, "Factors")
  G.supp  <- attr(GQ.res, "Supplied")["G"]
  Q.supp  <- attr(GQ.res, "Supplied")["Q"]
  method  <- attr(x, "Method")
  store   <- attr(x, "Store")
  param   <- match.arg(param)
  n.var   <- attr(x, "Vars")
  n.obs   <- attr(x, "Obs")
  if(missing(plot.meth))              stop("What type of plot would you like to produce?")
  if(is.element(plot.meth,
     c("G", "Q",
       "QG")))  {      plot.meth <- "GQ"
  }
  uni.type     <- unname(attr(x, "Uni.Meth")['Uni.Type'])
  plot.meth    <- match.arg(plot.meth)
  load.meth    <- match.arg(load.meth)
  type.x       <- missing(type)
  type         <- match.arg(type)
  m.sw         <- c(G.sw = FALSE, Z.sw = FALSE, E.sw = FALSE, P.sw = FALSE, C.sw = FALSE, D.sw = FALSE, M.sw = FALSE, T.sw = FALSE)
  v.sw         <- attr(x, "Switch")
  names(v.sw)  <- formals(sys.function(sys.parent()))$param
  ci.sw        <- v.sw
  var.names    <- rownames(x[[1]]$post.load)
  obs.names    <- rownames(x$Scores$post.eta)
  all.ind      <- plot.meth == "all"
  grp.ind      <- !is.element(method, c("FA", "IFA"))
  if(grp.ind)   {
    clust      <- x$Clust
    grp.size   <- clust$post.sizes
    labelmiss  <- !is.null(attr(clust, "Label.Sup")) && !attr(clust, "Label.Sup")
  }
  grp.ind      <- all(G != 1, grp.ind)
  if(all.ind)   {
    if(v.sw[param]) {
      m.sw[-(1:4)]  <- !m.sw[-(1:4)]
      graphics::layout(matrix(c(1, 2, 3, 4), nrow=2, ncol=2, byrow=TRUE))
      graphics::par(cex=0.8, mai=c(0.7, 0.7, 0.5, 0.2), mgp=c(2, 1, 0), oma=c(0, 0, 2, 0))
    }
  } else {
    sw.n  <- paste0(toupper(substring(plot.meth, 1, 1)), ".sw")
    m.sw[sw.n] <- TRUE
  }
  z.miss  <- missing(zlabels)
  if(!z.miss) {
    z.nam <- gsub("[[:space:]]", "", deparse(substitute(zlabels)))
    nam.z <- gsub("\\[.*", "", z.nam)
    nam.x <- gsub(".*\\[(.*)\\].*", "\\1)", z.nam)
    ptrn  <- c("(", ")")
    if(!exists(nam.z,
               envir=.GlobalEnv))     stop(paste0("Object ", match.call()$zlabels, " not found\n"))
    if(any(unlist(vapply(seq_along(ptrn), function(p) grepl(ptrn[p], nam.z, fixed=TRUE), logical(1))),
           !identical(z.nam,   nam.z) && (any(grepl("[[:alpha:]]", gsub('c', '', nam.x))) || grepl(":",
           nam.x, fixed=TRUE))))      stop("Extremely inadvisable to supply 'zlabels' subsetted by any means other than row/column numbers or c() indexing: best to create new object")
    labs  <- as.numeric(as.factor(zlabels))
    if(length(labs) != n.obs)         stop(paste0("'zlabels' must be a factor of length N=",  n.obs))
  }
  if(m.sw["P.sw"]) {
    if(!is.element(param, c("means",
       "loadings", "uniquenesses")))  stop("Can only plot parallel coordinates for means, loadings or uniquenesses")
  }
  if(!grp.ind)  {
    if(m.sw["Z.sw"])                  stop("Can't use 'Z' for 'plot.meth' as no clustering has taken place")
    if(param == "pis")                stop("Can't plot mixing proportions as no clustering has taken place")
  }
  if(all(m.sw["E.sw"],
         !attr(x, "Errors")))         stop("Can't plot error metrics as they were not calculated due to storage switches")
  if(all(!m.sw["G.sw"], !m.sw["Z.sw"], !m.sw["E.sw"],
     missing(param)))                  stop("What variable would you like to plot?")
  if(all(any(m.sw["M.sw"], all.ind),
     is.element(param, c("means", "uniquenesses")),
     !v.sw[param],
     is.element(method, c("FA", "IFA")))) {
    if(all.ind)                       warning(paste0("Can only plot posterior mean, as ", param, switch(param, alpha="wasn't", "weren't"), " stored"), call.=FALSE)
    v.sw[param]    <- !v.sw[param]
    all.ind        <- FALSE
    m.sw["M.sw"]   <- TRUE
  }
  if(all(!v.sw[param], !m.sw["G.sw"],
     !m.sw["Z.sw"],   !m.sw["E.sw"])) stop(paste0("Nothing to plot: ", param, ifelse(param == "alpha", ifelse(is.element(method, c("FA", "IFA")), paste0(" not used for the ", method, " method"), paste0(" was fixed at ", attr(x, "Alpha"))), " weren't stored")))
  if(any(!is.logical(intervals),
         length(intervals) != 1))     stop("'intervals' must be TRUE or FALSE")
  if(any(!is.logical(mat),
         length(mat)       != 1))     stop("'mat' must be TRUE or FALSE")
  if(any(!is.logical(partial),
         length(partial)   != 1))     stop("'partial' must be TRUE or FALSE")
  if(any(!is.logical(titles),
         length(titles)    != 1))     stop("'titles' must be TRUE or FALSE")
  if(any(!is.logical(by.fac),
         length(by.fac)    != 1))     stop("'by.fac' must be TRUE or FALSE")
  indx    <- missing(ind)
  facx    <- missing(fac)
  gx      <- missing(g)
  if(!indx) {
    ind   <- as.integer(ind)
    xind  <- ind
  }
  if(!facx) {
    fac   <- as.integer(fac)
    flen  <- length(fac)
    if(flen  == 1 && gx)     fac <- rep(fac, G)
    flen  <- length(fac)
    if(flen  != G && all(gx,
       param == "loadings"))          stop(paste0("'fac' must be supplied for each of the ", G, " groups"))
  }
  g.score <- all(grp.ind, !all.ind, param == "scores")
  if(!gx)                      g <- as.integer(g)
  if(!gx  && any(length(g) != 1,
                 !is.numeric(g)))     stop("If 'g' is supplied it must be of length 1")
  if(any(all(is.element(method, c("IMIFA", "OMIFA")), m.sw["G.sw"]), m.sw["Z.sw"])) {
    Gs    <- if(gx) seq_len(2L) else ifelse(g <= 2, g,
                                      stop("Invalid 'g' value"))
  } else if(any(all(is.element(param, c("scores", "pis", "alpha")), any(all.ind, param != "scores", !m.sw["M.sw"])),
            m.sw["G.sw"], all(m.sw["P.sw"], param != "loadings"), m.sw["E.sw"])) {
    Gs    <- 1L
  } else if(!gx) {
    if(!is.element(method, c("FA", "IFA"))) {
      if(!is.element(g, Gseq))        stop("This g value was not used during simulation")
      Gs  <- g
    } else if(g > 1)     {            message(paste0("Forced g=1 for the ", method, " method"))
      Gs  <- 1L
    }
  } else if(!interactive())  {        stop("g must be supplied for non-interactive sessions")
  } else {
    Gs    <- Gseq
  }

  for(g in Gs) {
    Q     <- Qs[g]
    ng    <- ifelse(grp.ind, grp.size[g], n.obs)
    g.ind <- which(Gs == g)
    msgx  <- all(interactive(), g != max(Gs))
    result     <- x[[g]]
    .ent_exit  <- function() {
      ent      <- readline("Hit <Return> to see next plot or type 'EXIT'/hit <Esc> to exit: ")
      options(show.error.messages=FALSE)
      on.exit(suppressWarnings(options(defopt)), add=TRUE)
      if(ent  %in% c("exit", "EXIT")) stop()
    }
    if(any(all(Qs == 0, param == "scores"),
           all(Q  == 0, param == "loadings"),
           all(ng == 0, param == "scores", m.sw["M.sw"]))) {
                                      warning(paste0("Can't plot ", param, paste0(ifelse(any(all(param == "scores", ng == 0), all(param == "loadings", grp.ind)), paste0(" for group ", g), "")), " as they contain no ", ifelse(all(param == "scores", ng == 0), "rows/observations", "columns/factors")), call.=FALSE)
      if(g == max(Gs)) {
        break
      } else {
        if(isTRUE(msgx)) .ent_exit()
        next
      }
    }
    if(any(param  == "alpha",
           all(is.element(param, c("means", "uniquenesses")), !indx),
           all(is.element(param, c("scores", "loadings")),
               Q  == 1))) { matx <- FALSE
    } else   {
      matx     <- mat
    }
    if(!matx) {
      iter     <- switch(param, scores=seq_along(attr(x$Score, "Eta.store")), pis=seq_along(store), seq_len(attr(result, "Store")))
    }
    if(is.element(param, c("scores", "loadings"))) {
      if(indx)               ind <- c(1L, 1L)
      if(!facx)           ind[2] <- fac[g]
      if(all(mat,
         length(ind) == 1))  ind <- rep(ind, 2)
      if(length(ind) != 2)            stop(paste0("Length of plotting indices must be 2 for the ", param, "parameter when 'mat' is FALSE"))
      if(param == "scores") {
        if(ind[1] >  n.obs)           stop(paste0("First index can't be greater than the number of observations: ",  n.obs))
        if(ind[2] >  Q.max) {         warning(paste0("Second index can't be greater than ", Q.max, ", the total number of factors", if(grp.ind) paste0(" across groups"), ".\n Try specifying a vector of fac values with maximum entries ", paste0(Qs, collapse=", "), "."), call.=FALSE)
        if(isTRUE(msgx)) .ent_exit()
        next
        }
      } else {
        if(ind[1] > n.var)            stop(paste0("First index can't be greater than the number of variables: ",  n.var))
        if(ind[2] > Q) {              warning(paste0("Second index can't be greater than ", Q, ", the number of factors", if(grp.ind) paste0(" in group ", g), ".\n Try specifying a vector of fac values with maximum entries ", paste0(Qs, collapse=", "), "."), call.=FALSE)
        if(isTRUE(msgx)) .ent_exit()
        next
        }
      }
    } else   {
      if(any(param  == "alpha",
             indx))       ind    <- 1L
      if(length(ind) >  1)            stop("Length of plotting indices can't be greater than 1")
      if(param == "pis")    {
        if(ind       >  G)            stop(paste0("Index can't be greater than the number of groups: ", G))
      } else {
        if(ind       > n.var)         stop(paste0("Index can't be greater than the number of variables: ", n.var))
      }
    }

    if(m.sw["T.sw"]) {
      if(param == "means")  {
        plot.x <- result$means
        if(matx) {
          graphics::matplot(t(plot.x), type="l", ylab="", xlab="Iteration", lty=1, ylim=if(is.element(method, c("FA", "IFA"))) c(-1, 1))
          if(titles) graphics::title(main=list(paste0("Trace", ifelse(all.ind, "", paste0(":\nMeans", ifelse(grp.ind, paste0(" - Group ", g), ""))))))
        } else {
          graphics::plot(x=iter, y=plot.x[ind,], type="l", ylab="", xlab="Iteration", ylim=if(is.element(method, c("FA", "IFA"))) c(-1, 1))
          if(titles) graphics::title(main=list(paste0("Trace", ifelse(all.ind, ":\n", paste0(":\nMean - ", ifelse(grp.ind, paste0("Group ", g, " - "), ""))), var.names[ind], " Variable")))
        }
      }
      if(param == "scores") {
        x.plot <- x$Scores$eta
        if(by.fac) {
          plot.x  <- x.plot[,ind[2],]
        } else {
          plot.x  <- if(Q.max > 1) x.plot[ind[1],,] else t(x.plot[ind[1],,])
        }
        if(matx) {
          graphics::matplot(t(plot.x), type="l", ylab="", xlab="Iteration", lty=1)
          if(by.fac) {
            if(titles) graphics::title(main=list(paste0("Trace", ifelse(all.ind, ":\n", ":\nScores - "), "Factor ", ind[2])))
          } else {
            if(titles) graphics::title(main=list(paste0("Trace", ifelse(all.ind, ":\n", ":\nScores - "), "Observation ", obs.names[ind[1]])))
          }
        } else {
          graphics::plot(x=iter, y=x.plot[ind[1],ind[2],], type="l", ylab="", xlab="Iteration")
          if(titles) graphics::title(main=list(paste0("Trace", ifelse(all.ind, ":\n", ":\nScores - "), "Observation ", obs.names[ind[1]], ", Factor ", ind[2])))
        }
      }
      if(param == "loadings") {
        x.plot <- result$loadings
        plot.x <- if(by.fac) x.plot[,ind[2],] else x.plot[ind[1],,]
        if(matx) {
          graphics::matplot(t(plot.x), type="l", ylab="", xlab="Iteration", lty=1)
          if(by.fac) {
            if(titles) graphics::title(main=list(paste0("Trace", ifelse(all.ind, ":\n", paste0(":\nLoadings - ", ifelse(grp.ind, paste0("Group ", g, " - "), ""))), "Factor ", ind[2])))
          } else {
            if(titles) graphics::title(main=list(paste0("Trace", ifelse(all.ind, ":\n", paste0(":\nLoadings - ", ifelse(grp.ind, paste0("Group ", g, " - "), ""))), var.names[ind[1]], " Variable")))
          }
        } else   {
          graphics::plot(x=iter, y=x.plot[ind[1],ind[2],], type="l", ylab="", xlab="Iteration")
          if(titles) graphics::title(main=list(paste0("Trace", ifelse(all.ind, ":\n", paste0(":\nLoadings - ", ifelse(grp.ind, paste0("Group ", g, " - "), ""))), var.names[ind[1]], " Variable, Factor ", ind[2])))
        }
      }
      if(param == "uniquenesses") {
        plot.x <- result$psi
        if(matx) {
          graphics::matplot(t(plot.x), type="l", ylab="", xlab="Iteration", lty=1)
          if(titles) graphics::title(main=list(paste0("Trace", ifelse(all.ind, "", paste0(":\nUniquenesses", ifelse(grp.ind, paste0(" - Group ", g), ""))))))
        } else   {
          graphics::plot(x=iter, y=plot.x[ind,], ylab="", type="l", xlab="Iteration")
          if(titles) graphics::title(main=list(paste0("Trace", ifelse(all.ind, ":\n", paste0(":\nUniqueness - ", ifelse(grp.ind, paste0("Group ", g, " - "), ""))), var.names[ind], " Variable")))
        }
      }
      if(param == "pis") {
        plot.x <- clust$pi.prop
        if(matx) {
          graphics::matplot(t(plot.x), type="l", ylab="", xlab="Iteration", lty=1)
          if(titles) graphics::title(main=list(paste0("Trace", ifelse(all.ind, "", paste0(":\nMixing Proportions")))))
        } else   {
          graphics::plot(x=iter, y=plot.x[ind,], ylab="", type="l", xlab="Iteration")
          if(titles) graphics::title(main=list(paste0("Trace", ifelse(all.ind, "", paste0(":\nMixing Proportion - Group ", ind)))))
        }
      }
      if(param == "alpha") {
        plot.x <- clust$DP.alpha
        graphics::plot(plot.x$alpha, ylab="", type="l", xlab="Iteration", main="")
        if(titles) graphics::title(main=list(paste0("Trace", ifelse(all.ind, "", paste0(":\nAlpha")))))
        if(all(intervals, ci.sw[param])) {
          ci.x <- plot.x$ci.alpha
          graphics::abline(h=plot.x$post.alpha,  col=2,    lty=2)
          graphics::abline(h=plot.x$ci.alpha[1], col=grey, lty=2)
          graphics::abline(h=plot.x$ci.alpha[2], col=grey, lty=2)
        }
      }
      if(!indx) {         ind[1] <- xind[1]
        if(all(facx, is.element(param, c("scores",
           "loadings")))) ind[2] <- xind[2]
      }
      if(all.ind)          xxind <- ind
    }

    if(m.sw["D.sw"]) {
      if(param == "means") {
        x.plot <- result$means
        if(matx) {
          plot.x  <- sapply(apply(x.plot, 1, stats::density), "[[", "y")
          graphics::matplot(plot.x, type="l", ylab="", lty=1)
          if(titles) graphics::title(main=list(paste0("Density", ifelse(all.ind, "", paste0(":\nMeans", ifelse(grp.ind, paste0(" - Group ", g), ""))))))
        } else   {
          plot.d  <- stats::density(x.plot[ind,])
          graphics::plot(plot.d, main="", ylab="")
          if(titles) graphics::title(main=list(paste0("Density", ifelse(all.ind, ":\n", paste0(":\nMeans - ", ifelse(grp.ind, paste0("Group ", g, " - "), ""))), var.names[ind], " Variable")))
          graphics::polygon(plot.d, col=grey)
        }
      }
      if(param == "scores") {
        x.plot <- x$Scores$eta
        if(by.fac) {
          plot.x  <- x.plot[,ind[2],]
        } else   {
          plot.x  <- if(Q > 1) x.plot[ind[1],,] else t(x.plot[ind[1],,])
        }
        if(matx) {
          plot.x  <- sapply(apply(plot.x, 1, stats::density), "[[", "y")
          graphics::matplot(plot.x, type="l", ylab="", lty=1)
          if(by.fac) {
            if(titles) graphics::title(main=list(paste0("Density", ifelse(all.ind, ":\n", ":\nScores - "), "Factor ", ind[2])))
          } else {
            if(titles) graphics::title(main=list(paste0("Density", ifelse(all.ind, ":\n", ":\nScores - "), "Observation ", obs.names[ind[1]])))
          }
        } else   {
          plot.d  <- stats::density(x.plot[ind[1],ind[2],])
          graphics::plot(plot.d, main="", ylab="")
          if(titles) graphics::title(main=list(paste0("Density", ifelse(all.ind, ":\n", ":\nScores - "), "Observation ", obs.names[ind[1]], ", Factor ", ind[2])))
          graphics::polygon(plot.d, col=grey)
        }
      }
      if(param == "loadings") {
        x.plot <- result$loadings
        plot.x    <- if(by.fac) x.plot[,ind[2],] else x.plot[ind[1],,]
        if(matx) {
          plot.x  <- sapply(apply(plot.x, 1, stats::density), "[[", "y")
          graphics::matplot(plot.x, type="l", ylab="", lty=1)
          if(by.fac) {
            if(titles) graphics::title(main=list(paste0("Density", ifelse(all.ind, ":\n", paste0(":\nLoadings - ", ifelse(grp.ind, paste0("Group ", g, " - "), ""))), "Factor ", ind[2])))
          } else {
            if(titles) graphics::title(main=list(paste0("Density", ifelse(all.ind, ":\n", paste0(":\nLoadings - ", ifelse(grp.ind, paste0("Group ", g, " - "), ""))), var.names[ind[1]], " Variable")))
          }
        } else   {
          plot.d  <- stats::density(x.plot[ind[1],ind[2],])
          graphics::plot(plot.d, main="", ylab="")
          if(titles) graphics::title(main=list(paste0("Density", ifelse(all.ind, ":\n", paste0(":\nLoadings - ", ifelse(grp.ind, paste0("Group ", g, " - "), ""))), var.names[ind[1]], " Variable, Factor ", ind[2])))
          graphics::polygon(plot.d, col=grey)
        }
      }
      if(param == "uniquenesses") {
        x.plot <- result$psi
        if(matx) {
          plot.x  <- sapply(apply(x.plot, 1, stats::density), "[[", "y")
          graphics::matplot(plot.x, type="l", ylab="", lty=1)
          if(titles) graphics::title(main=list(paste0("Density", ifelse(all.ind, "", paste0(":\nUniquenesses", ifelse(grp.ind, paste0(" - Group ", g), ""))))))
        } else   {
          plot.d  <- stats::density(x.plot[ind,])
          graphics::plot(plot.d, main="", ylab="")
          if(titles) graphics::title(main=list(paste0("Density", ifelse(all.ind, ":\n", paste0(":\nUniquenesses - ", ifelse(grp.ind, paste0("Group ", g, " - "), ""))), var.names[ind], " Variable")))
          graphics::polygon(plot.d, col=grey)
        }
      }
      if(param == "pis") {
        x.plot <- clust$pi.prop
        if(matx) {
          plot.x  <- sapply(apply(x.plot, 1, stats::density), "[[", "y")
          graphics::matplot(plot.x, type="l", ylab="", lty=1)
          if(titles) graphics::title(main=list(paste0("Density", ifelse(all.ind, "", paste0(":\nMixing Proportions")))))
        } else   {
          plot.d  <- stats::density(x.plot[ind,])
          graphics::plot(plot.d, main="", ylab="")
          if(titles) graphics::title(main=list(paste0("Density", ifelse(all.ind, "", paste0(":\nMixing Proportions - Group ", ind)))))
          graphics::polygon(plot.d, col=grey)
        }
      }
      if(param == "alpha") {
        plot.x <- clust$DP.alpha
        plot.d <- stats::density(plot.x$alpha)
        graphics::plot(plot.d, main="", ylab="")
        if(titles) graphics::title(main=list(paste0("Density", ifelse(all.ind, "", paste0(":\nAlpha")))))
        graphics::polygon(plot.d, col=grey)
        if(intervals) {
          avg  <- plot.x$post.alpha
          clip(avg, avg, 0, plot.d$y[which.min(abs(plot.d$x - avg))])
          graphics::abline(v=avg, col=2, lty=2)
        }
      }
    }

    if(m.sw["M.sw"])  {
      if(is.element(param, c("scores", "loadings"))) {
        if(indx)  {
          ind     <- switch(param, scores=c(1L, min(Q.max, 2L)), c(1L, 1L))
        }
        if(!facx) {
          ind[2]  <- fac[g]
        }
        if(param  == "scores") {
          if(any(ind[1]  > Q.max,
                 ind[2]  > Q.max))    stop(paste0("Only the first ", Q.max, " columns can be plotted"))
        } else if(ind[2] > Q)         stop(paste0("Only the first ", Q, " columns can be plotted"))
      }
      if(param == "means") {
        plot.x <- result$post.mu
        if(ci.sw[param])  ci.x   <- result$ci.mu
        graphics::plot(plot.x, type=type, ylab="", xlab="Variable", ylim=if(is.element(method, c("FA", "IFA"))) c(-1, 1) else if(ci.sw[param]) c(min(ci.x[,1]), max(ci.x[,2])))
        if(all(intervals, ci.sw[param])) plotrix::plotCI(plot.x, li=ci.x[,1], ui=ci.x[,2], slty=3, scol=grey, add=TRUE, gap=TRUE, pch=ifelse(type == "n", NA, 20))
        if(titles) graphics::title(main=list(paste0("Posterior Mean", ifelse(all.ind, "", paste0(":\nMeans", ifelse(grp.ind, paste0(" - Group ", g), ""))))))
        if(type  == "n") graphics::text(x=seq_along(plot.x), y=plot.x, var.names, cex=0.5)
      }
      if(param == "scores") {
        labs   <- if(grp.ind) clust$map else 1
        if(g.score)  {
          if(g.ind == 1)  tmplab <- labs
          z.ind  <- as.numeric(levels(tmplab))[tmplab] %in% g
          plot.x <- x$Scores$post.eta[z.ind,,drop=FALSE]
          ind2   <- ifelse(any(!facx, Q <= 1), ind[2], if(Q > 1) max(2, ind[2]))
          if(ci.sw[param]) ci.x  <- x$Scores$ci.eta[,z.ind,, drop=FALSE]
          labs   <- g
        } else       {
          plot.x <- x$Scores$post.eta
          ind2   <- ifelse(any(!facx, Q.max <= 1), ind[2], if(Q.max > 1) max(2, ind[2]))
          if(ci.sw[param]) ci.x  <- x$Scores$ci.eta
        }
        col.s  <- if(is.factor(labs)) as.numeric(levels(labs))[labs] else labs
        type.s <- ifelse(any(type.x, type == "l"), "p", type)
        if(ind2 != 1)  {
          if(all(intervals, ci.sw[param])) {
            plotrix::plotCI(plot.x[,ind[1]], plot.x[,ind2], li=ci.x[1,,ind2], ui=ci.x[2,,ind2], gap=TRUE, pch=NA, scol=grey, slty=3, xlab=paste0("Factor ", ind[1]), ylab=paste0("Factor ", ind2))
            plotrix::plotCI(plot.x[,ind[1]], plot.x[,ind2], li=ci.x[1,,ind[1]], ui=ci.x[2,,ind[1]], add=TRUE, gap=TRUE, pch=NA, scol=grey, slty=3, err="x")
            if(type.s != "n") graphics::points(plot.x[,ind[1]], plot.x[,ind2], type=type.s, col=col.s, pch=20)
          } else {
            graphics::plot(plot.x[,ind[1]], plot.x[,ind2], type=type.s, col=col.s, pch=20,
                 xlab=paste0("Factor ", ind[1]), ylab=paste0("Factor ", ind2))
          }
          if(titles) graphics::title(main=list(paste0("Posterior Mean", ifelse(all.ind, "", ":\nScores"), ifelse(g.score, paste0(" - Group ", g), ""))))
          if(type.s == "n") graphics::text(plot.x[,ind[1]], plot.x[,ind2], obs.names, col=col.s, cex=0.5)
        } else   {
          if(all(intervals, ci.sw[param])) {
            plotrix::plotCI(if(!g.score) seq_len(n.obs) else seq_len(sum(z.ind)), plot.x[,ind[1]], li=ci.x[1,,ind[1]], ui=ci.x[2,,ind[1]], gap=TRUE, pch=NA, scol=grey, slty=3, xlab="Observation", ylab=paste0("Factor ", ind[1]))
            graphics::points(plot.x[,ind[1]], type=type.s, col=col.s, pch=20)
          } else {
            graphics::plot(plot.x[,ind[1]], type=type.s, col=col.s, xlab="Observation", ylab=paste0("Factor ", ind[1]), pch=20)
          }
          if(titles) graphics::title(main=list(paste0("Posterior Mean", ifelse(all.ind, "", ":\nScores"), ifelse(g.score, paste0(" - Group ", g), ""))))
          if(type.s == "n") graphics::text(plot.x[,ind[1]], col=col.s, cex=0.5)
        }
      }
      if(param == "loadings") {
        plot.x <- result$post.load
        if(load.meth == "heatmap") {
          if(Q  > 1) {
            gclus::plotcolors(mat2cols(plot.x))
          } else {
            graphics::image(z=t(plot.x[seq(n.var, 1),seq_len(Q)]), xlab="", ylab="", xaxt="n", yaxt="n", col=viridis::viridis(30, option="C"))
          }
          if(titles) graphics::title(main=list(paste0("Posterior Mean", ifelse(!all.ind, " Loadings ", " "), "Heatmap", ifelse(all(!all.ind, grp.ind), paste0(" - Group ", g), ""))))
          graphics::axis(1, line=-0.5, tick=FALSE, at=if(Q != 1) seq_len(Q) else 0, labels=seq_len(Q))
          if(n.var < 100) {
            graphics::axis(2, cex.axis=0.5, line=-0.5, tick=FALSE, las=1, at=if(Q > 1) seq_len(n.var) else seq(from=0, to=1, by=1/(n.var - 1)), labels=substring(var.names[n.var:1], 1, 10))
          }
          graphics::box(lwd=2)
          graphics::mtext(ifelse(Q > 1, "Factors", "Factor"), side=1, line=2)
          if(Q != 1) graphics::abline(v=seq(1, Q - 1, 1) + 0.5, lty=2, lwd=1)
        } else {
          if(ci.sw[param]) ci.x  <- result$ci.load
          if(!by.fac) {
           if(ci.sw[param]) ci.x <- ci.x[,ind[1],]
            graphics::plot(plot.x[ind[1],], type=type, xaxt="n", xlab="", ylab="Loading", ylim=if(ci.sw[param]) c(min(ci.x[1,]), max(ci.x[2,])))
            if(all(intervals, ci.sw[param])) plotrix::plotCI(plot.x[ind[1],], li=ci.x[1,], ui=ci.x[2,], slty=3, scol=grey, add=TRUE, gap=TRUE, pch=ifelse(type == "n", NA, 20))
            graphics::axis(1, line=-0.5, tick=FALSE, at=seq_len(Q), labels=seq_len(Q))
            graphics::mtext("Factors", side=1, line=2)
            if(titles) graphics::title(main=list(paste0(ifelse(!all.ind, paste0("Loadings - ", ifelse(grp.ind, paste0("Group ", g, " - "), "")), ""), var.names[ind[1]], " Variable")))
            if(type == "n") graphics::text(x=plot.x[ind[1],], paste0("Factor ", seq_len(Q)), cex=0.5)
          } else     {
           if(ci.sw[param]) ci.x <- ci.x[,,ind[2]]
            graphics::plot(plot.x[,ind[2]], type=type, xaxt="n", xlab="", ylab="Loading", ylim=if(ci.sw[param]) c(min(ci.x[1,]), max(ci.x[2,])))
            if(all(intervals, ci.sw[param])) plotrix::plotCI(plot.x[,ind[2]], li=ci.x[1,], ui=ci.x[2,], slty=3, scol=grey, add=TRUE, gap=TRUE, pch=ifelse(type == "n", NA, 20))
            graphics::axis(1, line=-0.5, tick=FALSE, at=seq_len(n.var), labels=seq_len(n.var))
            graphics::mtext("Variable #", side=1, line=2, cex=0.8)
            if(titles) graphics::title(main=list(paste0(ifelse(!all.ind, paste0("Loadings - ", ifelse(grp.ind, paste0("Group ", g, " - "), "")), ""), "Factor ", ind[2])))
            if(type == "n") graphics::text(x=plot.x, var.names, cex=0.5)
          }
        }
      }
      if(param == "uniquenesses") {
        plot.x <- result$post.psi
        if(ci.sw[param])  ci.x   <- result$ci.psi
        graphics::plot(plot.x, type=type, ylab="", xlab="Variable", ylim=if(ci.sw[param]) c(min(ci.x[,1]), max(ci.x[,2])))
        if(all(intervals, ci.sw[param])) plotrix::plotCI(plot.x, li=ci.x[,1], ui=ci.x[,2], slty=3, scol=grey, add=TRUE, gap=TRUE, pch=ifelse(type == "n", NA, 20))
        if(titles) graphics::title(main=list(paste0("Posterior Mean", ifelse(all.ind, "", paste0(":\nUniquenesses", ifelse(grp.ind, paste0(" - Group ", g), ""))))))
        if(type  == "n") graphics::text(seq_along(plot.x), plot.x, var.names, cex=0.5)
      }
      if(param == "pis") {
        plot.x <- clust$post.pi
        if(ci.sw[param])  ci.x   <- clust$ci.pi
        if(matx) {
          if(all(intervals, ci.sw[param])) {
            plotrix::plotCI(graphics::barplot(plot.x, ylab="", xlab="", col=grey, ylim=c(0, 1), cex.names=0.7),
                   plot.x, li=ci.x[,1], ui=ci.x[,2], slty=3, scol=2, add=TRUE, gap=TRUE, pch=20)
          } else {
            graphics::barplot(plot.x, ylab="", xlab="", ylim=c(0, 1), cex.names=0.7)
          }
          if(titles) graphics::title(main=list(paste0("Posterior Mean", ifelse(all.ind, "", paste0(":\nMixing Proportions")))))
        } else {
          if(all(intervals, ci.sw[param])) {
            plotrix::plotCI(graphics::barplot(plot.x[ind], ylab="", xlab="", ylim=c(0, 1), cex.names=0.7),
                   plot.x[ind], li=ci.x[1,ind], ui=ci.x[2,ind], slty=3, scol=2, add=TRUE, gap=TRUE, pch=20)
          } else {
            graphics::barplot(plot.x[ind], ylab="", xlab="Variable", ylim=c(0, 1), cex.names=0.7)
          }
          if(titles) graphics::title(main=list(paste0("Posterior Mean", ifelse(all.ind, "", paste0(":\nMixing Proportions - Group ", ind)))))
        }
      }
      if(param  == "alpha") {
        graphics::plot(c(0, 1), c(0, 1), ann=FALSE, bty='n', type='n', xaxt='n', yaxt='n')
        if(titles) graphics::title(main=list(paste0("Summary Statistics", ifelse(all.ind, "", ":\nAlpha"))))
        plot.x <- clust$DP.alpha[-1]
        a.step <- attr(x, "Alph.step")
        conf   <- attr(x, "Conf.Level")
        digits <- options()$digits
        a.adj  <- rep(0.5, 2)
        a.cex  <- graphics::par()$fin[2]/switch(a.step, metropolis=5, 4)
        pen    <- switch(a.step, metropolis=0, 0.125)
        graphics::text(x=0.5, y=0.85 - pen, cex=a.cex, col="black", adj=a.adj, expression(bold("Posterior Mean:\n")))
        graphics::text(x=0.5, y=0.85 - pen, cex=a.cex, col="black", adj=a.adj, bquote(.(round(plot.x$post.alpha, digits))))
        graphics::text(x=0.5, y=0.57 - pen, cex=a.cex, col="black", adj=a.adj, expression(bold("\nVariance:\n")))
        graphics::text(x=0.5, y=0.57 - pen, cex=a.cex, col="black", adj=a.adj, bquote(.(round(plot.x$var.alpha, digits))))
        graphics::text(x=0.5, y=0.4  - pen, cex=a.cex, col="black", adj=a.adj, bquote(bold(.(100 * conf))~bold("% Confidence Interval:")))
        graphics::text(x=0.5, y=0.28 - pen, cex=a.cex, col="black", adj=a.adj, bquote(paste("[", .(round(plot.x$ci.alpha[1], digits)), ", ", .(round(plot.x$ci.alpha[2], digits)), "]")))
        if(a.step == "metropolis") {
          graphics::text(x=0.5, y=0.17, cex=a.cex, col="black", adj=a.adj, expression(bold("Acceptance Rate:")))
          graphics::text(x=0.5, y=0.1,  cex=a.cex, col="black", adj=a.adj, bquote(paste(.(round(100 * plot.x$acceptance.rate, 2)), "%")))
        }
      }
      if(!indx) {         ind[1] <- xind[1]
        if(all(facx, is.element(param, c("scores",
           "loadings")))) ind[2] <- xind[2]
      }
      if(all.ind)          ind   <- xxind
    }

    if(m.sw["G.sw"]) {
      plotG.ind  <- is.element(method, c("IMIFA", "IMFA", "OMIFA", "OMFA"))
      plotQ.ind  <- any(any(g > 1, is.element(method, c("IFA", "MIFA"))), all(is.element(method, c("IMIFA", "OMIFA")), g != 1))
      aicm       <- round(GQ.res$AICMs, 2)
      bicm       <- round(GQ.res$BICMs, 2)
      log.iLLH   <- round(GQ.res$LogIntegratedLikelihoods, 2)
      if(is.element(method, c("FA", "MFA", "OMFA", "IMFA"))) {
        aic.mcmc <- round(GQ.res$AIC.mcmcs, 2)
        bic.mcmc <- round(GQ.res$BIC.mcmcs, 2)
        dic      <- round(GQ.res$DICs, 2)
      }
      if(all(plotG.ind, g == 1))  {
        graphics::layout(1)
        graphics::par(mar=c(5.1, 4.1, 4.1, 2.1))
        plot.G <- GQ.res$G.Counts
        G.name <- names(plot.G)
        rangeG <- as.numeric(G.name)
        rangeG <- seq(from=min(rangeG), to=max(rangeG), by=1)
        missG  <- setdiff(rangeG, G.name)
        missG  <- stats::setNames(rep(0, length(missG)), as.character(missG))
        plot.G <- c(plot.G, missG)
        plot.G <- plot.G[Rfast::Order(as.numeric(names(plot.G)))]
        col.G  <- c(1, ceiling(length(palette)/2))[(rangeG == G) + 1]
        G.plot <- graphics::barplot(plot.G, ylab="Frequency", xaxt="n", col=col.G)
        if(titles) graphics::title(main=list("Posterior Distribution of G"))
        graphics::axis(1, at=G.plot, labels=names(plot.G), tick=FALSE)
        graphics::axis(1, at=Rfast::med(G.plot), labels="G", tick=FALSE, line=1.5)
      }
      if(all(method == "IFA", plotQ.ind)) {
        graphics::layout(1)
        graphics::par(mar=c(5.1, 4.1, 4.1, 2.1))
        plot.Q <- GQ.res$Q.Counts
        Q.name <- names(plot.Q)
        rangeQ <- as.numeric(Q.name)
        rangeQ <- seq(from=min(rangeQ), to=max(rangeQ), by=1)
        missQ  <- setdiff(rangeQ, Q.name)
        missQ  <- stats::setNames(rep(0, length(missQ)), as.character(missQ))
        plot.Q <- c(plot.Q, missQ)
        plot.Q <- plot.Q[Rfast::Order(as.numeric(names(plot.Q)))]
        col.Q  <- c(1, ceiling(length(palette)/2))[(rangeQ == Q) + 1]
        Q.plot <- graphics::barplot(plot.Q, ylab="Frequency", xaxt="n", col=col.Q)
        if(titles) graphics::title(main=list("Posterior Distribution of Q"))
        graphics::axis(1, at=Q.plot, labels=names(plot.Q), tick=FALSE)
        graphics::axis(1, at=Rfast::med(Q.plot), labels="Q", tick=FALSE, line=1.5)
      }
      if(all(method != "IFA", plotQ.ind)) {
        plot.Q <- GQ.res$Q.Counts
        plot.Q <- if(is.list(plot.Q)) plot.Q else list(plot.Q)
        Q.name <- lapply(plot.Q, names)
        rangeQ <- as.numeric(unique(unlist(Q.name, use.names=FALSE)))
        rangeQ <- seq(from=min(rangeQ), to=max(rangeQ), by=1)
        missQ  <- lapply(Gseq, function(g) setdiff(rangeQ, as.numeric(Q.name[[g]])))
        missQ  <- lapply(Gseq, function(g) stats::setNames(rep(0, length(missQ[[g]])), as.character(missQ[[g]])))
        plot.Q <- lapply(Gseq, function(g) c(plot.Q[[g]], missQ[[g]]))
        plot.Q <- do.call(rbind, lapply(Gseq, function(g) plot.Q[[g]][Rfast::Order(as.numeric(names(plot.Q[[g]])))]))
        if(titles) {
          graphics::layout(rbind(1, 2), heights=c(9, 1))
          graphics::par(mar=c(3.1, 4.1, 4.1, 2.1))
        }
        Q.plot <- graphics::barplot(plot.Q, beside=TRUE, ylab="Frequency", xaxt="n", col=Gseq, space=c(0, 2))
        if(titles) graphics::title(main=list(expression('Posterior Distribution of Q'["g"])))
        graphics::axis(1, at=matrixStats::colMedians(Q.plot), labels=colnames(plot.Q), tick=FALSE)
        graphics::axis(1, at=Rfast::med(Q.plot), labels="Q", tick=FALSE, line=1)
        if(titles) {
          graphics::par(mar=c(0, 0, 0, 0))
          graphics::plot.new()
          tmp  <- if(G > 5) unlist(lapply(Gseq, function(g) c(Gseq[g], Gseq[g + ceiling(G/2)])))[Gseq] else Gseq
          ltxt <- paste0("Group ", tmp)
          lcol <- Gseq[tmp]
          graphics::legend("center", legend=ltxt, ncol=if(G > 5) ceiling(G/2) else G, bty="n", pch=15, col=lcol, cex=max(0.7, 1 - 0.03 * G))
        }
      }
      if(!any(plotQ.ind, plotG.ind))  message("Nothing to plot")
      gq.nam <- substring(names(GQ.res), 1, 1)
      if(is.element(method, c("IMIFA", "OMIFA"))) {
        if(g == 1) {
          print(GQ.res[gq.nam == "G"])
        } else {
          print(GQ.res[gq.nam == "Q"])
        }
        if(g == max(Gs)) {
          print(GQ.res[gq.nam != "G" & gq.nam != "Q" & gq.nam != "S"])
        }
      } else if(is.element(method, c("MFA", "MIFA", "OMFA", "IMFA"))) {
          print(GQ.res[gq.nam != "S"])
      } else switch(method, IFA={
          print(utils::tail(GQ.res[gq.nam != "S"], -1))
        },
          cat(paste0("Q = ", Q, "\n"))
      )
      if(any(dim(bicm) > 1)) {
        G.ind  <- ifelse(any(G.supp, !is.element(method, c("MFA", "MIFA"))), 1, which(n.grp == G))
        Q.ind  <- ifelse(any(Q.supp, !is.element(method, c("FA", "MFA"))),   1, which(n.fac == Q))
        if(!is.element(method, c("IFA", "MIFA"))) {
          cat(paste0("AIC.mcmc = ", aic.mcmc[G.ind,Q.ind], "\n"))
          cat(paste0("BIC.mcmc = ", bic.mcmc[G.ind,Q.ind], "\n"))
          cat(paste0("DIC = ", dic[G.ind,Q.ind], "\n"))
        }
          cat(paste0("AICM = ", aicm[G.ind,Q.ind], "\n"))
          cat(paste0("BICM = ", bicm[G.ind,Q.ind], "\n"))
          cat(paste0("Log Integrated Likelihood = ", log.iLLH[G.ind,Q.ind], "\n"))
      }
      if(all(plotQ.ind,
             attr(GQ.res, "Q.big")))  warning("Q had to be prevented from exceeding its initial value.\n Consider re-running the model with a higher value for 'range.Q'", call.=FALSE)
    }

    if(m.sw["Z.sw"]) {
      if(type == "l")                 stop("'type' cannot be 'l' for clustering uncertainty plots")
      plot.x <- clust$uncertainty
      if(g == 1) {
        col.x  <- c(1, ceiling(length(palette)/2))[(plot.x >= 1/G) + 1]
        if(type != "h") col.x[plot.x == 0] <- NA
        if(titles) {
          graphics::layout(rbind(1, 2), heights=c(1, 6))
          graphics::par(mar=c(0, 4.1, 0.5, 2.1))
          graphics::plot.new()
          graphics::legend("center", legend=bquote(1/G == 1/.(G)), title="", lty=2, col=2, bty="n", y.intersp=graphics::par()$fin[2] * 7/5)
          graphics::legend("center", legend=c(" "," "), title=expression(bold("Clustering Uncertainty")), bty='n', y.intersp=graphics::par()$fin[2] * 2/5, cex=graphics::par()$cex.main)
          graphics::par(mar=c(5.1, 4.1, 0.5, 2.1))
        }
        graphics::plot(plot.x, type=type, ylim=c(0, 1 - 1/G), col=col.x, axes=FALSE, ylab="Uncertainty", xlab="Observation", pch=ifelse(type == "n", NA, 16))
        graphics::rect(0, 0, n.obs, 1 - 1/G)
        if(G == 2) {
          graphics::abline(h=0.5, col=graphics::par()$bg)
          graphics::abline(v=0,   col=graphics::par()$bg)
        }
        graphics::lines(x=c(0, n.obs), y=c(1/G, 1/G), lty=2, col=2)
        graphics::axis(1, las=1, pos=0, cex.axis=0.9)
        graphics::axis(2, at=c(seq(from=0, to=min(1 - 1/G - 1/1000, 0.8), by=0.1), 1 - 1/G), labels=c(seq(from=0, to=min(1 - 1/G - 1/1000, 0.8), by=0.1), "1 - 1/G"), las=2, pos=0, cex.axis=0.9)
        if(type == "n")  {
          znam  <- obs.names
          znam[plot.x == 0] <- ""
          graphics::text(x=seq_along(plot.x), y=plot.x, znam, col=col.x, cex=0.5)
        }
      } else {
        if(titles) {
          graphics::layout(rbind(1, 2), heights=c(1, 6))
          graphics::par(mar=c(0, 4.1, 0.5, 2.1))
          graphics::plot.new()
          graphics::legend("center", legend=bquote({NA >= 1/G} == 1/.(G)), title="", pch=15, col=3, bty="n", y.intersp=graphics::par()$fin[2] * 7/5)
          graphics::legend("center", legend=c(" "," "), title=expression(bold("Clustering Uncertainty")), bty='n', y.intersp=graphics::par()$fin[2] * 2/5, cex=graphics::par()$cex.main)
          graphics::par(mar=c(5.1, 4.1, 0.5, 2.1))
        }
        x.plot  <- graphics::hist(plot.x, plot=FALSE)
        breaks  <- if(sum(plot.x   != 0)) x.plot$breaks else seq(from=0, to=max(plot.x, 1/G), by=1/G)
        cols    <- 2     + (breaks >= 1/G)
        cols[cols == 2] <- grey
        graphics::plot(x.plot, main="", xlab="Uncertainties", xlim=c(0, 1 - 1/G), col=cols, xaxt="n", ylim=c(0, max(x.plot$counts)), yaxt="n")
        graphics::axis(1, at=c(breaks[round(breaks, 1) < min(0.8, 1 - 1/G)], 1 - 1/G), labels=(c(round(breaks[round(breaks, 1) < min(0.8, 1 - 1/G)], 3), "1 - 1/G")), las=2, pos=0, cex.axis=0.8)
        graphics::axis(2, at=if(sum(plot.x)  == 0) c(graphics::axTicks(2), max(x.plot$counts)) else graphics::axTicks(2), las=1, cex.axis=0.8)
      }
      if(g == min(Gs)) {
        if(any(!labelmiss,  !z.miss)) {
          if(all(!labelmiss, z.miss)) {
           prf  <- clust$perf
          } else   {
           pzs  <- clust$map
           if(nlevels(pzs) == nlevels(labs)) {
            lsw <- .lab.switch(z.new=pzs, z.old=labs, Gs=seq_len(G))
            pzs <- factor(lsw$z)
           }
           tab  <- table(pzs, labs, dnn=list("Predicted", "Observed"))
           prf  <- c(e1071::classAgreement(tab), mclust::classError(pzs, labs))
           if(nrow(tab) != ncol(tab))   {
            prf <- prf[-seq_len(2)]
            names(prf)[4]        <- "error.rate"
           } else {
            names(prf)[6]        <- "error.rate"
           }
           if(prf$error.rate     == 0)  {
            prf$misclassified    <- NULL
           }
           prf  <- c(list(confusion.matrix = tab), prf)
           if(nlevels(pzs) == nlevels(labs)) {
            names(prf)[1]  <- "matched.confusion.matrix"
           }
           class(prf)      <- "listof"
          }
          ucert <- attr(plot.x, "Obs")
          if(!is.null(ucert)) {
           prf  <- c(prf, list(uncertain = ucert))
          }
          prf$error.rate   <- paste0(round(100 * prf$error.rate, 2), "%")
          print(prf)
        } else                        message("Nothing to print: try supplying known cluster labels")
      }
    }

    if(m.sw["P.sw"]) {
      plot.x <- switch(param, means=x$Means$post.mu, uniquenesses=x$Uniquenesses$post.psi, x$Loadings$post.load[[g]])
      x.plot <- apply(plot.x, 1L, range, na.rm=TRUE)
      plot.x <- if(all(param == "uniquenesses", uni.type == "isotropic")) plot.x else apply(plot.x, 2L, function(x) (x - min(x, na.rm=TRUE))/(max(x, na.rm=TRUE) - min(x, na.rm=TRUE)))
      varnam <- paste0(toupper(substr(param, 1, 1)), substr(param, 2, nchar(param)))
      if(any(grp.ind, param == "loadings")) {
        grDevices::adjustcolor(grDevices::palette(viridis::viridis(Q.max)), alpha.f=transparency)
        graphics::layout(rbind(1, 2), heights=c(9, 1))
        graphics::par(mar=c(3.1, 4.1, 4.1, 2.1))
      }
      jitcol <- switch(param, loadings=Q, G)
      graphics::matplot(seq_len(n.var) + matrix(rnorm(jitcol * n.var, 0, min(0.1, 0.75/n.var^2)), nrow=n.var, ncol=jitcol), plot.x, type=switch(param, uniquenesses=switch(uni.type, unconstrained="p", isotropic="l"), "p"),
                        col=switch(param, loadings=seq_len(Q), seq_len(G)), pch=15, xlab="Variable", ylab=paste0("Standardised ", varnam), axes=FALSE, main=paste0("Parallel Coordinates: ", varnam, ifelse(all(grp.ind, param == "loadings"), paste0("\n Group ", g), "")), lty=1)
      graphics::axis(1, at=seq_len(n.var), labels=if(titles && n.var < 100) rownames(plot.x) else rep("", n.var), cex.axis=0.5, tick=FALSE)
      for(i in seq_len(n.var))    {
        graphics::lines(c(i, i), c(0, 1), col=grey)
        if(titles && n.var < 100) {
          graphics::text(c(i, i), c(0, 1), labels=format(x.plot[,i], digits=3), xpd=NA, offset=0.3, pos=c(1, 3), cex=0.5)
        }
      }
      if(any(grp.ind, param  == "loadings")) {
        graphics::par(mar=c(0, 0, 0, 0))
        graphics::plot.new()
        Xp   <- ifelse(param == "loadings", Q, G)
        Xseq <- seq_len(Xp)
        tmp  <- if(Xp > 5) unlist(lapply(Xseq, function(x) c(Xseq[x], Xseq[x + ceiling(Xp/2)])))[Xseq] else Xseq
        ltxt <- paste0(switch(param, loadings="Factor", "Group"), tmp)
        lcol <- Xseq[tmp]
        graphics::legend("center", pch=15, col=lcol, legend=ltxt, ncol=if(Xp > 5) ceiling(Xp/2) else Xp, bty="n", cex=max(0.7, 1 - 0.03 * Xp))
      }
    }

    if(m.sw["E.sw"]) {
      x.plot <- x$Error
      plot.x      <- if(G > 1) cbind(do.call(rbind, x.plot[-length(x.plot)]), Averages = x.plot$Averages) else x.plot
      if(titles) {
        graphics::layout(rbind(1, 2), heights=c(9, 1))
        graphics::par(mar=c(3.1, 4.1, 4.1, 2.1))
      }
      grDevices::adjustcolor(grDevices::palette(viridis::viridis(nrow(plot.x), option="C")), alpha.f=transparency)
      col.e  <- if(G > 1) seq_len(nrow(plot.x)) else seq_along(plot.x)
      if(G    > 1)   {
        dens <- matrix(-1, nrow=nrow(plot.x), ncol=G + 1)
        dens[,G + 1]       <- 30
      } else  {
        dens <- NULL
      }
      pl.x   <- graphics::barplot(plot.x, beside=TRUE, col=col.e, main="", ylab="Deviation", density=dens)
      na.x   <- G > 1 & is.na(x.plot[[1]])
      if(G > 1) graphics::points(x=matrixStats::colMedians(pl.x[,which(na.x)]), y=rep(0, sum(na.x)), pch=16, col="red")
      if(titles) graphics::title(main=list("Error Metrics"))
      if(titles) {
        graphics::par(mar=c(0, 0, 0, 0))
        graphics::plot.new()
        ltxt <- c("MSE", "MAE", "MEDSE", "MEDAE", "RMSE", "NRMSE"
                  #, "CVRMSE"
                  )
        lnc  <- length(col.e)
        lcol <- col.e
        xna  <- sum(na.x)   > 0
        lpch <- rep(15, nrow(plot.x))
        temp <- graphics::legend("center", legend=if(xna) c(ltxt, "Missing") else ltxt, ncol=ifelse(xna, lnc + 1, lnc), bty="n",
                       pch=if(xna) c(lpch, 16) else lpch, col=if(xna) c(lcol, "red") else lcol, cex=0.8)
        if(xna) graphics::text(x=temp$text$x[8] - 0.01, y=temp$text$y[8] + 0.01, "__")
      }
      if(G > 1) {
        avg  <- stats::setNames(list(x.plot$Averages), "Average Error Metrics")
        class(avg)         <- "listof"
      } else {
        avg  <- x.plot
      }
      print(avg)
    }

    if(m.sw["C.sw"]) {
      if(!all.ind)   {
       partial <- FALSE
       graphics::par(mai=c(1.25, 1, 0.75, 0.5), mfrow=c(1, 2), oma=c(0, 0, 2, 0))
      }
      if(param == "means")    {
        plot.x <- result$means
        if(!partial) {
          stats::acf(plot.x[ind,], main="", ci.col=4, ylab="")
          if(titles) graphics::title(main=list(paste0("ACF", ifelse(all.ind, paste0(":\n", var.names[ind], " Variable"), ""))))
        }
        if(any(!all.ind, partial)) {
          stats::acf(plot.x[ind,], main="", type="partial", ci.col=4, ylab="")
          if(titles) graphics::title(main=list(paste0("PACF", ifelse(partial, paste0(":\n", var.names[ind], " Variable"), ""))))
          if(all(!all.ind, titles)) graphics::title(main=list(paste0("Means - ", ifelse(grp.ind, paste0("Group ", g, ":\n "), ""), var.names[ind], " Variable")), outer=TRUE)
        }
      }
      if(param == "scores")   {
        plot.x <- x$Scores$eta
        if(!partial) {
          stats::acf(plot.x[ind[1],ind[2],], main="", ci.col=4, ylab="")
          if(titles) graphics::title(main=list(paste0("ACF", ifelse(all.ind, paste0(":\n", "Observation ", obs.names[ind[1]], ", Factor ", ind[2]), ""))))
        }
        if(any(!all.ind, partial)) {
          stats::acf(plot.x[ind[1],ind[2],], main="", type="partial", ci.col=4, ylab="")
          if(titles) graphics::title(main=list(paste0("PACF", ifelse(partial, paste0(":\n", "Observation ", obs.names[ind[1]], ", Factor ", ind[2]), ""))))
          if(all(!all.ind, titles)) graphics::title(main=list(paste0("Scores - ", "Observation ", obs.names[ind[1]], ", Factor ", ind[2])), outer=TRUE)
        }
      }
      if(param == "loadings") {
        plot.x <- result$loadings
        if(!partial) {
          stats::acf(plot.x[ind[1],ind[2],], main="", ci.col=4, ylab="")
          if(titles) graphics::title(main=list(paste0("ACF", ifelse(all.ind, paste0(":\n", var.names[ind[1]], " Variable, Factor ", ind[2]), ""))))
        }
        if(any(!all.ind, partial)) {
          stats::acf(plot.x[ind[1],ind[2],], main="", type="partial", ci.col=4, ylab="")
          if(titles) graphics::title(main=list(paste0("PACF", ifelse(partial, paste0(":\n", var.names[ind[1]], " Variable, Factor ", ind[2]), ""))))
          if(all(!all.ind, titles)) graphics::title(main=list(paste0("Loadings - ", ifelse(grp.ind, paste0("Group ", g, ":\n "), ""), var.names[ind[1]], " Variable, Factor ", ind[2])), outer=TRUE)
        }
      }
      if(param == "uniquenesses")  {
        plot.x <- result$psi
        if(!partial) {
          stats::acf(plot.x[ind,], main="", ci.col=4, ylab="")
          if(titles) graphics::title(main=list(paste0("ACF", ifelse(all.ind, paste0(":\n", var.names[ind], " Variable"), ""))))
        }
        if(any(!all.ind, partial)) {
          stats::acf(plot.x[ind,], main="", type="partial", ci.col=4, ylab="")
          if(titles) graphics::title(main=list(paste0("PACF", ifelse(partial, paste0(":\n", var.names[ind], " Variable"), ""))))
          if(all(!all.ind, titles)) graphics::title(main=list(paste0("Uniquenesses - ", ifelse(grp.ind, paste0("Group ", g, ":\n "), ""), var.names[ind], " Variable")), outer=TRUE)
        }
      }
      if(param == "pis")  {
        plot.x <- clust$pi.prop
        if(!partial) {
          stats::acf(plot.x[ind,], main="", ci.col=4, ylab="")
          if(titles) graphics::title(main=list(paste0("ACF", ifelse(all(all.ind, matx), paste0(" - Group ", ind), ""))))
        }
        if(any(!all.ind, partial)) {
          stats::acf(plot.x[ind,], main="", type="partial", ci.col=4, ylab="")
          if(titles) graphics::title(main=list(paste0("PACF", ifelse(all(all.ind, matx), paste0(" - Group ", ind), ""))))
          if(all(!all.ind, titles)) graphics::title(main=list(paste0("Mixing Proportions - Group ", ind)), outer=TRUE)
        }
      }
      if(param == "alpha") {
        plot.x <- clust$DP.alpha$alpha
        if(clust$DP.alpha$acceptance.rate == 0)  {
                                      warning(paste0("0% acceptance rate: can't plot ", ifelse(all.ind, ifelse(partial, "partial-", "auto-"), ""), "correlation function", ifelse(all.ind, "", "s")), call.=FALSE)
          next
        }
        if(!partial) {
          stats::acf(plot.x, main="", ci.col=4, ylab="")
          if(titles) graphics::title(main=list(paste0("ACF")))
        }
        if(any(!all.ind, partial)) {
          stats::acf(plot.x, main="", type="partial", ci.col=4, ylab="")
          if(titles) graphics::title(main=list(paste0("PACF")))
          if(all(!all.ind, titles)) graphics::title(main=list(paste0("Alpha")), outer=TRUE)
        }
      }
    }
    if(all(all.ind, titles)) graphics::title(ifelse(param != "pis", paste0(toupper(substr(param, 1, 1)), substr(param, 2, nchar(param)),
                             ifelse(all(grp.ind, !is.element(param, c("scores", "pis", "alpha"))), paste0(" - Group ", g), "")),
                             paste0("Mixing Proportions", ifelse(matx, "", paste0(" - Group ", ind)))), outer=TRUE)
    if(isTRUE(msgx)) .ent_exit()
  }
}

# Loadings Heatmaps
#' Convert a numeric matrix to colours
#'
#' Converts a matrix to a hex colour code representation for plotting using \code{\link[gclus]{plotcolors}}. Used internally by \code{\link{plot.Results_IMIFA}} for plotting posterior mean loadings heatmaps.
#' @param mat A matrix.
#' @param cols The colour palette to be used. The default palette uses \code{\link[viridis]{viridis}}. Will be checked for validity.
#' @param byrank Logical indicating whether to convert the matrix itself or the sample ranks of the values therein. Defaults to FALSE.
#' @param breaks Number of gradations in colour to use. Defaults to \code{length(cols)}.
#'
#' @return A matrix of hex colour code representations.
#' @export
#' @importFrom grDevices "col2rgb" "heat.colors"
#' @importFrom gclus "plotcolors"
#' @importFrom viridis "viridis"
#'
#' @seealso \code{\link[gclus]{plotcolors}}
#'
#' @examples
#' mat  <- matrix(rnorm(100), nrow=10, ncol=10)
#' cols <- mat2cols(mat)
#' cols
#'
#' # Use gclus::plotcolors() to visualise the colours matrix
#' # plotcolors(cols)
  mat2cols     <- function(mat, cols = NULL, byrank = FALSE, breaks = length(cols)) {
    m          <- as.matrix(mat)
    if(missing(cols)) cols <- viridis::viridis(30L, option="C")
    if(!all(.are_cols(cols)))         stop("Invalid colours supplied")
    if(any(!is.logical(byrank),
           length(byrank)  != 1))     stop("'byrank' must be TRUE or FALSE")
    if(any(!is.numeric(breaks),
           length(breaks)  != 1))     stop("'breaks' must be a single digit")
    m1         <- if(isTRUE(byrank))  rank(m) else m
    facs       <- cut(m1, breaks, include.lowest=TRUE)
    answer     <- matrix(cols[as.numeric(facs)], nrow=nrow(m), ncol=ncol(m))
    rownames(answer)       <- rownames(m)
    colnames(answer)       <- colnames(m)
      answer
  }

# Colour Checker
  .are_cols    <- function(cols) {
    vapply(cols,  function(x) { tryCatch(is.matrix(grDevices::col2rgb(x)), error = function(e) FALSE) }, logical(1))
  }

# Prior No. Groups (DP & PY)
#' Plot Dirichlet / Pitman-Yor process Priors
#'
#' Plots the prior distribution of the number of clusters under a Dirichlet / Pitman-Yor process prior, for a sample of size \code{N} at given values of the concentration parameter \code{alpha} and optionally also the \code{discount} parameter. Useful for soliciting sensible priors for \code{alpha} or suitable fixed values for \code{alpha} or \code{discount} under the "\code{IMFA}" and "\code{IMIFA}" methods for \code{\link{mcmc_IMIFA}}, All arguments are vectorised. Requires use of the \code{Rmpfr} and \code{gmp} libraries. Density values are returned invisibly.
#' @param N The sample size.
#' @param alpha The concentration parameter. Must be specified and must be strictly greater than \code{-discount}.
#' @param discount The discount parameter for the Pitman-Yor process. Must lie in the interval [0, 1). Defaults to 0 (i.e. the Dirichlet process) as plotting with non-zero discount is not yet implement. However, users can still consult \code{\link{G_expected}} and \code{\link{G_variance}} in order to solicit sensible \code{discount} values.
#' @param show.plot Logical indicating whether the plot should be displayed (default = TRUE).
#' @param avg Logical indicating whether perpendicular lines should be dropped at the expected value, using \code{\link{G_expected}} with the supplied arguments.
#' @param col Colour of the plotted lines.
#' @param main Title of the plot.
#' @param ... Other optional arguments typically passed to \code{\link{plot}}.
#'
#' @return A plot of the prior distribution if \code{show.plot} is TRUE. Density values are returned invisibly.
#' @export
#' @importFrom grDevices "palette" "adjustcolor"
#' @importFrom Rfast "colsums" "colMaxs"
#' @importFrom viridis "viridis"
#' @seealso \code{\link{G_expected}}, \code{\link{G_variance}}, \code{\link[Rmpfr]{Rmpfr}}
#'
#' @examples
#' # library(Rmpfr)
#' # x <- G_prior(150, alpha=5)
#' # x
  G_prior      <- function(N, alpha, discount = 0L, show.plot = TRUE,
                           avg = FALSE, col = NULL, main = NULL, ...) {
    firstex    <- suppressMessages(requireNamespace("Rmpfr", quietly=TRUE))
    if(isTRUE(firstex)) {
      on.exit(.detach_pkg("Rmpfr"))
      on.exit(.detach_pkg("gmp"), add=TRUE)
    } else                            stop("'Rmpfr' package not installed")
    if(missing(col))    {
      col      <- viridis::viridis(length(alpha))
    } else {
      if(!all(.are_cols(col)))        stop("Supplied colour palette contains invalid colours")
      grDevices::palette(grDevices::adjustcolor(rep(col, 2)))
    }
    if(missing(main))   {
      main     <- paste0("Prior Distribution of G\nN=", N)
    } else if(!is.character(main))    stop("'main' title must be a character string")
    on.exit(grDevices::palette("default"), add=!isTRUE(firstex))
    on.exit(do.call("clip", as.list(graphics::par("usr"))), add=TRUE)
    if(any(c(length(N), length(show.plot),
             length(avg)) > 1))       stop("Arguments 'N', 'show.plot', and 'avg' must be strictly of length 1")
    if(!is.logical(show.plot))        stop("'show.plot' must be TRUE or FALSE")
    max.len    <- max(length(alpha),  length(discount))
    if(!is.element(length(alpha),
       c(1, max.len)))                stop("'alpha' must be of length 1 or length(discount)")
    if(!is.element(length(discount),
       c(1, max.len)))                stop("'discount' must be of length 1 or length(alpha)")
    if(!all(is.numeric(discount), is.numeric(alpha),
            is.numeric(N)))           stop("All inputs must be numeric")
    if(any(discount < 0,
       discount >= 1))                stop("'discount' must lie in the interval [0,1)")
    if(any(alpha < -discount))        stop("'alpha' must be strictly greater than -discount")
    if(!is.logical(avg))              stop("'avg' must be TRUE or FALSE")
    if(length(alpha)    != max.len) {
      alpha    <- rep(alpha,    max.len)
    }
    if(length(discount) != max.len) {
      discount <- rep(discount, max.len)
    }
    rx         <- matrix(0, nrow=N + 1, ncol=max.len)
    for(i in seq_len(max.len))      {
      tmp      <- if(discount[i] == 0) alpha[i]^seq_len(N)/Rmpfr::pochMpfr(alpha[i], N) else       {
                  exp(unlist(vapply(seq_len(N), function(k, Gs=seq_len(k - 1), x=0) { for(g in Gs) {
                    x <- x + log(alpha[i] + g * discount[i]) }; x}, numeric(1))) -
                         log(Rmpfr::pochMpfr(alpha[i] + 1, N - 1))) / discount[i]^seq_len(N)       }
      if(discount[i]  == 0) {
        rx[,i] <- c(0, abs(gmp::asNumeric(gmp::Stirling1.all(N) * tmp)))
      } else                          stop("Plotting with non-zero discount not yet implemented\nTry supplying the same arguments to G_expected() or G_variance()")
    }
    rx         <- scale(rx, center=FALSE, scale=Rfast::colsums(rx))
    max.rx     <- Rfast::colMaxs(rx, value=TRUE)
    if(isTRUE(show.plot))   {
      graphics::matplot(x=c(0, seq_len(N)), y=rx, type="l", col=col, xlab="Clusters",
                        ylab="Density", main=main, ...)
    }
    if(isTRUE(avg))         {
      exp.g    <- G_expected(N, alpha, discount)
      cat("\t");  cat(paste("E(G) = ", signif(exp.g, options()$digits), "\n"))
      if(isTRUE(show.plot)) {
        for(i in seq_len(max.len))  {
          clip(exp.g[i], exp.g[i], 0, max.rx[i])
          graphics::abline(v=exp.g[i], lty=2, col=i)
        }
      }
    }
      invisible(as.vector(rx))
  }
#
