#' @title Total number of features for HTRX
#' @description The total number of features in principle from haplotypes (i.e. interactions between SNPs)
#' generated by Haplotype Trend Regression with eXtra flexibility (HTRX) .
#' @name htrx_nfeatures
#' @param nsnp a positive integer giving the number of single nucleotide polymorphisms (SNPs) included in the haplotypes.
#' @param max_int a positive integer which specifies the maximum number of SNPs that can interact.
#' If no value is given (by default), interactions between all the SNPs will be considered.
#' @param htr logical. If \code{htr=TRUE}, the function returns the total number of
#' features for HTR. By default, \code{htr=FALSE}.
#'
#' @details The total number of features in principle is \ifelse{html}{\out{3<sup>nsnp</sup>}}{\eqn{2^nsnp}}-1
#' for haplotypes containing interactions between all different numbers of SNPs.
#' However, if \code{max_int < nsnp}, i.e. only the interactions between at most \code{max_int} SNPs are investigated,
#' there will be fewer total number of features.
#'
#' @return \code{htrx_nfeatures} returns a positive integer giving the total number
#' of features that each analysis includes.
#' @examples
#' ## the total number of haplotypes consisted of 6 SNPs
#' ## for at most 3-SNP interactions
#' htrx_nfeatures(nsnp=6,max_int=3)
NULL


#' @rdname htrx_nfeatures
#' @export
htrx_nfeatures <- function(nsnp,max_int=NULL,htr=FALSE){
  if(htr){
    return(2^nsnp)
  }else{
    if(is.null(max_int)){
      return(3^nsnp-1)
    }else{
      if(max_int>nsnp){
        max_int=nsnp
        warning('max_int is reduced to nsnp because max_int should not be bigger than nsnp')
      }
      return(Reduce(sum,lapply(1:max_int,function(s) choose(nsnp,s)*(2^s))))
    }
  }
}
