% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/eco.lsa.R
\name{eco.lsa}
\alias{eco.lsa}
\title{Local spatial analysis}
\usage{
eco.lsa(Z, con, method = c("G*", "G", "I", "C"), zerocon = NA, nsim = 99,
  conditional = c("auto", TRUE, FALSE), test = c("permutation",
  "bootstrap"), alternative = c("auto", "two.sided", "greater", "less"),
  adjust = "none")
}
\arguments{
\item{Z}{Vector for the analysis.}

\item{con}{An object of class eco.weight obtained with the function \code{\link{eco.weight}},
a "listw" object, or a matrix, containing the weights for the analysis. If a matrix, an attribute "xy" with the
projected coordinates is required.}

\item{method}{Method of analysis: "G" for Getis-Ord G, "G*" for Getis-Ord G*,
"I" for local Moran's I or "C" for local Geary's C.}

\item{zerocon}{If zerocon = 0 the program assigns the value 0 to those individuals
with no connections; if zerocon = NA the program assigns NA. Default is NA.}

\item{nsim}{Number of Monte-Carlo simulations.}

\item{conditional}{Logical. Should be used a
conditional randomization? (Anselin 1998, Sokal and Thomson 2006). The option "auto"
sets conditional = TRUE for LISA methods and G, as suggested by Sokal (2008).}

\item{test}{If test = "bootstrap", for each individual test,
the program generates a bootstrap
resampling and the associated confidence intervals of the null hypotesis.
If test = "permutation" (default) a permutation test is made and the P-value
is computed.}

\item{alternative}{The alternative hypothesis for "permutation" test.
 If "auto" is selected (default) the
program determines the alternative hypothesis in each individual test.
Other options are: "two.sided", "greater" and "less".}

\item{adjust}{Correction method of P-values for multiple tests,
passed to \code{\link[stats]{p.adjust}}. Defalut is "none" (no correction).}
}
\value{
The program returns an object of class "eco.lsa" with the following slots:

@OUT results

@METHOD method (coefficent) used in the analysis

@TEST test method used (bootstrap, permutation)

@NSIM number of simulations

@PADJUST P-values adjust method for permutation tests

@COND conditional randomization (logical)

@XY input coordinates
}
\description{
This program computes Getis-Ord G and G*, and LISA's (local
Moran and local Geary) statistics for
the data Z, with P-values or bootstrap confidence intervals.
}
\examples{
\dontrun{

data(eco.test)

# ---Getis-Ord's G*---

con<- eco.weight(eco$XY, method = "knearest",  k = 4, self = TRUE) # self = TRUE for G*
getis.ak <- eco.lsa(eco$P[, 1], con, method = "G*", nsim = 99, adjust = "none")
getis.ak

### to plot the results, the function "eco.lsa" calls "eco.rankplot"
(see ?eco.rankplot) when test = "permutation" and "eco.forestplot" (see ?eco.forestplot)
 when test = "bootstrap"
p <- plot(getis.ak)      # rankplot graph
p    #  points with colors of the color-scale:
     #  points with P < 0.05. Yellow points : points with P > 0.05
p <- plot(getis.ak, significant = FALSE)
p    # all points have a color of the color-scale

## bootstrap example
getis.akb <- eco.lsa(eco$P[, 1], con, method = "G*", nsim = 99, test = "bootstrap")
p <- plot(getis.akb)      # forestplot graph
p + ggplot2::theme_bw()   # the plot can be modified with ggplot2
                          # In this case, the background is modified  (white color)

#-------------------------------------------------------------------------------------#

# ---Getis-Ord's G---

con<- eco.weight(eco$XY, method = "knearest", k = 4)
# self = FALSE for G
getis <- eco.lsa(eco$P[, 1], con, method = "G", nsim = 99)
plot(getis)

#-------------------------------------------------------------------------------------#

# ---local Moran's I---

# **test for phenotypic data**

con <- eco.weight(eco$XY, method = "knearest",  k = 4, row.sd = TRUE)
# row standardized weights = TRUE

localmoran <- eco.lsa(eco$P[, 1], con,
method = "I", nsim = 99)                       # test for the first trait

plot(localmoran)

# ---------#

# **test for genotypic data** (concept also valid for the other statistics)

# eco$GENIND$tab is a matrix with the genetic data of "eco"
# as frequencies for each allele in each individual. Each allele
# can be analyzed as single traits.
# see "tab" in the genind help (?genind) of the adegenet package.

head(eco$GENIND$tab)      # head of the matrix

con<- eco.weight(eco$XY, method = "knearest",  k = 4, row.sd = TRUE)
# row standardized weights = TRUE

localmoran.geno <- eco.lsa(eco$GENIND$tab[, 2], con,
method = "I", nsim = 99)  # test for the second allele

plot(localmoran.geno)

#-------------------------------------------------------------------------------------#

# ---local Geary's C---

con<- eco.weight(eco$XY, method = "knearest",  k = 4, row.sd = TRUE)
# row standardized weights = TRUE
localgeary <- eco.lsa(eco$P[, 1], con, method = "C", nsim = 99, adjust = "none")
plot(localgeary)

}
}
\author{
Leandro Roser \email{leandroroser@ege.fcen.uba.ar}
}
\references{
Anselin L. 1995. Local indicators of spatial association-LISA.
Geographical analysis. 27: 93-115.

Getis A., and J. Ord. 1992. The analysis of spatial association by
use of distance statistics. Geographical analysis, 24: 189-206.

Ord J., and A. Getis. 1995. Local spatial autocorrelation statistics:
distributional issues and an application. Geographical analysis, 27: 286-306.

Sokal R., N. Oden and B. Thomson. 1998. Local spatial autocorrelation
in a biological model. Geographical Analysis, 30: 331-354.

Sokal R. and B. Thomson. 2006. Population structure inferred by local
spatial autocorrelation: an example from an Amerindian tribal population.
American journal of physical anthropology, 129: 121-131.
}

