% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Swrappers.R
\name{colapply.DTSg}
\alias{colapply.DTSg}
\alias{colapply}
\title{Apply function column-wise}
\usage{
\method{colapply}{DTSg}(
  x,
  fun,
  ...,
  cols = self$cols(class = "numeric")[1L],
  resultCols = NULL,
  suffix = NULL,
  helpers = TRUE,
  funby = NULL,
  ignoreDST = FALSE,
  multiplier = 1L,
  funbyHelpers = NULL,
  funbyApproach = self$funbyApproach,
  clone = getOption("DTSgClone")
)
}
\arguments{
\item{x}{A \code{\link{DTSg}} object (S3 method only).}

\item{fun}{A \code{\link{function}}. Its return value must be of length one.}

\item{...}{Further arguments passed on to \code{fun}.}

\item{cols}{A character vector specifying the columns to apply \code{fun} to.
Another possibility is a character string containing either comma separated
column names, for example, \code{"x,y,z"}, or the start and end column separated
by a colon, for example, \code{"x:z"}.}

\item{resultCols}{An optional character vector of the same length as \code{cols}
specifying the column names for the return values of \code{fun}. Another
possibility is a character string containing comma separated column names,
for example, \code{"x,y,z"}. Non-existing columns are added and existing columns
are overwritten. Columns are matched element-wise between \code{cols} and
\code{resultCols}.}

\item{suffix}{An optional character string. The return values of \code{fun} are
added as new columns with names consisting of the columns specified in
\code{cols} and this suffix. Existing columns are never overwritten. Only used
when \code{resultCols} is not specified.}

\item{helpers}{A logical specifying if helper data shall be handed over to
\code{fun}. See corresponding section for further information.}

\item{funby}{One of the temporal aggregation level functions described in
\code{\link{TALFs}} or a user defined temporal aggregation level function. Can be
used to apply functions like \code{\link{cumsum}} to a certain temporal level. See
corresponding section and examples for further information.}

\item{ignoreDST}{A logical specifying if day saving time shall be ignored
by \code{funby}. See corresponding section for further information.}

\item{multiplier}{A positive integerish value \dQuote{multiplying} the
temporal aggregation level of certain \code{\link{TALFs}}. See corresponding section
for further information.}

\item{funbyHelpers}{An optional \code{\link{list}} with helper data passed on to
\code{funby}. See corresponding section for further information.}

\item{funbyApproach}{A character string specifying the flavour of the applied
temporal aggregation level function. Either \code{"base"}, which utilises
\code{\link{as.POSIXct}}, or \code{"fasttime"}, which utilises \code{\link[fasttime:fastPOSIXct]{fasttime::fastPOSIXct}},
or \code{"RcppCCTZ"}, which utilises \code{\link[RcppCCTZ:parseDatetime]{RcppCCTZ::parseDatetime}} as the main
function for transforming timestamps.}

\item{clone}{A logical specifying if the object shall be modified in place or
if a deep clone (copy) shall be made beforehand.}
}
\value{
Returns a \code{\link{DTSg}} object.
}
\description{
Applies an arbitrary function to selected columns of a \code{\link{DTSg}} object.
}
\section{Helper data}{

In addition to the \code{...} argument, this method optionally hands over a
\code{\link{list}} argument with helper data called \code{.helpers} to \code{fun}. This \code{\link{list}}
contains the following elements:
\itemize{
\item \emph{.dateTime:} A \code{\link{POSIXct}} vector containing the \emph{.dateTime} column.
\item \emph{periodicity:} Same as the \code{\link[=DTSg]{periodicity}} field.
\item \emph{minLag:} A \code{\link{difftime}} object containing the minimum time difference
between two subsequent timestamps.
\item \emph{maxLag:} A \code{\link{difftime}} object containing the maximum time difference
between two subsequent timestamps.
}
}

\section{User defined TALFs, TALFs helper data and multiplier}{

User defined temporal aggregation level functions have to return a
\code{\link{POSIXct}} vector of the same length as the time series and accept two
arguments: a \code{\link{POSIXct}} vector as its first and a \code{\link{list}} with helper data
as its second. The default elements of this \code{\link{list}} are as follows:
\itemize{
\item \emph{timezone:} Same as the \code{\link[=DTSg]{timezone}} field.
\item \emph{ignoreDST:} Same as the \code{ignoreDST} argument.
\item \emph{periodicity:} Same as the \code{\link[=DTSg]{periodicity}} field.
\item \emph{na.status:} Same as the \code{\link[=DTSg]{na.status}} field.
\item \emph{multiplier:} Same as the \code{multiplier} argument.
\item \emph{funbyApproach:} Same as the \code{funbyApproach} argument.
}

Any additional element specified in the \code{funbyHelpers} argument is appended
to the end of the helper data \code{\link{list}}. In case \code{funbyHelpers} contains an
\emph{ignoreDST, multiplier} or \emph{funbyApproach} element, it takes precedence over
the respective method argument. \emph{timezone, periodicity} and \emph{na.status}
elements are rejected, as they are always taken directly from the object.

The temporal aggregation level of certain \code{\link{TALFs}} can be adjusted with the
help of the \code{multiplier} argument. A \code{multiplier} of \code{10}, for example, makes
\code{\link{byY_____}} aggregate to decades instead of years. Another example
is a \code{multiplier} of \code{6} provided to \code{\link{by_m____}}. The function
then aggregates all months of all first and all months of all second half
years instead of all months of all years separately. This feature is
supported by the following \code{\link{TALFs}} of the package:
\itemize{
\item \code{\link{byY_____}}
\item \code{\link{byYm____}}
\item \code{\link{byYmdH__}} (UTC and equivalent as well as all Etc/GMT time zones only)
\item \code{\link{byYmdHM_}}
\item \code{\link{byYmdHMS}}
\item \code{\link{by_m____}}
\item \code{\link{by___H__}} (UTC and equivalent as well as all Etc/GMT time zones only)
\item \code{\link{by____M_}}
\item \code{\link{by_____S}}
}
}

\section{Ignore day saving time}{

\code{ignoreDST} tells a temporal aggregation level function if it is supposed to
ignore day saving time while transforming the timestamps. This can be a
desired feature for time series strictly following the position of the sun
such as hydrological time series. Doing so ensures that diurnal variations
are preserved by all means and all intervals are of the \dQuote{correct}
length, however, a possible limitation might be that the day saving time
shift is invariably assumed to be one hour long. This feature requires that
the periodicity of the time series was recognised and is supported by the
following \code{\link{TALFs}} of the package:
\itemize{
\item \code{\link{byY_____}}
\item \code{\link{byYQ____}}
\item \code{\link{byYm____}}
\item \code{\link{byYmd___}}
\item \code{\link{by_Q____}}
\item \code{\link{by_m____}}
\item \code{\link{by___H__}}
}
}

\examples{
# new DTSg object
x <- DTSg$new(values = flow)

# linear interpolation of missing values
## R6 method
x$colapply(fun = interpolateLinear)$print()

## S3 method
print(colapply(x = x, fun = interpolateLinear))

# daily cumulative sums per month
## R6 method
x$colapply(
  fun = cumsum,
  helpers = FALSE,
  funby = byYm____
)$print()

## S3 method
print(colapply(
  x = x,
  fun = cumsum,
  helpers = FALSE,
  funby = byYm____
))

# calculate moving averages with the help of 'runner' (all four given
# approaches provide the same result with explicitly missing timestamps)
if (requireNamespace("runner", quietly = TRUE) &&
    packageVersion("runner") >= package_version("0.3.5")) {
  wrapper <- function(..., .helpers) {
    runner::runner(..., idx = .helpers[[".dateTime"]])
  }

  ## R6 method
  x$colapply(
    fun = runner::runner,
    f = mean,
    k = 5,
    lag = -2
  )$print()
  x$colapply(
    fun = wrapper,
    f = mean,
    k = "5 days",
    lag = "-2 days"
  )$print()
  x$colapply(
    fun = runner::runner,
    f = mean,
    k = "5 days",
    lag = "-2 days",
    idx = x$getCol(col = ".dateTime")
  )$print()
  x$colapply(
    fun = runner::runner,
    f = mean,
    k = "5 days",
    lag = "-2 days",
    idx = x[".dateTime"]
  )$print()

  ## S3 method
  print(colapply(
    x = x,
    fun = runner::runner,
    f = mean,
    k = 5,
    lag = -2
  ))
  print(colapply(
    x = x,
    fun = wrapper,
    f = mean,
    k = "5 days",
    lag = "-2 days"
  ))
  print(colapply(
    x = x,
    fun = runner::runner,
    f = mean,
    k = "5 days",
    lag = "-2 days",
    idx = getCol(x = x, col = ".dateTime")
  ))
  print(colapply(
    x = x,
    fun = runner::runner,
    f = mean,
    k = "5 days",
    lag = "-2 days",
    idx = x[".dateTime"]
  ))
}

# calculate rolling correlations somewhat inefficiently with the help of
# 'runner'
if (requireNamespace("runner", quietly = TRUE) &&
    packageVersion("runner") >= package_version("0.3.8")) {
  wrapper <- function(x, y, f, k, lag, ...) {
    runner::runner(
      cbind(x, y),
      f = function(x) {f(x[, 1], x[, 2])},
      k = k,
      lag = lag
    )
  }

  ## R6 method
  x$colapply(
    fun = wrapper,
    y = x["flow"] + rnorm(length(x["flow"])),
    f = cor,
    k = 5,
    lag = -2
  )$print()

  ## S3 method
  print(colapply(
    x = x,
    fun = wrapper,
    y = x["flow"] + rnorm(length(x["flow"])),
    f = cor,
    k = 5,
    lag = -2
  ))
}
}
\seealso{
\code{\link{cols}}, \code{\link{getOption}}
}
