% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sample_mallows.R
\name{sample_mallows}
\alias{sample_mallows}
\title{Random Samples from the Mallows Rank Model}
\usage{
sample_mallows(rho0, alpha0, n_samples, burnin = NULL, thinning = NULL,
  leap_size = 1, metric = "footrule", diagnostic = FALSE,
  items_to_plot = NULL, max_lag = 1000L)
}
\arguments{
\item{rho0}{Vector specifying the latent consensus ranking in the Mallows rank model.}

\item{alpha0}{Scalar specifying the scale parameter in the Mallows rank model.}

\item{n_samples}{Integer specifying the number of random samples to generate.}

\item{burnin}{Integer specifying the number of iterations to discard as burn-in.}

\item{thinning}{Integer specifying the number of MCMC iterations to perform
between each time a random rank vector is sampled.}

\item{leap_size}{Integer specifying the step size of the leap-and-shift proposal distribution.}

\item{metric}{Character string specifying the distance measure to use. Available
options are \code{"footrule"} (default), \code{"spearman"}, \code{"cayley"}, and
\code{"kendall"}. For sampling from the Mallows model with Cayley and Kendall distances
the \code{rmm} function in the  \code{PerMallows} package \insertCite{irurozki2016}{BayesMallows}
is typically faster.}

\item{diagnostic}{Logical specifying whether to output convergence diagnostics. If \code{TRUE},
a diagnostic plot is printed, together with the returned samples.}

\item{items_to_plot}{Integer vector used if \code{diagnostic = TRUE}, in order to
specify the items to plot in the diagnostic output. If not provided, 5 items are picked
at random.}

\item{max_lag}{Integer specifying the maximum lag to use in the computation of autocorrelation.
Defaults to 1000L. This argument is passed to \code{stats::acf}. Only used when
\code{diagnostic = TRUE}.}
}
\description{
Generate random samples from the Mallows
Rank Model \insertCite{mallows1957}{BayesMallows} with consensus ranking \eqn{\rho}
and scale parameter \eqn{\alpha}. The samples are obtained by running the
Metropolis-Hastings algorithm described in Appendix C of
\insertCite{vitelli2018;textual}{BayesMallows}.
}
\examples{
# Sample 100 random rankings from a Mallows distribution with footrule distance
set.seed(1)
# Number of items
n_items <- 15
# Set the consensus ranking
rho0 <- seq(from = 1, to = n_items, by = 1)
# Set the scale
alpha0 <- 10
# Number of samples
n_samples <- 100
# We first do a diagnostic run, to find the thinning and burnin to use
# We set n_samples to 1000, in order to run 1000 diagnostic iterations.
test <- sample_mallows(rho0 = rho0, alpha0 = alpha0, n_samples = 1000,
                       burnin = 1, thinning = 1, diagnostic = TRUE)
# When items_to_plot is not set, 5 items are picked at random. We can change this.
# We can also reduce the number of lags computed in the autocorrelation plots
test <- sample_mallows(rho0 = rho0, alpha0 = alpha0, n_samples = 1000,
                       burnin = 1, thinning = 1, diagnostic = TRUE,
                       items_to_plot = c(1:3, 10, 15), max_lag = 500)
# From the autocorrelation plot, it looks like we should use
# a thinning of at least 200. We set thinning = 1000 to be safe,
# since the algorithm in any case is fast. The Markov Chain
# seems to mix quickly, but we set the burnin to 1000 to be safe.
# We now run sample_mallows again, to get the 100 samples we want:
samples <- sample_mallows(rho0 = rho0, alpha0 = alpha0, n_samples = 100,
                          burnin = 1000, thinning = 1000)
# The samples matrix now contains 100 rows with rankings of 15 items.
# A good diagnostic, in order to confirm that burnin and thinning are set high
# enough, is to run compute_mallows on the samples
model_fit <- compute_mallows(samples, nmc = 10000)
# The highest posterior density interval covers alpha0 = 10.
compute_posterior_intervals(model_fit, burnin = 2000, parameter = "alpha")

# The PerMallows package has a Gibbs sampler for sampling from the Mallows
# distribution with Cayley, Kendall, Hamming, and Ulam distances. For these
# distances, using the PerMallows package is typically faster.

# Let us sample 100 rankings from the Mallows model with Cayley distance,
# with the same consensus ranking and scale parameter as above.
library(PerMallows)
# Set the scale parameter of the PerMallows package corresponding to
# alpha0 in BayesMallows
theta0 = alpha0 / n_items
# Sample with PerMallows::rmm
sample1 <- rmm(n = 100, sigma0 = rho0, theta = theta0, dist.name = "cayley")
# Generate the same sample with sample_mallows
sample2 <- sample_mallows(rho0 = rho0, alpha0 = alpha0, n_samples = 100,
                          burnin = 1000, thinning = 1000, metric = "cayley")

}
\references{
\insertAllCited{}
}
