% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/build_run_modify.r
\name{umxACE}
\alias{umxACE}
\title{Build and run a 2-group Cholesky twin model (uni-variate or multi-variate)}
\usage{
umxACE(name = "ACE", selDVs, selCovs = NULL, covMethod = c("fixed",
  "random"), dzData, mzData, sep = NULL, dzAr = 0.5, dzCr = 1,
  addStd = TRUE, addCI = TRUE, numObsDZ = NULL, numObsMZ = NULL,
  boundDiag = 0, weightVar = NULL, equateMeans = TRUE, bVector = FALSE,
  thresholds = c("deviationBased", "WLS"),
  autoRun = getOption("umx_auto_run"), suffix = "deprecated",
  optimizer = NULL, intervals = FALSE)
}
\arguments{
\item{name}{The name of the model (defaults to"ACE").}

\item{selDVs}{The variables to include from the data: preferably, just "dep" not c("dep_T1", "dep_T2").}

\item{selCovs}{(optional) covariates to include from the data (do not include suffix in names)}

\item{covMethod}{How to treat covariates: "fixed" (default) or "random".}

\item{dzData}{The DZ dataframe.}

\item{mzData}{The MZ dataframe.}

\item{sep}{The separator in twin variable names, often "_T", e.g. "dep_T1". Simplifies selDVs.}

\item{dzAr}{The DZ genetic correlation (defaults to .5, vary to examine assortative mating).}

\item{dzCr}{The DZ "C" correlation (defaults to 1: set to .25 to make an ADE model).}

\item{addStd}{Whether to add the algebras to compute a std model (defaults to TRUE).}

\item{addCI}{Whether to add intervals to compute CIs (defaults to TRUE).}

\item{numObsDZ}{= Number of DZ twins: Set this if you input covariance data.}

\item{numObsMZ}{= Number of MZ twins: Set this if you input covariance data.}

\item{boundDiag}{= Numeric lbound for diagonal of the a, c, and e matrices. Defaults to 0 since umx version 1.8}

\item{weightVar}{= If provided, a vector objective will be used to weight the data. (default = NULL).}

\item{equateMeans}{Whether to equate the means across twins (defaults to TRUE).}

\item{bVector}{Whether to compute row-wise likelihoods (defaults to FALSE).}

\item{thresholds}{How to implement ordinal thresholds c("deviationBased", "WLS").}

\item{autoRun}{Whether to mxRun the model (default TRUE: the estimated model will be returned).}

\item{suffix}{Allowed as a synonym for "suffix".}

\item{optimizer}{Optionally set the optimizer (default NULL does nothing).}

\item{intervals}{Whether to run mxCI confidence intervals (default = FALSE)}
}
\value{
- \code{\link{mxModel}} of subclass mxModel.ACE
}
\description{
Implementing a core task in twin modeling, umxACE models the genetic and environmental
structure of one or more phenotypes (measured variables) using the Cholesky ACE model
(Neale and Cardon, 1996).

Classical twin modeling uses the genetic and environmental differences 
among pairs of mono-zygotic (MZ) and di-zygotic (DZ) twins reared together.

The Cholesky decomposes this phenotypic variance into Additive genetic,
unique environmental (E) and, optionally, either common or shared-environment (C) or 
non-additive genetic effects (D). Scroll down to details for how to use the function, a figure
and multiple examples.

The Cholesky or lower-triangle decomposition allows a model which is both sure to be 
solvable, and also to account for all the variance (with some restrictions) in the data. 
This model creates as many latent A C and E variables as there are phenotypes, and, moving 
from left to right, decomposes the variance in each component into successively restricted 
factors. The following figure shows how the ACE model appears as a path diagram: See the details section below
for additional information on using umxACE.

\figure{ACE.png}
}
\details{
\strong{Data Input}
The function flexibly accepts raw data, and also summary covariance data 
(in which case the user must also supple numbers of observations for the two input data sets).

\strong{Ordinal Data}
In an important capability, the model transparently handles ordinal (binary or multi-level
ordered factor data) inputs, and can handle mixtures of continuous, binary, and ordinal
data in any combination. An experimental feature is under development to allow Tobit modeling. 

The function also supports weighting of individual data rows. In this case,
the model is estimated for each row individually, then each row likelihood
is multiplied by its weight, and these weighted likelihoods summed to form
the model-likelihood, which is to be minimized.
This feature is used in the non-linear GxE model functions.

\strong{Additional features}
The umxACE function supports varying the DZ genetic association (defaulting to .5)
to allow exploring assortative mating effects, as well as varying the DZ \dQuote{C} factor
from 1 (the default for modeling family-level effects shared 100% by twins in a pair),
to .25 to model dominance effects.

\strong{Matrices and Labels in ACE model}

Matrices 'a'', 'c', and 'e' contain the path loadings of the Cholesky ACE factor model.

So, labels relevant to modifying the model are of the form "a_r1c1", "c_r1c1" etc.

Variables are in rows, and factors are in columns. So to drop the influence of factor 2 on variable 3, you would say

   m2 = umxModify(m1, update = "c_r3c2")

Less commonly-modified matrices are the mean matrix `expMean`. This has 1 row, and the columns are laid out for each variable for twin 1, followed by each variable for twin 2.
So, in a model where the means for twin 1 and twin 2 had been equated (set = to T1), you could make them independent again with this script:

m1$top$expMean$labels[1,4:6] =  c("expMean_r1c4", "expMean_r1c5", "expMean_r1c6")

\emph{note}: Only one of C or D may be estimated simultaneously. This restriction reflects the lack
of degrees of freedom to simultaneously model C and D with only MZ and DZ twin pairs (Eaves et al. 1978 p267).
}
\examples{

# ============================
# = How heritable is height? =
# ============================
require(umx)
data(twinData) # ?twinData from Australian twins.
# Pick the variables
selDVs = c("ht1", "ht2")
mzData <- twinData[twinData$zygosity \%in\% "MZFF", ]
dzData <- twinData[twinData$zygosity \%in\% "DZFF", ]
m1 = umxACE(selDVs = selDVs, dzData = dzData, mzData = mzData) # -2ll= 9659
umxSummary(m1, std = FALSE) # unstandardized
# tip: with report = "html", umxSummary can print the table to your browser!
plot(m1)

# ========================================================
# = Evidence for dominance ? (DZ correlation set to .25) =
# ========================================================
m2 = umxACE("ADE", selDVs = selDVs, dzData = dzData, mzData = mzData, dzCr = .25)
umxCompare(m2, m1) # ADE is better
umxSummary(m2, comparison = m1) # nb: though this is ADE, matrices are still called A, C & E

# ==============================
# = Univariate model of weight =
# ==============================

# Things to note:

# 1. This variable has a large variance, but umx picks good starts.

# 2. umxACE can figure out variable names: provide sep= "_T" and selVar = "wt" -> "wt_T1" "wt_T2"

# 3. umxACE picks the variables it needs from the data.
# 4. note: the default boundDiag = 0 lower-bounds a, c, and e at 0 (prevents mirror-solutions).
        # can remove this by setting boundDiag = NULL
m1 = umxACE(selDVs = "wt", dzData = dzData, mzData = mzData, sep = "", boundDiag = NULL)

# MODEL MODIFICATION
# We can modify this model, say testing shared environment, and see a comparison:

m2 = umxModify(m1, update = "c_r1c1", comparison = TRUE)
# =====================================
# = Bivariate height and weight model =
# =====================================
data(twinData)
mzData <- twinData[twinData$zygosity \%in\% c("MZFF", "MZMM"),]
dzData <- twinData[twinData$zygosity \%in\% c("DZFF", "DZMM", "DZOS"), ]
mzData <- mzData[1:80,] # quicker run to keep CRAN happy
dzData <- dzData[1:80,]
selDVs = c("ht", "wt") # umx will add sep (in this case "") + "1" or '2'
m1 = umxACE(selDVs = selDVs, dzData = dzData, mzData = mzData, sep = '')
umxSummary(m1)

# =========================================================
# = Well done! Now you can make modify twin models in umx =
# =========================================================


# ===================
# = Ordinal example =
# ===================
require(umx)
data(twinData)
# Cut BMI column to form ordinal obesity variables
ordDVs = c("obese1", "obese2")
selDVs = c("obese")
obesityLevels = c('normal', 'overweight', 'obese')
cutPoints <- quantile(twinData[, "bmi1"], probs = c(.5, .2), na.rm = TRUE)
twinData$obese1 <- cut(twinData$bmi1, breaks = c(-Inf, cutPoints, Inf), labels = obesityLevels) 
twinData$obese2 <- cut(twinData$bmi2, breaks = c(-Inf, cutPoints, Inf), labels = obesityLevels) 
# Make the ordinal variables into mxFactors (ensure ordered is TRUE, and require levels)
twinData[, ordDVs] <- mxFactor(twinData[, ordDVs], levels = obesityLevels)
mzData <- twinData[twinData$zygosity \%in\% "MZFF", ]
dzData <- twinData[twinData$zygosity \%in\% "DZFF", ]
mzData <- mzData[1:80, ] # Just top 80 pairs to run fast
dzData <- dzData[1:80, ]
str(mzData) # make sure mz, dz, and t1 and t2 have the same levels!
m1 = umxACE(selDVs = selDVs, dzData = dzData, mzData = mzData, suffix = '')
umxSummary(m1)

# ============================================
# = Bivariate continuous and ordinal example =
# ============================================
data(twinData)
selDVs = c("wt", "obese")
# Cut BMI column to form ordinal obesity variables
ordDVs = c("obese1", "obese2")
obesityLevels = c('normal', 'overweight', 'obese')
cutPoints <- quantile(twinData[, "bmi1"], probs = c(.5, .2), na.rm = TRUE)
twinData$obese1 <- cut(twinData$bmi1, breaks = c(-Inf, cutPoints, Inf), labels = obesityLevels) 
twinData$obese2 <- cut(twinData$bmi2, breaks = c(-Inf, cutPoints, Inf), labels = obesityLevels) 
# Make the ordinal variables into mxFactors (ensure ordered is TRUE, and require levels)
twinData[, ordDVs] <- mxFactor(twinData[, ordDVs], levels = obesityLevels)
# umxACE can trim out unused variables on its own: just select the rows you want.
mzData <- twinData[twinData$zygosity \%in\%  "MZFF",] 
dzData <- twinData[twinData$zygosity \%in\%  "DZFF",]
mzData <- mzData[1:80,] # just top 80 so example runs in a couple of secs
dzData <- dzData[1:80,]
m1 = umxACE(selDVs = selDVs, dzData = dzData, mzData = mzData, suffix = '')
plot(m1)

# =======================================
# = Mixed continuous and binary example =
# =======================================
require(umx)
data(twinData)
# Cut to form category of 20\% obese subjects
# and make into mxFactors (ensure ordered is TRUE, and require levels)
cutPoints <- quantile(twinData[, "bmi1"], probs = .2, na.rm = TRUE)
obesityLevels = c('normal', 'obese')
twinData$obese1 <- cut(twinData$bmi1, breaks = c(-Inf, cutPoints, Inf), labels = obesityLevels) 
twinData$obese2 <- cut(twinData$bmi2, breaks = c(-Inf, cutPoints, Inf), labels = obesityLevels) 
ordDVs = c("obese1", "obese2")
twinData[, ordDVs] <- mxFactor(twinData[, ordDVs], levels = obesityLevels)

selDVs = c("wt", "obese")
mzData <- twinData[twinData$zygosity \%in\% "MZFF",]
dzData <- twinData[twinData$zygosity \%in\% "DZFF",]
\dontrun{
m1 = umxACE(selDVs = selDVs, dzData = dzData, mzData = mzData, suffix = '')
umxSummary(m1)
}

# ===================================
# Example with covariance data only =
# ===================================

require(umx)
data(twinData)
selDVs = c("wt1", "wt2")
mz = cov(twinData[twinData$zygosity \%in\%  "MZFF", selDVs], use = "complete")
dz = cov(twinData[twinData$zygosity \%in\%  "DZFF", selDVs], use = "complete")
m1 = umxACE(selDVs = selDVs, dzData = dz, mzData = mz, numObsDZ=569, numObsMZ=351)
umxSummary(m1)
plot(m1)
}
\references{
- Eaves, L. J., Last, K. A., Young, P. A., & Martin, N. G. (1978). Model-fitting approaches 
to the analysis of human behaviour. Heredity, 41(3), 249-320. \url{https://www.nature.com/articles/hdy1978101.pdf}
}
\seealso{
- \code{\link{plot.MxModel.ACE}}, \code{\link{plot.MxModel.ACE}}, \code{\link{umxSummaryACE}}, \code{\link{umxModify}}

Other Twin Modeling Functions: \code{\link{umxACE_cov_fixed}},
  \code{\link{umxACEcov}}, \code{\link{umxACEv}},
  \code{\link{umxCP}}, \code{\link{umxGxE_biv}},
  \code{\link{umxGxE_window}}, \code{\link{umxGxE}},
  \code{\link{umxIP}}, \code{\link{umxSexLim}},
  \code{\link{umxSummaryACEcov}},
  \code{\link{umxSummaryACEv}},
  \code{\link{umxSummaryACE}}, \code{\link{umxSummaryCP}},
  \code{\link{umxSummaryGxE_biv}},
  \code{\link{umxSummaryGxE}}, \code{\link{umxSummaryIP}},
  \code{\link{umxSummarySexLim}}, \code{\link{umx}},
  \code{\link{xmu_twin_check}}
}
