#' Removes XML/HTML Tags and Umlauts
#'
#' Removes XML tags (removeXML), remove or resolve HTML tags (removeHTML) and changes german umlauts in a standardized form (removeUmlauts).
#'
#'
#' @param x Character: Vector or list of character vectors.
#' @details The decision which u.type is used should consider the language of the corpus, because in some languages the replacement of umlauts can change the meaning of a word.
#' To change which columns are used by removeXML use argument xmlAction in \code{\link{readTextmeta}}.
#' @return Adjusted character string or list, depending on input.
#' @keywords manip
#' @examples
#' xml <- "<text>Some <b>important</b> text</text>"
#' removeXML(xml)
#'
#' x <- "&#x00f8; &#248; &oslash;"
#' removeHTML(x=x, symbolList = 1, dec=TRUE, hex=FALSE, entity=FALSE, delete = FALSE)
#' removeHTML(x=x, symbolList = c(1,3))
#'
#' y <- c("Bl\UFChende Apfelb\UE4ume")
#' removeUmlauts(y)
#'
#' @export removeXML

removeXML <- function(x){
    inputlist <- FALSE
    if(is.list(x)){listlength <- lengths(x)
                   listid <- names(x)
                   x <- unlist(x)
                   inputlist <- TRUE}
    x <- gsub(pattern="<[^><]*>", replacement=" ", x, perl=TRUE)
    x <- trimws(x)
    if(inputlist){x <- split(unname(x), f=rep(listid, listlength))}
    return(x)
}

#' @rdname removeXML
#' @export

removeUmlauts <- function(x){
    inputlist <- FALSE
    if(is.list(x)){listlength <- lengths(x)
                   listid <- names(x)
                   x <- unlist(x)
                   inputlist <- TRUE}
    x <- gsub(pattern="\U00E4", replacement="ae",x)
    x <- gsub(pattern="\U00C4", replacement="Ae",x)
    x <- gsub(pattern="\U00F6", replacement="oe",x)
    x <- gsub(pattern="\U00D6", replacement="Oe",x)
    x <- gsub(pattern="\U00FC", replacement="ue",x)
    x <- gsub(pattern="\U00DC", replacement="Ue",x)
    x <- gsub(pattern="\U00DF", replacement="ss",x)
    if(inputlist){x <- split(unname(x), f=rep(listid, listlength))}
    return(x)
}

#' @rdname removeXML
#' @param dec Logical: If \code{TRUE} HTML-entities in decimal-style would be resolved.
#' @param hex Logical: If \code{TRUE} HTML-entities in hexadecimal-style would be resolved.
#' @param entity Logical: If \code{TRUE} HTML-entities in text-style would be resolved.
#' @param symbolList numeric vector to chhose from the 16 ISO-8859 Lists (ISO-8859 12 did not exists and is empty).
#' @param delete Logical: If \code{TRUE} all not resolved HTML-entities would bei deleted?
#' @param symbols Logical: If \code{TRUE} most symbols from ISO-8859 would be not resolved (DEC: 32:64, 91:96, 123:126, 160:191, 215, 247, 818, 8194:8222, 8254, 8291, 8364, 8417, 8470).
#' @export

removeHTML <- function(x, dec=TRUE, hex=TRUE, entity=TRUE, symbolList=c(1:4,9,13,15,16), delete=TRUE, symbols=FALSE){
    entityList <- sort(unique(as.vector(ISO8859()[,symbolList])))

    if(!symbols){ # remove symbols from entityList
        symbolList <- toupper(paste0(as.hexmode(c(32:64,91:96,123:126,160:191,215,247, 818, 8194:8222, 8254, 8291, 8364, 8417, 8470))))
        symbolList <- gsub(pattern="^0*", replacement="", symbolList)
        entityList <- entityList[!entityList %in% symbolList]
    }

    # if list
    inputlist <- FALSE
    if(is.list(x)){listlength <- lengths(x)
                   listid <- names(x)
                   x <- unlist(x)
                   inputlist <- TRUE}

    if(dec){cat("Decimal html \n")
            for(i in entityList){
                cat(eval(parse(text = paste0("'\\u", i, "'"))))
                x <- gsub(pattern=paste0("&#[0]*",strtoi(i, base=16L),";"), replacement=eval(parse(text = paste0("'\\u", i, "'"))),x, useBytes=TRUE)}
                    cat("\n")
                }

    if(hex){cat("Hex html \n")
            for(i in entityList){
                cat(eval(parse(text = paste0("'\\u", i, "'"))))
                x <- gsub(pattern=paste0("&#x[0]*",i,";"), replacement=eval(parse(text = paste0("'\\u", i, "'"))),x, ignore.case = TRUE, useBytes=TRUE)}
                    cat("\n")
                }

    if(entity){cat("Entity html \n")
               entities <- namedEntity()
               matchedEntities <- entities[match(entityList, entities[,2]),]
               matchedEntities <- matchedEntities[!is.na(matchedEntities[,1]),]
               for(i in 1:nrow(matchedEntities)){
                   cat(eval(parse(text = paste0("'\\u", matchedEntities[i,2], "'"))))
                   x <- gsub(pattern=matchedEntities[i,1], replacement=eval(parse(text = paste0("'\\u", matchedEntities[i,2], "'"))),x, useBytes=TRUE)}
               cat("\n")
           }

    if(delete){
        cat("delete remaining entities\n")
        x <- gsub(pattern="&[^;]*;", replacement="", x)}
    x <- trimws(x)
    Encoding(x) <- "UTF-8"
    if(inputlist){x <- split(unname(x), f=rep(listid, listlength))}
    return(x)
}

ISO8859 <- function(){
ISO <- matrix(c(
"A0","A0","A0","A0","A0","A0","A0","A0","A0","A0","A0","","A0","A0","A0","A0",
"A1","104","126","104","401","A0","2018","A0","A1","104","E01","","201D","1E02","A1","104",
"A2","2D8","2D8","183","402","A0","2019","A2","A2","112","E02","","A2","1E03","A2","105",
"A3","141","A3","156","403","A0","A3","A3","A3","122","E03","","A3","A3","A3","141",
"A4","A4","A4","A4","404","A0","A0","A4","A4","12A","E04","","A4","10A","20AC","20AC",
"A5","13D","A0","128","405","A0","A0","A5","A5","128","E05","","20E1","10B","A5","201E",
"A6","15A","124","13B","406","A0","A6","A6","A6","136","E06","","A6","1E0A","160","160",
"A7","A7","A7","A7","407","A0","A7","A7","A7","A7","E07","","A7","A7","A7","A7",
"A8","A8","A8","A8","408","A0","A8","A8","A8","13B","E07","","D8","1E80","161","161",
"A9","160","130","160","4A9","A0","A9","A9","A9","110","E09","","A9","A9","A9","A9",
"AA","15E","15E","112","40A","A0","A0","D7","AA","160","E0A","","156","1E82","AA","218",
"AB","164","11E","122","40B","A0","AB","AB","AB","166","E0B","","AB","1E0B","AB","AB",
"AC","179","134","166","40C","60C","AC","AC","AC","17D","E0C","","AC","1EF2","AC","179",
"AD","AD","AD","AD","AD","AD","AD","AD","AD","AD","E0D","","AD","AD","AD","AD",
"AE","17D","A0","17D","40E","A0","A0","AE","AE","16A","E0E","","AE","AE","AE","17A",
"AF","17B","17B","AF","40F","A0","2015","203E","AF","145","E0F","","C6","178","AF","17B",
"B0","B0","B0","B0","410","A0","410","B0","B0","B0","E10","","B0","1E1E","B0","B0",
"B1","105","127","105","411","A0","B1","B1","B1","105","E11","","B1","1E1F","B1","B1",
"B2","2DB","B2","2DB","412","A0","B2","B2","B2","113","E12","","B2","120","B2","10C",
"B3","142","B3","157","413","A0","B3","B3","B3","123","E13","","B3","121","B3","142",
"B4","B4","B4","B4","414","A0","384","B4","B4","12B","E14","","201C","1E40","17D","17D",
"B5","13E","B5","129","415","A0","385","A0","B5","129","E15","","B5","1E41","B5","201D",
"B6","15B","125","13C","416","A0","386","B6","B6","137","E16","","B6","B6","B6","B6",
"B7","2C7","B7","2C7","417","A0","B7","B7","B7","B7","E17","","B7","1E56","B7","B7",
"B8","B8","B8","B8","418","A0","338","B8","B8","13C","E18","","F8","1E81","17E","17E",
"B9","161","131","161","419","A0","339","B9","B9","111","E19","","B9","1E57","B9","10D",
"BA","15F","15F","113","41A","A0","33A","F7","BA","161","E1A","","157","1E83","BA","219",
"BB","165","11F","123","41B","61B","33B","BB","BB","167","E1B","","BB","1E60","BB","BB",
"BC","165","135","167","41C","A0","33C","BC","BC","17E","E1C","","BC","1EF3","152","152",
"BD","2DD","BD","14A","41D","A0","BD","BD","BD","2015","E1D","","BD","1E84","153","153",
"BE","17E","A0","17E","41E","A0","33E","BE","BE","16B","E1E","","BE","1E85","178","178",
"BF","17C","17C","14B","41F","61F","33F","A0","BF","14B","E1F","","E6","1E61","BF","17C",
"C0","154","C0","100","420","A0","390","A0","C0","100","E20","","104","C0","C0","C0",
"C1","C1","C1","C1","421","621","391","A0","C1","C1","E21","","12E","C1","C1","C1",
"C2","C2","C2","C2","422","622","392","A0","C2","C2","E22","","100","C2","C2","C2",
"C3","102","A0","C3","423","623","393","A0","C3","C3","E23","","106","C3","C3","102",
"C4","C4","C4","C4","424","624","394","A0","C4","C4","E24","","C4","C4","C4","C4",
"C5","139","10A","C5","425","625","395","A0","C5","C5","E25","","C5","C5","C5","106",
"C6","106","108","C6","426","626","396","A0","C6","C6","E26","","118","C6","C6","C6",
"C7","C7","C7","12E","427","627","397","A0","C7","12E","E27","","112","C7","C7","C7",
"C8","10C","C8","10C","428","628","398","A0","C8","10C","E28","","10C","C8","C8","C8",
"C9","C9","C9","C9","429","629","399","A0","C9","C9","E29","","C9","C9","C9","C9",
"CA","118","CA","118","42A","62A","39A","A0","CA","118","E2A","","179","CA","CA","CA",
"CB","CB","CB","CB","42B","62B","39B","A0","CB","CB","E2B","","116","CB","CB","CB",
"CC","11A","CC","116","42C","62C","39C","A0","CC","116","E2C","","122","CC","CC","CC",
"CD","CD","CD","CD","42D","62D","39D","A0","CD","CD","E2D","","136","CD","CD","CD",
"CE","CE","CE","CE","42E","62E","39E","A0","CE","CE","E2E","","12A","CE","CE","CE",
"CF","10E","CF","12A","42F","62F","39F","A0","CF","CF","E2F","","13B","CF","CF","CF",
"D0","D0","D0","D0","430","630","3A0","A0","11E","D0","E30","","160","174","D0","D0",
"D1","143","D1","145","431","631","3A1","A0","D1","145","E31","","143","D1","D1","143",
"D2","147","D2","14C","432","632","A0","A0","D2","14C","E32","","145","D2","D2","D2",
"D3","D3","D3","136","433","633","3A3","A0","D3","D3","E33","","D3","D3","D3","D3",
"D4","D4","D4","D4","434","634","3A4","A0","D4","D4","E34","","14C","D4","D4","D4",
"D5","150","120","D5","435","635","3A5","A0","D5","D5","E35","","D5","D5","D5","150",
"D6","D6","D6","D6","436","636","3A6","A0","D6","D6","E36","","D6","D6","D6","D6",
"D7","D7","D7","D7","437","637","3A7","A0","D7","168","E37","","D7","1E6A","D7","15A",
"D8","158","11C","D8","438","638","3A8","A0","D8","D8","E38","","172","D8","D8","170",
"D9","16E","D9","172","439","639","3A9","A0","D9","172","E39","","141","D9","D9","D9",
"DA","DA","DA","DA","43A","63A","3AA","A0","DA","DA","E3A","","15A","DA","DA","DA",
"DB","170","D8","DB","34B","A0","3AB","A0","DB","DB","A0","","16A","DB","DB","DB",
"DC","DC","DC","DC","34C","A0","3AC","A0","DC","DC","A0","","DC","DC","DC","DC",
"DD","DD","16C","168","34D","A0","3AD","A0","130","DD","A0","","17B","DD","DD","118",
"DE","162","15C","16A","34E","A0","3AE","A0","15E","DE","A0","","17D","176","DE","21A",
"DF","DF","DF","DF","34F","A0","3AF","2017","DF","DF","E3F","","DF","DF","DF","DF",
"E0","155","E0","101","440","640","3B0","5D0","E0","101","E40","","105","E0","E0","E0",
"E1","E1","E1","E1","441","641","3B1","5D1","E1","E1","E41","","12F","E1","E1","E1",
"E2","E2","E2","E2","442","642","3B2","5D2","E2","E2","E42","","101","E2","E2","E2",
"E3","103","A0","E3","443","643","3B3","5D3","E3","E3","E43","","107","E3","E3","103",
"E4","E4","E4","E4","444","644","3B4","5D4","E4","E4","E44","","E4","E4","E4","E4",
"E5","13A","10B","E5","445","645","3B5","5D5","E5","E5","E45","","E5","E5","E5","107",
"E6","107","109","E6","446","646","3B6","5D6","E6","E6","E46","","119","E6","E6","E6",
"E7","E7","E7","12F","447","647","3B7","5D7","E7","12F","E47","","113","E7","E7","E7",
"E8","10D","10D","10D","448","648","3B8","5D8","E8","10D","E48","","10D","E8","E8","E8",
"E9","E9","E9","E9","449","649","3B9","5D9","E9","E9","E49","","E9","E9","E9","E9",
"EA","119","EA","119","44A","64A","3BA","5DA","EA","119","E4A","","17A","EA","EA","EA",
"EB","EB","EB","EB","44B","64B","3BB","5DB","EB","EB","E4B","","117","EB","EB","EB",
"EC","11B","EC","117","44C","64C","3BC","5DC","EC","117","E4C","","123","EC","EC","EC",
"ED","ED","ED","ED","44D","64D","3BD","5DD","ED","ED","E4D","","137","ED","ED","ED",
"EE","EE","EE","EE","44E","64E","3BE","5DE","EE","EE","E4E","","12B","EE","EE","EE",
"EF","10F","EF","EF","44F","64F","3BF","5DF","EF","EF","E4F","","13C","EF","EF","EF",
"F0","111","A0","111","2116","640","3C0","5E0","11F","F0","E50","","161","175","F0","111",
"F1","144","F1","146","451","641","3C1","5E1","F1","146","E51","","144","F1","F1","144",
"F2","148","F2","14D","452","642","3C2","5E2","F2","14D","E52","","146","F2","F2","F2",
"F3","F3","F3","137","453","A0","3C3","5E3","F3","F3","E53","","F3","F3","F3","F3",
"F4","F4","F4","F4","454","A0","3C4","5E4","F4","F4","E54","","14D","F4","F4","F4",
"F5","151","121","F5","455","A0","3C5","5E5","F5","F5","E55","","F5","F5","F5","151",
"F6","F6","F6","F6","456","A0","3C6","5E6","F6","F6","E56","","F6","F6","F6","F6",
"F7","F7","F7","F7","457","A0","3C7","5E7","F7","169","E57","","F7","1E6B","F7","15B",
"F8","159","11D","F8","458","A0","3C8","5E8","F8","F8","E58","","173","F8","F8","171",
"F9","16F","F9","173","459","A0","3C9","5E9","F9","173","E59","","142","F9","F9","F9",
"FA","FA","FA","FA","45A","A0","3CA","5EA","FA","FA","E5A","","15B","FA","FA","FA",
"FB","171","FB","FB","45B","A0","3CB","A0","FB","FB","E5B","","16B","FB","FB","FB",
"FC","FC","FC","FC","45C","A0","3CC","A0","FC","FC","A0","","FC","FC","FC","FC",
"FD","FD","16D","169","A7","A0","3CD","A0","131","FD","A0","","17C","FD","FD","119",
"FE","163","15D","16B","45E","A0","3CE","A0","15F","FE","A0","","17E","177","FE","21B",
"FF","2D9","2D9","2D9","45F","A0","A0","A0","FF","138","A0","","2019","FF","FF","FF"
), byrow=TRUE, ncol=16)
return(ISO)
}

namedEntity <- function(){
  nE <- matrix(c(
"&Aacute;", "C1",
"&Aacute", "C1",
"&aacute;", "E1",
"&aacute", "E1",
"&Abreve;", "102",
"&abreve;", "103",
"&ac;", "223E",
"&acd;", "223F",
"&acE;", "223E 333",
"&Acirc;", "C2",
"&Acirc", "C2",
"&acirc;", "E2",
"&acirc", "E2",
"&acute;", "B4",
"&acute", "B4",
"&Acy;", "410",
"&acy;", "430",
"&AElig;", "C6",
"&AElig", "C6",
"&aelig;", "E6",
"&aelig", "E6",
"&af;", "2061",
"&Afr;", "D835 DD04",
"&afr;", "D835 DD1E",
"&Agrave;", "C0",
"&Agrave", "C0",
"&agrave;", "E0",
"&agrave", "E0",
"&alefsym;", "2135",
"&aleph;", "2135",
"&Alpha;", "391",
"&alpha;", "3B1",
"&Amacr;", "100",
"&amacr;", "101",
"&amalg;", "2A3F",
"&amp;", "26",
"&amp", "26",
"&AMP;", "26",
"&AMP", "26",
"&andand;", "2A55",
"&And;", "2A53",
"&and;", "2227",
"&andd;", "2A5C",
"&andslope;", "2A58",
"&andv;", "2A5A",
"&ang;", "2220",
"&ange;", "29A4",
"&angle;", "2220",
"&angmsdaa;", "29A8",
"&angmsdab;", "29A9",
"&angmsdac;", "29AA",
"&angmsdad;", "29AB",
"&angmsdae;", "29AC",
"&angmsdaf;", "29AD",
"&angmsdag;", "29AE",
"&angmsdah;", "29AF",
"&angmsd;", "2221",
"&angrt;", "221F",
"&angrtvb;", "22BE",
"&angrtvbd;", "299D",
"&angsph;", "2222",
"&angst;", "C5",
"&angzarr;", "237C",
"&Aogon;", "104",
"&aogon;", "105",
"&Aopf;", "D835 DD38",
"&aopf;", "D835 DD52",
"&apacir;", "2A6F",
"&ap;", "2248",
"&apE;", "2A70",
"&ape;", "224A",
"&apid;", "224B",
"&apos;", "27",
"&ApplyFunction;", "2061",
"&approx;", "2248",
"&approxeq;", "224A",
"&Aring;", "C5",
"&Aring", "C5",
"&aring;", "E5",
"&aring", "E5",
"&Ascr;", "D835 DC9C",
"&ascr;", "D835 DCB6",
"&Assign;", "2254",
"&ast;", "2A",
"&asymp;", "2248",
"&asympeq;", "224D",
"&Atilde;", "C3",
"&Atilde", "C3",
"&atilde;", "E3",
"&atilde", "E3",
"&Auml;", "C4",
"&Auml", "C4",
"&auml;", "E4",
"&auml", "E4",
"&awconint;", "2233",
"&awint;", "2A11",
"&backcong;", "224C",
"&backepsilon;", "3F6",
"&backprime;", "2035",
"&backsim;", "223D",
"&backsimeq;", "22CD",
"&Backslash;", "2216",
"&Barv;", "2AE7",
"&barvee;", "22BD",
"&barwed;", "2305",
"&Barwed;", "2306",
"&barwedge;", "2305",
"&bbrk;", "23B5",
"&bbrktbrk;", "23B6",
"&bcong;", "224C",
"&Bcy;", "411",
"&bcy;", "431",
"&bdquo;", "201E",
"&becaus;", "2235",
"&because;", "2235",
"&Because;", "2235",
"&bemptyv;", "29B0",
"&bepsi;", "3F6",
"&bernou;", "212C",
"&Bernoullis;", "212C",
"&Beta;", "392",
"&beta;", "3B2",
"&beth;", "2136",
"&between;", "226C",
"&Bfr;", "D835 DD05",
"&bfr;", "D835 DD1F",
"&bigcap;", "22C2",
"&bigcirc;", "25EF",
"&bigcup;", "22C3",
"&bigodot;", "2A00",
"&bigoplus;", "2A01",
"&bigotimes;", "2A02",
"&bigsqcup;", "2A06",
"&bigstar;", "2605",
"&bigtriangledown;", "25BD",
"&bigtriangleup;", "25B3",
"&biguplus;", "2A04",
"&bigvee;", "22C1",
"&bigwedge;", "22C0",
"&bkarow;", "290D",
"&blacklozenge;", "29EB",
"&blacksquare;", "25AA",
"&blacktriangle;", "25B4",
"&blacktriangledown;", "25BE",
"&blacktriangleleft;", "25C2",
"&blacktriangleright;", "25B8",
"&blank;", "2423",
"&blk12;", "2592",
"&blk14;", "2591",
"&blk34;", "2593",
"&block;", "2588",
"&bne;", "3D 20E5",
"&bnequiv;", "2261 20E5",
"&bNot;", "2AED",
"&bnot;", "2310",
"&Bopf;", "D835 DD39",
"&bopf;", "D835 DD53",
"&bot;", "22A5",
"&bottom;", "22A5",
"&bowtie;", "22C8",
"&boxbox;", "29C9",
"&boxdl;", "2510",
"&boxdL;", "2555",
"&boxDl;", "2556",
"&boxDL;", "2557",
"&boxdr;", "250C",
"&boxdR;", "2552",
"&boxDr;", "2553",
"&boxDR;", "2554",
"&boxh;", "2500",
"&boxH;", "2550",
"&boxhd;", "252C",
"&boxHd;", "2564",
"&boxhD;", "2565",
"&boxHD;", "2566",
"&boxhu;", "2534",
"&boxHu;", "2567",
"&boxhU;", "2568",
"&boxHU;", "2569",
"&boxminus;", "229F",
"&boxplus;", "229E",
"&boxtimes;", "22A0",
"&boxul;", "2518",
"&boxuL;", "255B",
"&boxUl;", "255C",
"&boxUL;", "255D",
"&boxur;", "2514",
"&boxuR;", "2558",
"&boxUr;", "2559",
"&boxUR;", "255A",
"&boxv;", "2502",
"&boxV;", "2551",
"&boxvh;", "253C",
"&boxvH;", "256A",
"&boxVh;", "256B",
"&boxVH;", "256C",
"&boxvl;", "2524",
"&boxvL;", "2561",
"&boxVl;", "2562",
"&boxVL;", "2563",
"&boxvr;", "251C",
"&boxvR;", "255E",
"&boxVr;", "255F",
"&boxVR;", "2560",
"&bprime;", "2035",
"&breve;", "2D8",
"&Breve;", "2D8",
"&brvbar;", "A6",
"&brvbar", "A6",
"&bscr;", "D835 DCB7",
"&Bscr;", "212C",
"&bsemi;", "204F",
"&bsim;", "223D",
"&bsime;", "22CD",
"&bsolb;", "29C5",
"&bsol;", "5C",
"&bsolhsub;", "27C8",
"&bull;", "2022",
"&bullet;", "2022",
"&bump;", "224E",
"&bumpE;", "2AAE",
"&bumpe;", "224F",
"&Bumpeq;", "224E",
"&bumpeq;", "224F",
"&Cacute;", "106",
"&cacute;", "107",
"&capand;", "2A44",
"&capbrcup;", "2A49",
"&capcap;", "2A4B",
"&cap;", "2229",
"&Cap;", "22D2",
"&capcup;", "2A47",
"&capdot;", "2A40",
"&CapitalDifferentialD;", "2145",
"&caps;", "2229 FE00",
"&caret;", "2041",
"&caron;", "2C7",
"&Cayleys;", "212D",
"&ccaps;", "2A4D",
"&Ccaron;", "10C",
"&ccaron;", "10D",
"&Ccedil;", "C7",
"&Ccedil", "C7",
"&ccedil;", "E7",
"&ccedil", "E7",
"&Ccirc;", "108",
"&ccirc;", "109",
"&Cconint;", "2230",
"&ccups;", "2A4C",
"&ccupssm;", "2A50",
"&Cdot;", "10A",
"&cdot;", "10B",
"&cedil;", "B8",
"&cedil", "B8",
"&Cedilla;", "B8",
"&cemptyv;", "29B2",
"&cent;", "A2",
"&cent", "A2",
"&centerdot;", "B7",
"&CenterDot;", "B7",
"&cfr;", "D835 DD20",
"&Cfr;", "212D",
"&CHcy;", "427",
"&chcy;", "447",
"&check;", "2713",
"&checkmark;", "2713",
"&Chi;", "3A7",
"&chi;", "3C7",
"&circ;", "2C6",
"&circeq;", "2257",
"&circlearrowleft;", "21BA",
"&circlearrowright;", "21BB",
"&circledast;", "229B",
"&circledcirc;", "229A",
"&circleddash;", "229D",
"&CircleDot;", "2299",
"&circledR;", "AE",
"&circledS;", "24C8",
"&CircleMinus;", "2296",
"&CirclePlus;", "2295",
"&CircleTimes;", "2297",
"&cir;", "25CB",
"&cirE;", "29C3",
"&cire;", "2257",
"&cirfnint;", "2A10",
"&cirmid;", "2AEF",
"&cirscir;", "29C2",
"&ClockwiseContourIntegral;", "2232",
"&CloseCurlyDoubleQuote;", "201D",
"&CloseCurlyQuote;", "2019",
"&clubs;", "2663",
"&clubsuit;", "2663",
"&colon;", "3A",
"&Colon;", "2237",
"&Colone;", "2A74",
"&colone;", "2254",
"&coloneq;", "2254",
"&comma;", "2C",
"&commat;", "40",
"&comp;", "2201",
"&compfn;", "2218",
"&complement;", "2201",
"&complexes;", "2102",
"&cong;", "2245",
"&congdot;", "2A6D",
"&Congruent;", "2261",
"&conint;", "222E",
"&Conint;", "222F",
"&ContourIntegral;", "222E",
"&copf;", "D835 DD54",
"&Copf;", "2102",
"&coprod;", "2210",
"&Coproduct;", "2210",
"&copy;", "A9",
"&copy", "A9",
"&COPY;", "A9",
"&COPY", "A9",
"&copysr;", "2117",
"&CounterClockwiseContourIntegral;", "2233",
"&crarr;", "21B5",
"&cross;", "2717",
"&Cross;", "2A2F",
"&Cscr;", "D835 DC9E",
"&cscr;", "D835 DCB8",
"&csub;", "2ACF",
"&csube;", "2AD1",
"&csup;", "2AD0",
"&csupe;", "2AD2",
"&ctdot;", "22EF",
"&cudarrl;", "2938",
"&cudarrr;", "2935",
"&cuepr;", "22DE",
"&cuesc;", "22DF",
"&cularr;", "21B6",
"&cularrp;", "293D",
"&cupbrcap;", "2A48",
"&cupcap;", "2A46",
"&CupCap;", "224D",
"&cup;", "222A",
"&Cup;", "22D3",
"&cupcup;", "2A4A",
"&cupdot;", "228D",
"&cupor;", "2A45",
"&cups;", "222A FE00",
"&curarr;", "21B7",
"&curarrm;", "293C",
"&curlyeqprec;", "22DE",
"&curlyeqsucc;", "22DF",
"&curlyvee;", "22CE",
"&curlywedge;", "22CF",
"&curren;", "A4",
"&curren", "A4",
"&curvearrowleft;", "21B6",
"&curvearrowright;", "21B7",
"&cuvee;", "22CE",
"&cuwed;", "22CF",
"&cwconint;", "2232",
"&cwint;", "2231",
"&cylcty;", "232D",
"&dagger;", "2020",
"&Dagger;", "2021",
"&daleth;", "2138",
"&darr;", "2193",
"&Darr;", "21A1",
"&dArr;", "21D3",
"&dash;", "2010",
"&Dashv;", "2AE4",
"&dashv;", "22A3",
"&dbkarow;", "290F",
"&dblac;", "2DD",
"&Dcaron;", "10E",
"&dcaron;", "10F",
"&Dcy;", "414",
"&dcy;", "434",
"&ddagger;", "2021",
"&ddarr;", "21CA",
"&DD;", "2145",
"&dd;", "2146",
"&DDotrahd;", "2911",
"&ddotseq;", "2A77",
"&deg;", "B0",
"&deg", "B0",
"&Del;", "2207",
"&Delta;", "394",
"&delta;", "3B4",
"&demptyv;", "29B1",
"&dfisht;", "297F",
"&Dfr;", "D835 DD07",
"&dfr;", "D835 DD21",
"&dHar;", "2965",
"&dharl;", "21C3",
"&dharr;", "21C2",
"&DiacriticalAcute;", "B4",
"&DiacriticalDot;", "2D9",
"&DiacriticalDoubleAcute;", "2DD",
"&DiacriticalGrave;", "60",
"&DiacriticalTilde;", "2DC",
"&diam;", "22C4",
"&diamond;", "22C4",
"&Diamond;", "22C4",
"&diamondsuit;", "2666",
"&diams;", "2666",
"&die;", "A8",
"&DifferentialD;", "2146",
"&digamma;", "3DD",
"&disin;", "22F2",
"&div;", "F7",
"&divide;", "F7",
"&divide", "F7",
"&divideontimes;", "22C7",
"&divonx;", "22C7",
"&DJcy;", "402",
"&djcy;", "452",
"&dlcorn;", "231E",
"&dlcrop;", "230D",
"&dollar;", "24",
"&Dopf;", "D835 DD3B",
"&dopf;", "D835 DD55",
"&Dot;", "A8",
"&dot;", "2D9",
"&DotDot;", "20DC",
"&doteq;", "2250",
"&doteqdot;", "2251",
"&DotEqual;", "2250",
"&dotminus;", "2238",
"&dotplus;", "2214",
"&dotsquare;", "22A1",
"&doublebarwedge;", "2306",
"&DoubleContourIntegral;", "222F",
"&DoubleDot;", "A8",
"&DoubleDownArrow;", "21D3",
"&DoubleLeftArrow;", "21D0",
"&DoubleLeftRightArrow;", "21D4",
"&DoubleLeftTee;", "2AE4",
"&DoubleLongLeftArrow;", "27F8",
"&DoubleLongLeftRightArrow;", "27FA",
"&DoubleLongRightArrow;", "27F9",
"&DoubleRightArrow;", "21D2",
"&DoubleRightTee;", "22A8",
"&DoubleUpArrow;", "21D1",
"&DoubleUpDownArrow;", "21D5",
"&DoubleVerticalBar;", "2225",
"&DownArrowBar;", "2913",
"&downarrow;", "2193",
"&DownArrow;", "2193",
"&Downarrow;", "21D3",
"&DownArrowUpArrow;", "21F5",
"&DownBreve;", "311",
"&downdownarrows;", "21CA",
"&downharpoonleft;", "21C3",
"&downharpoonright;", "21C2",
"&DownLeftRightVector;", "2950",
"&DownLeftTeeVector;", "295E",
"&DownLeftVectorBar;", "2956",
"&DownLeftVector;", "21BD",
"&DownRightTeeVector;", "295F",
"&DownRightVectorBar;", "2957",
"&DownRightVector;", "21C1",
"&DownTeeArrow;", "21A7",
"&DownTee;", "22A4",
"&drbkarow;", "2910",
"&drcorn;", "231F",
"&drcrop;", "230C",
"&Dscr;", "D835 DC9F",
"&dscr;", "D835 DCB9",
"&DScy;", "405",
"&dscy;", "455",
"&dsol;", "29F6",
"&Dstrok;", "110",
"&dstrok;", "111",
"&dtdot;", "22F1",
"&dtri;", "25BF",
"&dtrif;", "25BE",
"&duarr;", "21F5",
"&duhar;", "296F",
"&dwangle;", "29A6",
"&DZcy;", "40F",
"&dzcy;", "45F",
"&dzigrarr;", "27FF",
"&Eacute;", "C9",
"&Eacute", "C9",
"&eacute;", "E9",
"&eacute", "E9",
"&easter;", "2A6E",
"&Ecaron;", "11A",
"&ecaron;", "11B",
"&Ecirc;", "CA",
"&Ecirc", "CA",
"&ecirc;", "EA",
"&ecirc", "EA",
"&ecir;", "2256",
"&ecolon;", "2255",
"&Ecy;", "42D",
"&ecy;", "44D",
"&eDDot;", "2A77",
"&Edot;", "116",
"&edot;", "117",
"&eDot;", "2251",
"&ee;", "2147",
"&efDot;", "2252",
"&Efr;", "D835 DD08",
"&efr;", "D835 DD22",
"&eg;", "2A9A",
"&Egrave;", "C8",
"&Egrave", "C8",
"&egrave;", "E8",
"&egrave", "E8",
"&egs;", "2A96",
"&egsdot;", "2A98",
"&el;", "2A99",
"&Element;", "2208",
"&elinters;", "23E7",
"&ell;", "2113",
"&els;", "2A95",
"&elsdot;", "2A97",
"&Emacr;", "112",
"&emacr;", "113",
"&empty;", "2205",
"&emptyset;", "2205",
"&EmptySmallSquare;", "25FB",
"&emptyv;", "2205",
"&EmptyVerySmallSquare;", "25AB",
"&emsp13;", "2004",
"&emsp14;", "2005",
"&emsp;", "2003",
"&ENG;", "14A",
"&eng;", "14B",
"&ensp;", "2002",
"&Eogon;", "118",
"&eogon;", "119",
"&Eopf;", "D835 DD3C",
"&eopf;", "D835 DD56",
"&epar;", "22D5",
"&eparsl;", "29E3",
"&eplus;", "2A71",
"&epsi;", "3B5",
"&Epsilon;", "395",
"&epsilon;", "3B5",
"&epsiv;", "3F5",
"&eqcirc;", "2256",
"&eqcolon;", "2255",
"&eqsim;", "2242",
"&eqslantgtr;", "2A96",
"&eqslantless;", "2A95",
"&Equal;", "2A75",
"&equals;", "3D",
"&EqualTilde;", "2242",
"&equest;", "225F",
"&Equilibrium;", "21CC",
"&equiv;", "2261",
"&equivDD;", "2A78",
"&eqvparsl;", "29E5",
"&erarr;", "2971",
"&erDot;", "2253",
"&escr;", "212F",
"&Escr;", "2130",
"&esdot;", "2250",
"&Esim;", "2A73",
"&esim;", "2242",
"&Eta;", "397",
"&eta;", "3B7",
"&ETH;", "D0",
"&ETH", "D0",
"&eth;", "F0",
"&eth", "F0",
"&Euml;", "CB",
"&Euml", "CB",
"&euml;", "EB",
"&euml", "EB",
"&euro;", "20AC",
"&excl;", "21",
"&exist;", "2203",
"&Exists;", "2203",
"&expectation;", "2130",
"&exponentiale;", "2147",
"&ExponentialE;", "2147",
"&fallingdotseq;", "2252",
"&Fcy;", "424",
"&fcy;", "444",
"&female;", "2640",
"&ffilig;", "FB03",
"&fflig;", "FB00",
"&ffllig;", "FB04",
"&Ffr;", "D835 DD09",
"&ffr;", "D835 DD23",
"&filig;", "FB01",
"&FilledSmallSquare;", "25FC",
"&FilledVerySmallSquare;", "25AA",
"&fjlig;", "66 6A",
"&flat;", "266D",
"&fllig;", "FB02",
"&fltns;", "25B1",
"&fnof;", "192",
"&Fopf;", "D835 DD3D",
"&fopf;", "D835 DD57",
"&forall;", "2200",
"&ForAll;", "2200",
"&fork;", "22D4",
"&forkv;", "2AD9",
"&Fouriertrf;", "2131",
"&fpartint;", "2A0D",
"&frac12;", "BD",
"&frac12", "BD",
"&frac13;", "2153",
"&frac14;", "BC",
"&frac14", "BC",
"&frac15;", "2155",
"&frac16;", "2159",
"&frac18;", "215B",
"&frac23;", "2154",
"&frac25;", "2156",
"&frac34;", "BE",
"&frac34", "BE",
"&frac35;", "2157",
"&frac38;", "215C",
"&frac45;", "2158",
"&frac56;", "215A",
"&frac58;", "215D",
"&frac78;", "215E",
"&frasl;", "2044",
"&frown;", "2322",
"&fscr;", "D835 DCBB",
"&Fscr;", "2131",
"&gacute;", "1F5",
"&Gamma;", "393",
"&gamma;", "3B3",
"&Gammad;", "3DC",
"&gammad;", "3DD",
"&gap;", "2A86",
"&Gbreve;", "11E",
"&gbreve;", "11F",
"&Gcedil;", "122",
"&Gcirc;", "11C",
"&gcirc;", "11D",
"&Gcy;", "413",
"&gcy;", "433",
"&Gdot;", "120",
"&gdot;", "121",
"&ge;", "2265",
"&gE;", "2267",
"&gEl;", "2A8C",
"&gel;", "22DB",
"&geq;", "2265",
"&geqq;", "2267",
"&geqslant;", "2A7E",
"&gescc;", "2AA9",
"&ges;", "2A7E",
"&gesdot;", "2A80",
"&gesdoto;", "2A82",
"&gesdotol;", "2A84",
"&gesl;", "22DB FE00",
"&gesles;", "2A94",
"&Gfr;", "D835 DD0A",
"&gfr;", "D835 DD24",
"&gg;", "226B",
"&Gg;", "22D9",
"&ggg;", "22D9",
"&gimel;", "2137",
"&GJcy;", "403",
"&gjcy;", "453",
"&gla;", "2AA5",
"&gl;", "2277",
"&glE;", "2A92",
"&glj;", "2AA4",
"&gnap;", "2A8A",
"&gnapprox;", "2A8A",
"&gne;", "2A88",
"&gnE;", "2269",
"&gneq;", "2A88",
"&gneqq;", "2269",
"&gnsim;", "22E7",
"&Gopf;", "D835 DD3E",
"&gopf;", "D835 DD58",
"&grave;", "60",
"&GreaterEqual;", "2265",
"&GreaterEqualLess;", "22DB",
"&GreaterFullEqual;", "2267",
"&GreaterGreater;", "2AA2",
"&GreaterLess;", "2277",
"&GreaterSlantEqual;", "2A7E",
"&GreaterTilde;", "2273",
"&Gscr;", "D835 DCA2",
"&gscr;", "210A",
"&gsim;", "2273",
"&gsime;", "2A8E",
"&gsiml;", "2A90",
"&gtcc;", "2AA7",
"&gtcir;", "2A7A",
"&gt;", "3E",
"&gt", "3E",
"&GT;", "3E",
"&GT", "3E",
"&Gt;", "226B",
"&gtdot;", "22D7",
"&gtlPar;", "2995",
"&gtquest;", "2A7C",
"&gtrapprox;", "2A86",
"&gtrarr;", "2978",
"&gtrdot;", "22D7",
"&gtreqless;", "22DB",
"&gtreqqless;", "2A8C",
"&gtrless;", "2277",
"&gtrsim;", "2273",
"&gvertneqq;", "2269 FE00",
"&gvnE;", "2269 FE00",
"&Hacek;", "2C7",
"&hairsp;", "200A",
"&half;", "BD",
"&hamilt;", "210B",
"&HARDcy;", "42A",
"&hardcy;", "44A",
"&harrcir;", "2948",
"&harr;", "2194",
"&hArr;", "21D4",
"&harrw;", "21AD",
"&Hat;", "5E",
"&hbar;", "210F",
"&Hcirc;", "124",
"&hcirc;", "125",
"&hearts;", "2665",
"&heartsuit;", "2665",
"&hellip;", "2026",
"&hercon;", "22B9",
"&hfr;", "D835 DD25",
"&Hfr;", "210C",
"&HilbertSpace;", "210B",
"&hksearow;", "2925",
"&hkswarow;", "2926",
"&hoarr;", "21FF",
"&homtht;", "223B",
"&hookleftarrow;", "21A9",
"&hookrightarrow;", "21AA",
"&hopf;", "D835 DD59",
"&Hopf;", "210D",
"&horbar;", "2015",
"&HorizontalLine;", "2500",
"&hscr;", "D835 DCBD",
"&Hscr;", "210B",
"&hslash;", "210F",
"&Hstrok;", "126",
"&hstrok;", "127",
"&HumpDownHump;", "224E",
"&HumpEqual;", "224F",
"&hybull;", "2043",
"&hyphen;", "2010",
"&Iacute;", "CD",
"&Iacute", "CD",
"&iacute;", "ED",
"&iacute", "ED",
"&ic;", "2063",
"&Icirc;", "CE",
"&Icirc", "CE",
"&icirc;", "EE",
"&icirc", "EE",
"&Icy;", "418",
"&icy;", "438",
"&Idot;", "130",
"&IEcy;", "415",
"&iecy;", "435",
"&iexcl;", "A1",
"&iexcl", "A1",
"&iff;", "21D4",
"&ifr;", "D835 DD26",
"&Ifr;", "2111",
"&Igrave;", "CC",
"&Igrave", "CC",
"&igrave;", "EC",
"&igrave", "EC",
"&ii;", "2148",
"&iiiint;", "2A0C",
"&iiint;", "222D",
"&iinfin;", "29DC",
"&iiota;", "2129",
"&IJlig;", "132",
"&ijlig;", "133",
"&Imacr;", "12A",
"&imacr;", "12B",
"&image;", "2111",
"&ImaginaryI;", "2148",
"&imagline;", "2110",
"&imagpart;", "2111",
"&imath;", "131",
"&Im;", "2111",
"&imof;", "22B7",
"&imped;", "1B5",
"&Implies;", "21D2",
"&incare;", "2105",
"&in;", "2208",
"&infin;", "221E",
"&infintie;", "29DD",
"&inodot;", "131",
"&intcal;", "22BA",
"&int;", "222B",
"&Int;", "222C",
"&integers;", "2124",
"&Integral;", "222B",
"&intercal;", "22BA",
"&Intersection;", "22C2",
"&intlarhk;", "2A17",
"&intprod;", "2A3C",
"&InvisibleComma;", "2063",
"&InvisibleTimes;", "2062",
"&IOcy;", "401",
"&iocy;", "451",
"&Iogon;", "12E",
"&iogon;", "12F",
"&Iopf;", "D835 DD40",
"&iopf;", "D835 DD5A",
"&Iota;", "399",
"&iota;", "3B9",
"&iprod;", "2A3C",
"&iquest;", "BF",
"&iquest", "BF",
"&iscr;", "D835 DCBE",
"&Iscr;", "2110",
"&isin;", "2208",
"&isindot;", "22F5",
"&isinE;", "22F9",
"&isins;", "22F4",
"&isinsv;", "22F3",
"&isinv;", "2208",
"&it;", "2062",
"&Itilde;", "128",
"&itilde;", "129",
"&Iukcy;", "406",
"&iukcy;", "456",
"&Iuml;", "CF",
"&Iuml", "CF",
"&iuml;", "EF",
"&iuml", "EF",
"&Jcirc;", "134",
"&jcirc;", "135",
"&Jcy;", "419",
"&jcy;", "439",
"&Jfr;", "D835 DD0D",
"&jfr;", "D835 DD27",
"&jmath;", "237",
"&Jopf;", "D835 DD41",
"&jopf;", "D835 DD5B",
"&Jscr;", "D835 DCA5",
"&jscr;", "D835 DCBF",
"&Jsercy;", "408",
"&jsercy;", "458",
"&Jukcy;", "404",
"&jukcy;", "454",
"&Kappa;", "39A",
"&kappa;", "3BA",
"&kappav;", "3F0",
"&Kcedil;", "136",
"&kcedil;", "137",
"&Kcy;", "41A",
"&kcy;", "43A",
"&Kfr;", "D835 DD0E",
"&kfr;", "D835 DD28",
"&kgreen;", "138",
"&KHcy;", "425",
"&khcy;", "445",
"&KJcy;", "40C",
"&kjcy;", "45C",
"&Kopf;", "D835 DD42",
"&kopf;", "D835 DD5C",
"&Kscr;", "D835 DCA6",
"&kscr;", "D835 DCC0",
"&lAarr;", "21DA",
"&Lacute;", "139",
"&lacute;", "13A",
"&laemptyv;", "29B4",
"&lagran;", "2112",
"&Lambda;", "39B",
"&lambda;", "3BB",
"&lang;", "27E8",
"&Lang;", "27EA",
"&langd;", "2991",
"&langle;", "27E8",
"&lap;", "2A85",
"&Laplacetrf;", "2112",
"&laquo;", "AB",
"&laquo", "AB",
"&larrb;", "21E4",
"&larrbfs;", "291F",
"&larr;", "2190",
"&Larr;", "219E",
"&lArr;", "21D0",
"&larrfs;", "291D",
"&larrhk;", "21A9",
"&larrlp;", "21AB",
"&larrpl;", "2939",
"&larrsim;", "2973",
"&larrtl;", "21A2",
"&latail;", "2919",
"&lAtail;", "291B",
"&lat;", "2AAB",
"&late;", "2AAD",
"&lates;", "2AAD FE00",
"&lbarr;", "290C",
"&lBarr;", "290E",
"&lbbrk;", "2772",
"&lbrace;", "7B",
"&lbrack;", "5B",
"&lbrke;", "298B",
"&lbrksld;", "298F",
"&lbrkslu;", "298D",
"&Lcaron;", "13D",
"&lcaron;", "13E",
"&Lcedil;", "13B",
"&lcedil;", "13C",
"&lceil;", "2308",
"&lcub;", "7B",
"&Lcy;", "41B",
"&lcy;", "43B",
"&ldca;", "2936",
"&ldquo;", "201C",
"&ldquor;", "201E",
"&ldrdhar;", "2967",
"&ldrushar;", "294B",
"&ldsh;", "21B2",
"&le;", "2264",
"&lE;", "2266",
"&LeftAngleBracket;", "27E8",
"&LeftArrowBar;", "21E4",
"&leftarrow;", "2190",
"&LeftArrow;", "2190",
"&Leftarrow;", "21D0",
"&LeftArrowRightArrow;", "21C6",
"&leftarrowtail;", "21A2",
"&LeftCeiling;", "2308",
"&LeftDoubleBracket;", "27E6",
"&LeftDownTeeVector;", "2961",
"&LeftDownVectorBar;", "2959",
"&LeftDownVector;", "21C3",
"&LeftFloor;", "230A",
"&leftharpoondown;", "21BD",
"&leftharpoonup;", "21BC",
"&leftleftarrows;", "21C7",
"&leftrightarrow;", "2194",
"&LeftRightArrow;", "2194",
"&Leftrightarrow;", "21D4",
"&leftrightarrows;", "21C6",
"&leftrightharpoons;", "21CB",
"&leftrightsquigarrow;", "21AD",
"&LeftRightVector;", "294E",
"&LeftTeeArrow;", "21A4",
"&LeftTee;", "22A3",
"&LeftTeeVector;", "295A",
"&leftthreetimes;", "22CB",
"&LeftTriangleBar;", "29CF",
"&LeftTriangle;", "22B2",
"&LeftTriangleEqual;", "22B4",
"&LeftUpDownVector;", "2951",
"&LeftUpTeeVector;", "2960",
"&LeftUpVectorBar;", "2958",
"&LeftUpVector;", "21BF",
"&LeftVectorBar;", "2952",
"&LeftVector;", "21BC",
"&lEg;", "2A8B",
"&leg;", "22DA",
"&leq;", "2264",
"&leqq;", "2266",
"&leqslant;", "2A7D",
"&lescc;", "2AA8",
"&les;", "2A7D",
"&lesdot;", "2A7F",
"&lesdoto;", "2A81",
"&lesdotor;", "2A83",
"&lesg;", "22DA FE00",
"&lesges;", "2A93",
"&lessapprox;", "2A85",
"&lessdot;", "22D6",
"&lesseqgtr;", "22DA",
"&lesseqqgtr;", "2A8B",
"&LessEqualGreater;", "22DA",
"&LessFullEqual;", "2266",
"&LessGreater;", "2276",
"&lessgtr;", "2276",
"&LessLess;", "2AA1",
"&lesssim;", "2272",
"&LessSlantEqual;", "2A7D",
"&LessTilde;", "2272",
"&lfisht;", "297C",
"&lfloor;", "230A",
"&Lfr;", "D835 DD0F",
"&lfr;", "D835 DD29",
"&lg;", "2276",
"&lgE;", "2A91",
"&lHar;", "2962",
"&lhard;", "21BD",
"&lharu;", "21BC",
"&lharul;", "296A",
"&lhblk;", "2584",
"&LJcy;", "409",
"&ljcy;", "459",
"&llarr;", "21C7",
"&ll;", "226A",
"&Ll;", "22D8",
"&llcorner;", "231E",
"&Lleftarrow;", "21DA",
"&llhard;", "296B",
"&lltri;", "25FA",
"&Lmidot;", "13F",
"&lmidot;", "140",
"&lmoustache;", "23B0",
"&lmoust;", "23B0",
"&lnap;", "2A89",
"&lnapprox;", "2A89",
"&lne;", "2A87",
"&lnE;", "2268",
"&lneq;", "2A87",
"&lneqq;", "2268",
"&lnsim;", "22E6",
"&loang;", "27EC",
"&loarr;", "21FD",
"&lobrk;", "27E6",
"&longleftarrow;", "27F5",
"&LongLeftArrow;", "27F5",
"&Longleftarrow;", "27F8",
"&longleftrightarrow;", "27F7",
"&LongLeftRightArrow;", "27F7",
"&Longleftrightarrow;", "27FA",
"&longmapsto;", "27FC",
"&longrightarrow;", "27F6",
"&LongRightArrow;", "27F6",
"&Longrightarrow;", "27F9",
"&looparrowleft;", "21AB",
"&looparrowright;", "21AC",
"&lopar;", "2985",
"&Lopf;", "D835 DD43",
"&lopf;", "D835 DD5D",
"&loplus;", "2A2D",
"&lotimes;", "2A34",
"&lowast;", "2217",
"&lowbar;", "5F",
"&LowerLeftArrow;", "2199",
"&LowerRightArrow;", "2198",
"&loz;", "25CA",
"&lozenge;", "25CA",
"&lozf;", "29EB",
"&lpar;", "28",
"&lparlt;", "2993",
"&lrarr;", "21C6",
"&lrcorner;", "231F",
"&lrhar;", "21CB",
"&lrhard;", "296D",
"&lrm;", "200E",
"&lrtri;", "22BF",
"&lsaquo;", "2039",
"&lscr;", "D835 DCC1",
"&Lscr;", "2112",
"&lsh;", "21B0",
"&Lsh;", "21B0",
"&lsim;", "2272",
"&lsime;", "2A8D",
"&lsimg;", "2A8F",
"&lsqb;", "5B",
"&lsquo;", "2018",
"&lsquor;", "201A",
"&Lstrok;", "141",
"&lstrok;", "142",
"&ltcc;", "2AA6",
"&ltcir;", "2A79",
"&lt;", "3C",
"&lt", "3C",
"&LT;", "3C",
"&LT", "3C",
"&Lt;", "226A",
"&ltdot;", "22D6",
"&lthree;", "22CB",
"&ltimes;", "22C9",
"&ltlarr;", "2976",
"&ltquest;", "2A7B",
"&ltri;", "25C3",
"&ltrie;", "22B4",
"&ltrif;", "25C2",
"&ltrPar;", "2996",
"&lurdshar;", "294A",
"&luruhar;", "2966",
"&lvertneqq;", "2268 FE00",
"&lvnE;", "2268 FE00",
"&macr;", "AF",
"&macr", "AF",
"&male;", "2642",
"&malt;", "2720",
"&maltese;", "2720",
"&Map;", "2905",
"&map;", "21A6",
"&mapsto;", "21A6",
"&mapstodown;", "21A7",
"&mapstoleft;", "21A4",
"&mapstoup;", "21A5",
"&marker;", "25AE",
"&mcomma;", "2A29",
"&Mcy;", "41C",
"&mcy;", "43C",
"&mdash;", "2014",
"&mDDot;", "223A",
"&measuredangle;", "2221",
"&MediumSpace;", "205F",
"&Mellintrf;", "2133",
"&Mfr;", "D835 DD10",
"&mfr;", "D835 DD2A",
"&mho;", "2127",
"&micro;", "B5",
"&micro", "B5",
"&midast;", "2A",
"&midcir;", "2AF0",
"&mid;", "2223",
"&middot;", "B7",
"&middot", "B7",
"&minusb;", "229F",
"&minus;", "2212",
"&minusd;", "2238",
"&minusdu;", "2A2A",
"&MinusPlus;", "2213",
"&mlcp;", "2ADB",
"&mldr;", "2026",
"&mnplus;", "2213",
"&models;", "22A7",
"&Mopf;", "D835 DD44",
"&mopf;", "D835 DD5E",
"&mp;", "2213",
"&mscr;", "D835 DCC2",
"&Mscr;", "2133",
"&mstpos;", "223E",
"&Mu;", "39C",
"&mu;", "3BC",
"&multimap;", "22B8",
"&mumap;", "22B8",
"&nabla;", "2207",
"&Nacute;", "143",
"&nacute;", "144",
"&nang;", "2220 20D2",
"&nap;", "2249",
"&napE;", "2A70 338",
"&napid;", "224B 338",
"&napos;", "149",
"&napprox;", "2249",
"&natural;", "266E",
"&naturals;", "2115",
"&natur;", "266E",
"&nbsp;", "A0",
"&nbsp", "A0",
"&nbump;", "224E 338",
"&nbumpe;", "224F 338",
"&ncap;", "2A43",
"&Ncaron;", "147",
"&ncaron;", "148",
"&Ncedil;", "145",
"&ncedil;", "146",
"&ncong;", "2247",
"&ncongdot;", "2A6D 338",
"&ncup;", "2A42",
"&Ncy;", "41D",
"&ncy;", "43D",
"&ndash;", "2013",
"&nearhk;", "2924",
"&nearr;", "2197",
"&neArr;", "21D7",
"&nearrow;", "2197",
"&ne;", "2260",
"&nedot;", "2250 338",
"&NegativeMediumSpace;", "200B",
"&NegativeThickSpace;", "200B",
"&NegativeThinSpace;", "200B",
"&NegativeVeryThinSpace;", "200B",
"&nequiv;", "2262",
"&nesear;", "2928",
"&nesim;", "2242 338",
"&NestedGreaterGreater;", "226B",
"&NestedLessLess;", "226A",
"&NewLine;", "0A",
"&nexist;", "2204",
"&nexists;", "2204",
"&Nfr;", "D835 DD11",
"&nfr;", "D835 DD2B",
"&ngE;", "2267 338",
"&nge;", "2271",
"&ngeq;", "2271",
"&ngeqq;", "2267 338",
"&ngeqslant;", "2A7E 338",
"&nges;", "2A7E 338",
"&nGg;", "22D9 338",
"&ngsim;", "2275",
"&nGt;", "226B 20D2",
"&ngt;", "226F",
"&ngtr;", "226F",
"&nGtv;", "226B 338",
"&nharr;", "21AE",
"&nhArr;", "21CE",
"&nhpar;", "2AF2",
"&ni;", "220B",
"&nis;", "22FC",
"&nisd;", "22FA",
"&niv;", "220B",
"&NJcy;", "40A",
"&njcy;", "45A",
"&nlarr;", "219A",
"&nlArr;", "21CD",
"&nldr;", "2025",
"&nlE;", "2266 338",
"&nle;", "2270",
"&nleftarrow;", "219A",
"&nLeftarrow;", "21CD",
"&nleftrightarrow;", "21AE",
"&nLeftrightarrow;", "21CE",
"&nleq;", "2270",
"&nleqq;", "2266 338",
"&nleqslant;", "2A7D 338",
"&nles;", "2A7D 338",
"&nless;", "226E",
"&nLl;", "22D8 338",
"&nlsim;", "2274",
"&nLt;", "226A 20D2",
"&nlt;", "226E",
"&nltri;", "22EA",
"&nltrie;", "22EC",
"&nLtv;", "226A 338",
"&nmid;", "2224",
"&NoBreak;", "2060",
"&NonBreakingSpace;", "A0",
"&nopf;", "D835 DD5F",
"&Nopf;", "2115",
"&Not;", "2AEC",
"&not;", "AC",
"&not", "AC",
"&NotCongruent;", "2262",
"&NotCupCap;", "226D",
"&NotDoubleVerticalBar;", "2226",
"&NotElement;", "2209",
"&NotEqual;", "2260",
"&NotEqualTilde;", "2242 338",
"&NotExists;", "2204",
"&NotGreater;", "226F",
"&NotGreaterEqual;", "2271",
"&NotGreaterFullEqual;", "2267 338",
"&NotGreaterGreater;", "226B 338",
"&NotGreaterLess;", "2279",
"&NotGreaterSlantEqual;", "2A7E 338",
"&NotGreaterTilde;", "2275",
"&NotHumpDownHump;", "224E 338",
"&NotHumpEqual;", "224F 338",
"&notin;", "2209",
"&notindot;", "22F5 338",
"&notinE;", "22F9 338",
"&notinva;", "2209",
"&notinvb;", "22F7",
"&notinvc;", "22F6",
"&NotLeftTriangleBar;", "29CF 338",
"&NotLeftTriangle;", "22EA",
"&NotLeftTriangleEqual;", "22EC",
"&NotLess;", "226E",
"&NotLessEqual;", "2270",
"&NotLessGreater;", "2278",
"&NotLessLess;", "226A 338",
"&NotLessSlantEqual;", "2A7D 338",
"&NotLessTilde;", "2274",
"&NotNestedGreaterGreater;", "2AA2 338",
"&NotNestedLessLess;", "2AA1 338",
"&notni;", "220C",
"&notniva;", "220C",
"&notnivb;", "22FE",
"&notnivc;", "22FD",
"&NotPrecedes;", "2280",
"&NotPrecedesEqual;", "2AAF 338",
"&NotPrecedesSlantEqual;", "22E0",
"&NotReverseElement;", "220C",
"&NotRightTriangleBar;", "29D0 338",
"&NotRightTriangle;", "22EB",
"&NotRightTriangleEqual;", "22ED",
"&NotSquareSubset;", "228F 338",
"&NotSquareSubsetEqual;", "22E2",
"&NotSquareSuperset;", "2290 338",
"&NotSquareSupersetEqual;", "22E3",
"&NotSubset;", "2282 20D2",
"&NotSubsetEqual;", "2288",
"&NotSucceeds;", "2281",
"&NotSucceedsEqual;", "2AB0 338",
"&NotSucceedsSlantEqual;", "22E1",
"&NotSucceedsTilde;", "227F 338",
"&NotSuperset;", "2283 20D2",
"&NotSupersetEqual;", "2289",
"&NotTilde;", "2241",
"&NotTildeEqual;", "2244",
"&NotTildeFullEqual;", "2247",
"&NotTildeTilde;", "2249",
"&NotVerticalBar;", "2224",
"&nparallel;", "2226",
"&npar;", "2226",
"&nparsl;", "2AFD 20E5",
"&npart;", "2202 338",
"&npolint;", "2A14",
"&npr;", "2280",
"&nprcue;", "22E0",
"&nprec;", "2280",
"&npreceq;", "2AAF 338",
"&npre;", "2AAF 338",
"&nrarrc;", "2933 338",
"&nrarr;", "219B",
"&nrArr;", "21CF",
"&nrarrw;", "219D 338",
"&nrightarrow;", "219B",
"&nRightarrow;", "21CF",
"&nrtri;", "22EB",
"&nrtrie;", "22ED",
"&nsc;", "2281",
"&nsccue;", "22E1",
"&nsce;", "2AB0 338",
"&Nscr;", "D835 DCA9",
"&nscr;", "D835 DCC3",
"&nshortmid;", "2224",
"&nshortparallel;", "2226",
"&nsim;", "2241",
"&nsime;", "2244",
"&nsimeq;", "2244",
"&nsmid;", "2224",
"&nspar;", "2226",
"&nsqsube;", "22E2",
"&nsqsupe;", "22E3",
"&nsub;", "2284",
"&nsubE;", "2AC5 338",
"&nsube;", "2288",
"&nsubset;", "2282 20D2",
"&nsubseteq;", "2288",
"&nsubseteqq;", "2AC5 338",
"&nsucc;", "2281",
"&nsucceq;", "2AB0 338",
"&nsup;", "2285",
"&nsupE;", "2AC6 338",
"&nsupe;", "2289",
"&nsupset;", "2283 20D2",
"&nsupseteq;", "2289",
"&nsupseteqq;", "2AC6 338",
"&ntgl;", "2279",
"&Ntilde;", "D1",
"&Ntilde", "D1",
"&ntilde;", "F1",
"&ntilde", "F1",
"&ntlg;", "2278",
"&ntriangleleft;", "22EA",
"&ntrianglelefteq;", "22EC",
"&ntriangleright;", "22EB",
"&ntrianglerighteq;", "22ED",
"&Nu;", "39D",
"&nu;", "3BD",
"&num;", "23",
"&numero;", "2116",
"&numsp;", "2007",
"&nvap;", "224D 20D2",
"&nvdash;", "22AC",
"&nvDash;", "22AD",
"&nVdash;", "22AE",
"&nVDash;", "22AF",
"&nvge;", "2265 20D2",
"&nvgt;", "3E 20D2",
"&nvHarr;", "2904",
"&nvinfin;", "29DE",
"&nvlArr;", "2902",
"&nvle;", "2264 20D2",
"&nvlt;", "3C 20D2",
"&nvltrie;", "22B4 20D2",
"&nvrArr;", "2903",
"&nvrtrie;", "22B5 20D2",
"&nvsim;", "223C 20D2",
"&nwarhk;", "2923",
"&nwarr;", "2196",
"&nwArr;", "21D6",
"&nwarrow;", "2196",
"&nwnear;", "2927",
"&Oacute;", "D3",
"&Oacute", "D3",
"&oacute;", "F3",
"&oacute", "F3",
"&oast;", "229B",
"&Ocirc;", "D4",
"&Ocirc", "D4",
"&ocirc;", "F4",
"&ocirc", "F4",
"&ocir;", "229A",
"&Ocy;", "41E",
"&ocy;", "43E",
"&odash;", "229D",
"&Odblac;", "150",
"&odblac;", "151",
"&odiv;", "2A38",
"&odot;", "2299",
"&odsold;", "29BC",
"&OElig;", "152",
"&oelig;", "153",
"&ofcir;", "29BF",
"&Ofr;", "D835 DD12",
"&ofr;", "D835 DD2C",
"&ogon;", "2DB",
"&Ograve;", "D2",
"&Ograve", "D2",
"&ograve;", "F2",
"&ograve", "F2",
"&ogt;", "29C1",
"&ohbar;", "29B5",
"&ohm;", "3A9",
"&oint;", "222E",
"&olarr;", "21BA",
"&olcir;", "29BE",
"&olcross;", "29BB",
"&oline;", "203E",
"&olt;", "29C0",
"&Omacr;", "14C",
"&omacr;", "14D",
"&Omega;", "3A9",
"&omega;", "3C9",
"&Omicron;", "39F",
"&omicron;", "3BF",
"&omid;", "29B6",
"&ominus;", "2296",
"&Oopf;", "D835 DD46",
"&oopf;", "D835 DD60",
"&opar;", "29B7",
"&OpenCurlyDoubleQuote;", "201C",
"&OpenCurlyQuote;", "2018",
"&operp;", "29B9",
"&oplus;", "2295",
"&orarr;", "21BB",
"&Or;", "2A54",
"&or;", "2228",
"&ord;", "2A5D",
"&order;", "2134",
"&orderof;", "2134",
"&ordf;", "AA",
"&ordf", "AA",
"&ordm;", "BA",
"&ordm", "BA",
"&origof;", "22B6",
"&oror;", "2A56",
"&orslope;", "2A57",
"&orv;", "2A5B",
"&oS;", "24C8",
"&Oscr;", "D835 DCAA",
"&oscr;", "2134",
"&Oslash;", "D8",
"&Oslash", "D8",
"&oslash;", "F8",
"&oslash", "F8",
"&osol;", "2298",
"&Otilde;", "D5",
"&Otilde", "D5",
"&otilde;", "F5",
"&otilde", "F5",
"&otimesas;", "2A36",
"&Otimes;", "2A37",
"&otimes;", "2297",
"&Ouml;", "D6",
"&Ouml", "D6",
"&ouml;", "F6",
"&ouml", "F6",
"&ovbar;", "233D",
"&OverBar;", "203E",
"&OverBrace;", "23DE",
"&OverBracket;", "23B4",
"&OverParenthesis;", "23DC",
"&para;", "B6",
"&para", "B6",
"&parallel;", "2225",
"&par;", "2225",
"&parsim;", "2AF3",
"&parsl;", "2AFD",
"&part;", "2202",
"&PartialD;", "2202",
"&Pcy;", "41F",
"&pcy;", "43F",
"&percnt;", "25",
"&period;", "2E",
"&permil;", "2030",
"&perp;", "22A5",
"&pertenk;", "2031",
"&Pfr;", "D835 DD13",
"&pfr;", "D835 DD2D",
"&Phi;", "3A6",
"&phi;", "3C6",
"&phiv;", "3D5",
"&phmmat;", "2133",
"&phone;", "260E",
"&Pi;", "3A0",
"&pi;", "3C0",
"&pitchfork;", "22D4",
"&piv;", "3D6",
"&planck;", "210F",
"&planckh;", "210E",
"&plankv;", "210F",
"&plusacir;", "2A23",
"&plusb;", "229E",
"&pluscir;", "2A22",
"&plus;", "2B",
"&plusdo;", "2214",
"&plusdu;", "2A25",
"&pluse;", "2A72",
"&PlusMinus;", "B1",
"&plusmn;", "B1",
"&plusmn", "B1",
"&plussim;", "2A26",
"&plustwo;", "2A27",
"&pm;", "B1",
"&Poincareplane;", "210C",
"&pointint;", "2A15",
"&popf;", "D835 DD61",
"&Popf;", "2119",
"&pound;", "A3",
"&pound", "A3",
"&prap;", "2AB7",
"&Pr;", "2ABB",
"&pr;", "227A",
"&prcue;", "227C",
"&precapprox;", "2AB7",
"&prec;", "227A",
"&preccurlyeq;", "227C",
"&Precedes;", "227A",
"&PrecedesEqual;", "2AAF",
"&PrecedesSlantEqual;", "227C",
"&PrecedesTilde;", "227E",
"&preceq;", "2AAF",
"&precnapprox;", "2AB9",
"&precneqq;", "2AB5",
"&precnsim;", "22E8",
"&pre;", "2AAF",
"&prE;", "2AB3",
"&precsim;", "227E",
"&prime;", "2032",
"&Prime;", "2033",
"&primes;", "2119",
"&prnap;", "2AB9",
"&prnE;", "2AB5",
"&prnsim;", "22E8",
"&prod;", "220F",
"&Product;", "220F",
"&profalar;", "232E",
"&profline;", "2312",
"&profsurf;", "2313",
"&prop;", "221D",
"&Proportional;", "221D",
"&Proportion;", "2237",
"&propto;", "221D",
"&prsim;", "227E",
"&prurel;", "22B0",
"&Pscr;", "D835 DCAB",
"&pscr;", "D835 DCC5",
"&Psi;", "3A8",
"&psi;", "3C8",
"&puncsp;", "2008",
"&Qfr;", "D835 DD14",
"&qfr;", "D835 DD2E",
"&qint;", "2A0C",
"&qopf;", "D835 DD62",
"&Qopf;", "211A",
"&qprime;", "2057",
"&Qscr;", "D835 DCAC",
"&qscr;", "D835 DCC6",
"&quaternions;", "210D",
"&quatint;", "2A16",
"&quest;", "3F",
"&questeq;", "225F",
"&quot;", "22",
"&quot", "22",
"&QUOT;", "22",
"&QUOT", "22",
"&rAarr;", "21DB",
"&race;", "223D 331",
"&Racute;", "154",
"&racute;", "155",
"&radic;", "221A",
"&raemptyv;", "29B3",
"&rang;", "27E9",
"&Rang;", "27EB",
"&rangd;", "2992",
"&range;", "29A5",
"&rangle;", "27E9",
"&raquo;", "BB",
"&raquo", "BB",
"&rarrap;", "2975",
"&rarrb;", "21E5",
"&rarrbfs;", "2920",
"&rarrc;", "2933",
"&rarr;", "2192",
"&Rarr;", "21A0",
"&rArr;", "21D2",
"&rarrfs;", "291E",
"&rarrhk;", "21AA",
"&rarrlp;", "21AC",
"&rarrpl;", "2945",
"&rarrsim;", "2974",
"&Rarrtl;", "2916",
"&rarrtl;", "21A3",
"&rarrw;", "219D",
"&ratail;", "291A",
"&rAtail;", "291C",
"&ratio;", "2236",
"&rationals;", "211A",
"&rbarr;", "290D",
"&rBarr;", "290F",
"&RBarr;", "2910",
"&rbbrk;", "2773",
"&rbrace;", "7D",
"&rbrack;", "5D",
"&rbrke;", "298C",
"&rbrksld;", "298E",
"&rbrkslu;", "2990",
"&Rcaron;", "158",
"&rcaron;", "159",
"&Rcedil;", "156",
"&rcedil;", "157",
"&rceil;", "2309",
"&rcub;", "7D",
"&Rcy;", "420",
"&rcy;", "440",
"&rdca;", "2937",
"&rdldhar;", "2969",
"&rdquo;", "201D",
"&rdquor;", "201D",
"&rdsh;", "21B3",
"&real;", "211C",
"&realine;", "211B",
"&realpart;", "211C",
"&reals;", "211D",
"&Re;", "211C",
"&rect;", "25AD",
"&reg;", "AE",
"&reg", "AE",
"&REG;", "AE",
"&REG", "AE",
"&ReverseElement;", "220B",
"&ReverseEquilibrium;", "21CB",
"&ReverseUpEquilibrium;", "296F",
"&rfisht;", "297D",
"&rfloor;", "230B",
"&rfr;", "D835 DD2F",
"&Rfr;", "211C",
"&rHar;", "2964",
"&rhard;", "21C1",
"&rharu;", "21C0",
"&rharul;", "296C",
"&Rho;", "3A1",
"&rho;", "3C1",
"&rhov;", "3F1",
"&RightAngleBracket;", "27E9",
"&RightArrowBar;", "21E5",
"&rightarrow;", "2192",
"&RightArrow;", "2192",
"&Rightarrow;", "21D2",
"&RightArrowLeftArrow;", "21C4",
"&rightarrowtail;", "21A3",
"&RightCeiling;", "2309",
"&RightDoubleBracket;", "27E7",
"&RightDownTeeVector;", "295D",
"&RightDownVectorBar;", "2955",
"&RightDownVector;", "21C2",
"&RightFloor;", "230B",
"&rightharpoondown;", "21C1",
"&rightharpoonup;", "21C0",
"&rightleftarrows;", "21C4",
"&rightleftharpoons;", "21CC",
"&rightrightarrows;", "21C9",
"&rightsquigarrow;", "219D",
"&RightTeeArrow;", "21A6",
"&RightTee;", "22A2",
"&RightTeeVector;", "295B",
"&rightthreetimes;", "22CC",
"&RightTriangleBar;", "29D0",
"&RightTriangle;", "22B3",
"&RightTriangleEqual;", "22B5",
"&RightUpDownVector;", "294F",
"&RightUpTeeVector;", "295C",
"&RightUpVectorBar;", "2954",
"&RightUpVector;", "21BE",
"&RightVectorBar;", "2953",
"&RightVector;", "21C0",
"&ring;", "2DA",
"&risingdotseq;", "2253",
"&rlarr;", "21C4",
"&rlhar;", "21CC",
"&rlm;", "200F",
"&rmoustache;", "23B1",
"&rmoust;", "23B1",
"&rnmid;", "2AEE",
"&roang;", "27ED",
"&roarr;", "21FE",
"&robrk;", "27E7",
"&ropar;", "2986",
"&ropf;", "D835 DD63",
"&Ropf;", "211D",
"&roplus;", "2A2E",
"&rotimes;", "2A35",
"&RoundImplies;", "2970",
"&rpar;", "29",
"&rpargt;", "2994",
"&rppolint;", "2A12",
"&rrarr;", "21C9",
"&Rrightarrow;", "21DB",
"&rsaquo;", "203A",
"&rscr;", "D835 DCC7",
"&Rscr;", "211B",
"&rsh;", "21B1",
"&Rsh;", "21B1",
"&rsqb;", "5D",
"&rsquo;", "2019",
"&rsquor;", "2019",
"&rthree;", "22CC",
"&rtimes;", "22CA",
"&rtri;", "25B9",
"&rtrie;", "22B5",
"&rtrif;", "25B8",
"&rtriltri;", "29CE",
"&RuleDelayed;", "29F4",
"&ruluhar;", "2968",
"&rx;", "211E",
"&Sacute;", "15A",
"&sacute;", "15B",
"&sbquo;", "201A",
"&scap;", "2AB8",
"&Scaron;", "160",
"&scaron;", "161",
"&Sc;", "2ABC",
"&sc;", "227B",
"&sccue;", "227D",
"&sce;", "2AB0",
"&scE;", "2AB4",
"&Scedil;", "15E",
"&scedil;", "15F",
"&Scirc;", "15C",
"&scirc;", "15D",
"&scnap;", "2ABA",
"&scnE;", "2AB6",
"&scnsim;", "22E9",
"&scpolint;", "2A13",
"&scsim;", "227F",
"&Scy;", "421",
"&scy;", "441",
"&sdotb;", "22A1",
"&sdot;", "22C5",
"&sdote;", "2A66",
"&searhk;", "2925",
"&searr;", "2198",
"&seArr;", "21D8",
"&searrow;", "2198",
"&sect;", "A7",
"&sect", "A7",
"&semi;", "3B",
"&seswar;", "2929",
"&setminus;", "2216",
"&setmn;", "2216",
"&sext;", "2736",
"&Sfr;", "D835 DD16",
"&sfr;", "D835 DD30",
"&sfrown;", "2322",
"&sharp;", "266F",
"&SHCHcy;", "429",
"&shchcy;", "449",
"&SHcy;", "428",
"&shcy;", "448",
"&ShortDownArrow;", "2193",
"&ShortLeftArrow;", "2190",
"&shortmid;", "2223",
"&shortparallel;", "2225",
"&ShortRightArrow;", "2192",
"&ShortUpArrow;", "2191",
"&shy;", "AD",
"&shy", "AD",
"&Sigma;", "3A3",
"&sigma;", "3C3",
"&sigmaf;", "3C2",
"&sigmav;", "3C2",
"&sim;", "223C",
"&simdot;", "2A6A",
"&sime;", "2243",
"&simeq;", "2243",
"&simg;", "2A9E",
"&simgE;", "2AA0",
"&siml;", "2A9D",
"&simlE;", "2A9F",
"&simne;", "2246",
"&simplus;", "2A24",
"&simrarr;", "2972",
"&slarr;", "2190",
"&SmallCircle;", "2218",
"&smallsetminus;", "2216",
"&smashp;", "2A33",
"&smeparsl;", "29E4",
"&smid;", "2223",
"&smile;", "2323",
"&smt;", "2AAA",
"&smte;", "2AAC",
"&smtes;", "2AAC FE00",
"&SOFTcy;", "42C",
"&softcy;", "44C",
"&solbar;", "233F",
"&solb;", "29C4",
"&sol;", "2F",
"&Sopf;", "D835 DD4A",
"&sopf;", "D835 DD64",
"&spades;", "2660",
"&spadesuit;", "2660",
"&spar;", "2225",
"&sqcap;", "2293",
"&sqcaps;", "2293 FE00",
"&sqcup;", "2294",
"&sqcups;", "2294 FE00",
"&Sqrt;", "221A",
"&sqsub;", "228F",
"&sqsube;", "2291",
"&sqsubset;", "228F",
"&sqsubseteq;", "2291",
"&sqsup;", "2290",
"&sqsupe;", "2292",
"&sqsupset;", "2290",
"&sqsupseteq;", "2292",
"&square;", "25A1",
"&Square;", "25A1",
"&SquareIntersection;", "2293",
"&SquareSubset;", "228F",
"&SquareSubsetEqual;", "2291",
"&SquareSuperset;", "2290",
"&SquareSupersetEqual;", "2292",
"&SquareUnion;", "2294",
"&squarf;", "25AA",
"&squ;", "25A1",
"&squf;", "25AA",
"&srarr;", "2192",
"&Sscr;", "D835 DCAE",
"&sscr;", "D835 DCC8",
"&ssetmn;", "2216",
"&ssmile;", "2323",
"&sstarf;", "22C6",
"&Star;", "22C6",
"&star;", "2606",
"&starf;", "2605",
"&straightepsilon;", "3F5",
"&straightphi;", "3D5",
"&strns;", "AF",
"&sub;", "2282",
"&Sub;", "22D0",
"&subdot;", "2ABD",
"&subE;", "2AC5",
"&sube;", "2286",
"&subedot;", "2AC3",
"&submult;", "2AC1",
"&subnE;", "2ACB",
"&subne;", "228A",
"&subplus;", "2ABF",
"&subrarr;", "2979",
"&subset;", "2282",
"&Subset;", "22D0",
"&subseteq;", "2286",
"&subseteqq;", "2AC5",
"&SubsetEqual;", "2286",
"&subsetneq;", "228A",
"&subsetneqq;", "2ACB",
"&subsim;", "2AC7",
"&subsub;", "2AD5",
"&subsup;", "2AD3",
"&succapprox;", "2AB8",
"&succ;", "227B",
"&succcurlyeq;", "227D",
"&Succeeds;", "227B",
"&SucceedsEqual;", "2AB0",
"&SucceedsSlantEqual;", "227D",
"&SucceedsTilde;", "227F",
"&succeq;", "2AB0",
"&succnapprox;", "2ABA",
"&succneqq;", "2AB6",
"&succnsim;", "22E9",
"&succsim;", "227F",
"&SuchThat;", "220B",
"&sum;", "2211",
"&Sum;", "2211",
"&sung;", "266A",
"&sup1;", "B9",
"&sup1", "B9",
"&sup2;", "B2",
"&sup2", "B2",
"&sup3;", "B3",
"&sup3", "B3",
"&sup;", "2283",
"&Sup;", "22D1",
"&supdot;", "2ABE",
"&supdsub;", "2AD8",
"&supE;", "2AC6",
"&supe;", "2287",
"&supedot;", "2AC4",
"&Superset;", "2283",
"&SupersetEqual;", "2287",
"&suphsol;", "27C9",
"&suphsub;", "2AD7",
"&suplarr;", "297B",
"&supmult;", "2AC2",
"&supnE;", "2ACC",
"&supne;", "228B",
"&supplus;", "2AC0",
"&supset;", "2283",
"&Supset;", "22D1",
"&supseteq;", "2287",
"&supseteqq;", "2AC6",
"&supsetneq;", "228B",
"&supsetneqq;", "2ACC",
"&supsim;", "2AC8",
"&supsub;", "2AD4",
"&supsup;", "2AD6",
"&swarhk;", "2926",
"&swarr;", "2199",
"&swArr;", "21D9",
"&swarrow;", "2199",
"&swnwar;", "292A",
"&szlig;", "DF",
"&szlig", "DF",
"&Tab;", "09",
"&target;", "2316",
"&Tau;", "3A4",
"&tau;", "3C4",
"&tbrk;", "23B4",
"&Tcaron;", "164",
"&tcaron;", "165",
"&Tcedil;", "162",
"&tcedil;", "163",
"&Tcy;", "422",
"&tcy;", "442",
"&tdot;", "20DB",
"&telrec;", "2315",
"&Tfr;", "D835 DD17",
"&tfr;", "D835 DD31",
"&there4;", "2234",
"&therefore;", "2234",
"&Therefore;", "2234",
"&Theta;", "398",
"&theta;", "3B8",
"&thetasym;", "3D1",
"&thetav;", "3D1",
"&thickapprox;", "2248",
"&thicksim;", "223C",
"&ThickSpace;", "205F 200A",
"&ThinSpace;", "2009",
"&thinsp;", "2009",
"&thkap;", "2248",
"&thksim;", "223C",
"&THORN;", "DE",
"&THORN", "DE",
"&thorn;", "FE",
"&thorn", "FE",
"&tilde;", "2DC",
"&Tilde;", "223C",
"&TildeEqual;", "2243",
"&TildeFullEqual;", "2245",
"&TildeTilde;", "2248",
"&timesbar;", "2A31",
"&timesb;", "22A0",
"&times;", "D7",
"&times", "D7",
"&timesd;", "2A30",
"&tint;", "222D",
"&toea;", "2928",
"&topbot;", "2336",
"&topcir;", "2AF1",
"&top;", "22A4",
"&Topf;", "D835 DD4B",
"&topf;", "D835 DD65",
"&topfork;", "2ADA",
"&tosa;", "2929",
"&tprime;", "2034",
"&trade;", "2122",
"&TRADE;", "2122",
"&triangle;", "25B5",
"&triangledown;", "25BF",
"&triangleleft;", "25C3",
"&trianglelefteq;", "22B4",
"&triangleq;", "225C",
"&triangleright;", "25B9",
"&trianglerighteq;", "22B5",
"&tridot;", "25EC",
"&trie;", "225C",
"&triminus;", "2A3A",
"&TripleDot;", "20DB",
"&triplus;", "2A39",
"&trisb;", "29CD",
"&tritime;", "2A3B",
"&trpezium;", "23E2",
"&Tscr;", "D835 DCAF",
"&tscr;", "D835 DCC9",
"&TScy;", "426",
"&tscy;", "446",
"&TSHcy;", "40B",
"&tshcy;", "45B",
"&Tstrok;", "166",
"&tstrok;", "167",
"&twixt;", "226C",
"&twoheadleftarrow;", "219E",
"&twoheadrightarrow;", "21A0",
"&Uacute;", "DA",
"&Uacute", "DA",
"&uacute;", "FA",
"&uacute", "FA",
"&uarr;", "2191",
"&Uarr;", "219F",
"&uArr;", "21D1",
"&Uarrocir;", "2949",
"&Ubrcy;", "40E",
"&ubrcy;", "45E",
"&Ubreve;", "16C",
"&ubreve;", "16D",
"&Ucirc;", "DB",
"&Ucirc", "DB",
"&ucirc;", "FB",
"&ucirc", "FB",
"&Ucy;", "423",
"&ucy;", "443",
"&udarr;", "21C5",
"&Udblac;", "170",
"&udblac;", "171",
"&udhar;", "296E",
"&ufisht;", "297E",
"&Ufr;", "D835 DD18",
"&ufr;", "D835 DD32",
"&Ugrave;", "D9",
"&Ugrave", "D9",
"&ugrave;", "F9",
"&ugrave", "F9",
"&uHar;", "2963",
"&uharl;", "21BF",
"&uharr;", "21BE",
"&uhblk;", "2580",
"&ulcorn;", "231C",
"&ulcorner;", "231C",
"&ulcrop;", "230F",
"&ultri;", "25F8",
"&Umacr;", "16A",
"&umacr;", "16B",
"&uml;", "A8",
"&uml", "A8",
"&UnderBar;", "5F",
"&UnderBrace;", "23DF",
"&UnderBracket;", "23B5",
"&UnderParenthesis;", "23DD",
"&Union;", "22C3",
"&UnionPlus;", "228E",
"&Uogon;", "172",
"&uogon;", "173",
"&Uopf;", "D835 DD4C",
"&uopf;", "D835 DD66",
"&UpArrowBar;", "2912",
"&uparrow;", "2191",
"&UpArrow;", "2191",
"&Uparrow;", "21D1",
"&UpArrowDownArrow;", "21C5",
"&updownarrow;", "2195",
"&UpDownArrow;", "2195",
"&Updownarrow;", "21D5",
"&UpEquilibrium;", "296E",
"&upharpoonleft;", "21BF",
"&upharpoonright;", "21BE",
"&uplus;", "228E",
"&UpperLeftArrow;", "2196",
"&UpperRightArrow;", "2197",
"&upsi;", "3C5",
"&Upsi;", "3D2",
"&upsih;", "3D2",
"&Upsilon;", "3A5",
"&upsilon;", "3C5",
"&UpTeeArrow;", "21A5",
"&UpTee;", "22A5",
"&upuparrows;", "21C8",
"&urcorn;", "231D",
"&urcorner;", "231D",
"&urcrop;", "230E",
"&Uring;", "16E",
"&uring;", "16F",
"&urtri;", "25F9",
"&Uscr;", "D835 DCB0",
"&uscr;", "D835 DCCA",
"&utdot;", "22F0",
"&Utilde;", "168",
"&utilde;", "169",
"&utri;", "25B5",
"&utrif;", "25B4",
"&uuarr;", "21C8",
"&Uuml;", "DC",
"&Uuml", "DC",
"&uuml;", "FC",
"&uuml", "FC",
"&uwangle;", "29A7",
"&vangrt;", "299C",
"&varepsilon;", "3F5",
"&varkappa;", "3F0",
"&varnothing;", "2205",
"&varphi;", "3D5",
"&varpi;", "3D6",
"&varpropto;", "221D",
"&varr;", "2195",
"&vArr;", "21D5",
"&varrho;", "3F1",
"&varsigma;", "3C2",
"&varsubsetneq;", "228A FE00",
"&varsubsetneqq;", "2ACB FE00",
"&varsupsetneq;", "228B FE00",
"&varsupsetneqq;", "2ACC FE00",
"&vartheta;", "3D1",
"&vartriangleleft;", "22B2",
"&vartriangleright;", "22B3",
"&vBar;", "2AE8",
"&Vbar;", "2AEB",
"&vBarv;", "2AE9",
"&Vcy;", "412",
"&vcy;", "432",
"&vdash;", "22A2",
"&vDash;", "22A8",
"&Vdash;", "22A9",
"&VDash;", "22AB",
"&Vdashl;", "2AE6",
"&veebar;", "22BB",
"&vee;", "2228",
"&Vee;", "22C1",
"&veeeq;", "225A",
"&vellip;", "22EE",
"&verbar;", "7C",
"&Verbar;", "2016",
"&vert;", "7C",
"&Vert;", "2016",
"&VerticalBar;", "2223",
"&VerticalLine;", "7C",
"&VerticalSeparator;", "2758",
"&VerticalTilde;", "2240",
"&VeryThinSpace;", "200A",
"&Vfr;", "D835 DD19",
"&vfr;", "D835 DD33",
"&vltri;", "22B2",
"&vnsub;", "2282 20D2",
"&vnsup;", "2283 20D2",
"&Vopf;", "D835 DD4D",
"&vopf;", "D835 DD67",
"&vprop;", "221D",
"&vrtri;", "22B3",
"&Vscr;", "D835 DCB1",
"&vscr;", "D835 DCCB",
"&vsubnE;", "2ACB FE00",
"&vsubne;", "228A FE00",
"&vsupnE;", "2ACC FE00",
"&vsupne;", "228B FE00",
"&Vvdash;", "22AA",
"&vzigzag;", "299A",
"&Wcirc;", "174",
"&wcirc;", "175",
"&wedbar;", "2A5F",
"&wedge;", "2227",
"&Wedge;", "22C0",
"&wedgeq;", "2259",
"&weierp;", "2118",
"&Wfr;", "D835 DD1A",
"&wfr;", "D835 DD34",
"&Wopf;", "D835 DD4E",
"&wopf;", "D835 DD68",
"&wp;", "2118",
"&wr;", "2240",
"&wreath;", "2240",
"&Wscr;", "D835 DCB2",
"&wscr;", "D835 DCCC",
"&xcap;", "22C2",
"&xcirc;", "25EF",
"&xcup;", "22C3",
"&xdtri;", "25BD",
"&Xfr;", "D835 DD1B",
"&xfr;", "D835 DD35",
"&xharr;", "27F7",
"&xhArr;", "27FA",
"&Xi;", "39E",
"&xi;", "3BE",
"&xlarr;", "27F5",
"&xlArr;", "27F8",
"&xmap;", "27FC",
"&xnis;", "22FB",
"&xodot;", "2A00",
"&Xopf;", "D835 DD4F",
"&xopf;", "D835 DD69",
"&xoplus;", "2A01",
"&xotime;", "2A02",
"&xrarr;", "27F6",
"&xrArr;", "27F9",
"&Xscr;", "D835 DCB3",
"&xscr;", "D835 DCCD",
"&xsqcup;", "2A06",
"&xuplus;", "2A04",
"&xutri;", "25B3",
"&xvee;", "22C1",
"&xwedge;", "22C0",
"&Yacute;", "DD",
"&Yacute", "DD",
"&yacute;", "FD",
"&yacute", "FD",
"&YAcy;", "42F",
"&yacy;", "44F",
"&Ycirc;", "176",
"&ycirc;", "177",
"&Ycy;", "42B",
"&ycy;", "44B",
"&yen;", "A5",
"&yen", "A5",
"&Yfr;", "D835 DD1C",
"&yfr;", "D835 DD36",
"&YIcy;", "407",
"&yicy;", "457",
"&Yopf;", "D835 DD50",
"&yopf;", "D835 DD6A",
"&Yscr;", "D835 DCB4",
"&yscr;", "D835 DCCE",
"&YUcy;", "42E",
"&yucy;", "44E",
"&yuml;", "FF",
"&yuml", "FF",
"&Yuml;", "178",
"&Zacute;", "179",
"&zacute;", "17A",
"&Zcaron;", "17D",
"&zcaron;", "17E",
"&Zcy;", "417",
"&zcy;", "437",
"&Zdot;", "17B",
"&zdot;", "17C",
"&zeetrf;", "2128",
"&ZeroWidthSpace;", "200B",
"&Zeta;", "396",
"&zeta;", "3B6",
"&zfr;", "D835 DD37",
"&Zfr;", "2128",
"&ZHcy;", "416",
"&zhcy;", "436",
"&zigrarr;", "21DD",
"&zopf;", "D835 DD6B",
"&Zopf;", "2124",
"&Zscr;", "D835 DCB5",
"&zscr;", "D835 DCCF",
"&zwj;", "200D",
"&zwnj;", "200C"
    ), byrow=TRUE, ncol=2)
return(nE)
}
