% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/linalg.R
\name{linalg_eigh}
\alias{linalg_eigh}
\title{Computes the eigenvalue decomposition of a complex Hermitian or real symmetric matrix.}
\usage{
linalg_eigh(A, UPLO = "L")
}
\arguments{
\item{A}{(Tensor): tensor of shape \verb{(*, n, n)} where \code{*} is zero or more batch dimensions
consisting of symmetric or Hermitian matrices.}

\item{UPLO}{('L', 'U', optional): controls whether to use the upper or lower triangular part
of \code{A} in the computations. Default: \code{'L'}.}
}
\value{
A list \verb{(eigenvalues, eigenvectors)} which corresponds to \teqn{\Lambda} and \teqn{Q} above.
\code{eigenvalues} will always be real-valued, even when \code{A} is complex.

It will also be ordered in ascending order.
\code{eigenvectors} will have the same dtype as \code{A} and will contain the eigenvectors as its columns.
}
\description{
Letting \teqn{\mathbb{K}} be \teqn{\mathbb{R}} or \teqn{\mathbb{C}},
the \strong{eigenvalue decomposition} of a complex Hermitian or real symmetric matrix
\teqn{A \in \mathbb{K}^{n \times n}} is defined as
}
\details{
\Sexpr[results=rd, stage=build]{torch:::math_to_rd("
A = Q \\\\operatorname{diag}(\\\\Lambda) Q^{H}\\\\mathrlap{\\\\qquad Q \\\\in \\\\mathbb{K}^{n \\\\times n}, \\\\Lambda \\\\in \\\\mathbb{R}^n}
")}

where \teqn{Q^{H}} is the conjugate transpose when \teqn{Q} is complex, and the transpose when \teqn{Q} is real-valued.
\teqn{Q} is orthogonal in the real case and unitary in the complex case.

Supports input of float, double, cfloat and cdouble dtypes.
Also supports batches of matrices, and if \code{A} is a batch of matrices then
the output has the same batch dimensions.

\code{A} is assumed to be Hermitian (resp. symmetric), but this is not checked internally, instead:
\itemize{
\item If \code{UPLO}\ \verb{= 'L'} (default), only the lower triangular part of the matrix is used in the computation.
\item If \code{UPLO}\ \verb{= 'U'}, only the upper triangular part of the matrix is used.
The eigenvalues are returned in ascending order.
}
}
\note{
The eigenvalues of real symmetric or complex Hermitian matrices are always real.
}
\section{Warning}{

\itemize{
\item The eigenvectors of a symmetric matrix are not unique, nor are they continuous with
respect to \code{A}. Due to this lack of uniqueness, different hardware and
software may compute different eigenvectors.
This non-uniqueness is caused by the fact that multiplying an eigenvector by
\code{-1} in the real case or by \teqn{e^{i \phi}, \phi \in \mathbb{R}} in the complex
case produces another set of valid eigenvectors of the matrix.
This non-uniqueness problem is even worse when the matrix has repeated eigenvalues.
In this case, one may multiply the associated eigenvectors spanning
the subspace by a rotation matrix and the resulting eigenvectors will be valid
eigenvectors.
\item Gradients computed using the \code{eigenvectors} tensor will only be finite when
\code{A} has unique eigenvalues.
Furthermore, if the distance between any two eigvalues is close to zero,
the gradient will be numerically unstable, as it depends on the eigenvalues
\teqn{\lambda_i} through the computation of
\teqn{\frac{1}{\min_{i \neq j} \lambda_i - \lambda_j}}.
}
}

\examples{
if (torch_is_installed()) {
a <- torch_randn(2, 2)
linalg_eigh(a)
}
}
\seealso{
\itemize{
\item \code{\link[=linalg_eigvalsh]{linalg_eigvalsh()}} computes only the eigenvalues values of a Hermitian matrix.
Unlike \code{\link[=linalg_eigh]{linalg_eigh()}}, the gradients of \code{\link[=linalg_eigvalsh]{linalg_eigvalsh()}} are always
numerically stable.
\item \code{\link[=linalg_cholesky]{linalg_cholesky()}} for a different decomposition of a Hermitian matrix.
The Cholesky decomposition gives less information about the matrix but is much faster
to compute than the eigenvalue decomposition.
\item \code{\link[=linalg_eig]{linalg_eig()}} for a (slower) function that computes the eigenvalue decomposition
of a not necessarily Hermitian square matrix.
\item \code{\link[=linalg_svd]{linalg_svd()}} for a (slower) function that computes the more general SVD
decomposition of matrices of any shape.
\item \code{\link[=linalg_qr]{linalg_qr()}} for another (much faster) decomposition that works on general
matrices.
}

Other linalg: 
\code{\link{linalg_cholesky_ex}()},
\code{\link{linalg_cholesky}()},
\code{\link{linalg_det}()},
\code{\link{linalg_eigvalsh}()},
\code{\link{linalg_eigvals}()},
\code{\link{linalg_eig}()},
\code{\link{linalg_householder_product}()},
\code{\link{linalg_inv_ex}()},
\code{\link{linalg_inv}()},
\code{\link{linalg_lstsq}()},
\code{\link{linalg_matrix_norm}()},
\code{\link{linalg_matrix_power}()},
\code{\link{linalg_matrix_rank}()},
\code{\link{linalg_multi_dot}()},
\code{\link{linalg_norm}()},
\code{\link{linalg_pinv}()},
\code{\link{linalg_qr}()},
\code{\link{linalg_slogdet}()},
\code{\link{linalg_solve_triangular}()},
\code{\link{linalg_solve}()},
\code{\link{linalg_svdvals}()},
\code{\link{linalg_svd}()},
\code{\link{linalg_tensorinv}()},
\code{\link{linalg_tensorsolve}()},
\code{\link{linalg_vector_norm}()}
}
\concept{linalg}
