\name{nicheSimilarity}

\alias{nicheSimilarity}
\alias{nicheSimilarity,SpatRaster,SpatRaster-method}
\alias{nicheSimilarity,.nicheSpatRaster,.nicheSpatRaster-method}
\alias{nicheSimilarity,SpatRaster,missing-method}


\title{Niche Similarity}

\description{
Compute multiple niche similarity (overlap) statistics between two rasters with probability of occurrence (habitat suitability) values (e.g., outputs of the predict/ensemble functions). The statistics range between 0 (no similarity) and 1 (maximum similarity; identical). The calculations can be done either in geographic space (when x and y are raster maps representing geographical distributions of species) or in environmental (niche) space (when x and y are the output of \code{\link{niche}} function).
}

\usage{
nicheSimilarity(x,y,stat,...)

}

\arguments{
  \item{x}{habitat suitabiliy of the first species in geographic or niche space: a single-layer \code{SpatRaster} object (or with two layers if y is missing) containing probability of occurrence; or a .nicheSpatRaster object (the output of the niche function) for the first species}
  \item{y}{habitat suitabiliy of the second species in geographic or niche space: a \code{SpatRaster} object with a single layer containing probability of occurrence, or a .nicheSpatRaster object (the output of the niche function) for the secies species}
  \item{stat}{A character vector specifying the names of niche similarity statistics that can be one or multiple items from c("Imod","Icor","D","O","BC","R"); "all" (or NULL) for all statistics }
  \item{...}{not implemented.}
  }
\details{
Six metrics are implemented to quantify niche overlap (similarity) between two species (or two separate populations of the same species) including:

- D: Schoener's D

- Imod: Modified Hellinger distance

- Icor: Corrected Modified Hellinger distance

- R: Horn's R

- O: Pianka's O

- BC: Bray-Curtis distance

The equations for these metrics are described in Rodder & Engler (2011).

The probability raster maps (geographic distributions) of the two species can be provided in \code{x} and \code{y} (so, nlyr(x) = nlyr(y) = 1 should be valid), or both rasters can be provided in \code{x}  when \code{y} is missing (then, nlyr(x) = 2 should be valid).

Alternatively, the niche similarity can be calculated in environmental space given the object generated by the \code{\link{niche}} function for each species. Of course the niche for both species should be generated based on the same set of predictors. Given that the \code{\link{niche}} function generates the niche raster based on only two predictors, the niche similarity calculation may be repeated for different combinations of predictors, or all the predictor variables can be first transformed and reduced into two components (using priciple component analysis; \code{\link{pca}}), then the niche for each species can be generated based on the first two components (see example.)


}


\value{
 a numeric vector with values of niche similarity for different metrics.
}


\references{
Naimi, B., Araujo, M.B. (2016) sdm: a reproducible and extensible R platform for species distribution modelling, Ecography, 39:368-375, DOI: 10.1111/ecog.01881

Rodder, D., & Engler, J. O. (2011). Quantitative metrics of overlaps in Grinnellian niches: advances and possible drawbacks. Global Ecology and Biogeography, 20(6), 915-927.

}

\author{Babak Naimi \email{naimi.b@gmail.com}

\url{https://www.r-gis.net/}

\url{https://www.biogeoinformatics.org/}
}


\examples{
\dontrun{

file <- system.file("external/sp1.shp", package="sdm") # get the path to the species data

sp1 <- vect(file) # read the shapefile for species 1

file <- system.file("external/sp2.shp", package="sdm")

sp2 <- vect(file) # read the shapefile for species 2

path <- system.file("external", package="sdm") # path to the folder contains the data

lst <- list.files(path=path,pattern='asc$',full.names = T) # list of predictor filenames

lst

preds <- rast(lst) # making a raster object

names(preds) # 4 environmental variables are used!



d1 <- sdmData(formula=Occurrence~., train=sp1, predictors=preds)

d1

d2 <- sdmData(formula=Occurrence~., train=sp2, predictors=preds)

d2

# fit models for species 1
m1 <- sdm(Occurrence~.,data=d1,methods=c('rf','glm','brt'),
          replication='sub',test.p=30)
m1

# fit models for species 2:
m2 <- sdm(Occurrence~.,data=d2,methods=c('rf','glm','brt'),
          replication='sub',test.p=30)
m2


# ensemble using weighted averaging based on AUC statistic (species 1):    
p1 <- ensemble(m1, newdata=preds,setting=list(method='weighted',stat='AUC'))
plot(p1, main='Habitat Suitability in Geographic Space (species 1)')

# ensemble for species 2:
p2 <- ensemble(m2, newdata=preds,setting=list(method='weighted',stat='AUC'))
plot(p2, main='Habitat Suitability in Geographic Space (species 2)')

# maps together:
plot(c(p1,p2), main=c('species 1','species 2'))


# calculating niche similarity (all metrics) in geographic space:
nicheSimilarity(p1,p2)

nicheSimilarity(p1,p2, stat=c('Icor','Imod'))

######################################

# calculating niche similarity in environmental space:

# Mapping Ecological Niche using selected two variables
n1 <- niche(x=preds, h=p1, c('precipitation','temperature'),out = T)

n2 <- niche(x=preds, h=p2, c('precipitation','temperature'),out=T)

nicheSimilarity(n1,n2)

###################
#### Alternatively, predictors can be transformed to two components using PCA

pc <- pca(preds)

pc


# niche for first species based on the first two components of PCA:
n1 <- niche(pc@data,p1,c("Comp.1","Comp.2"),out=T)

n2 <- niche(pc@data,p2,c("Comp.1","Comp.2"),out=T)


nicheSimilarity(n1,n2)

}
}

\keyword{spatial}
\keyword{species}
\keyword{data}
