% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/iidCox.R
\name{iidCox}
\alias{iidCox}
\alias{iidCox.coxph}
\alias{iidCox.cph}
\alias{iidCox.phreg}
\alias{iidCox.CauseSpecificCox}
\title{Extract iid decomposition from a Cox model}
\usage{
iidCox(object, newdata, baseline.iid, tau.hazard, tau.max, store.iid,
  keep.times, return.object)

\method{iidCox}{coxph}(object, newdata = NULL, baseline.iid = TRUE,
  tau.hazard = NULL, tau.max = NULL, store.iid = "full",
  keep.times = TRUE, return.object = TRUE)

\method{iidCox}{cph}(object, newdata = NULL, baseline.iid = TRUE,
  tau.hazard = NULL, tau.max = NULL, store.iid = "full",
  keep.times = TRUE, return.object = TRUE)

\method{iidCox}{phreg}(object, newdata = NULL, baseline.iid = TRUE,
  tau.hazard = NULL, tau.max = NULL, store.iid = "full",
  keep.times = TRUE, return.object = TRUE)

\method{iidCox}{CauseSpecificCox}(object, newdata = NULL,
  baseline.iid = TRUE, tau.hazard = NULL, tau.max = NULL,
  store.iid = "full", keep.times = TRUE, return.object = TRUE)
}
\arguments{
\item{object}{object The fitted Cox regression model object either
obtained with \code{coxph} (survival package) or \code{cph}
(rms package).}

\item{newdata}{[data.frame] Optional new data at which to do iid decomposition}

\item{baseline.iid}{[logical] Should the influence function for the baseline hazard be computed.}

\item{tau.hazard}{[numeric vector] the vector of times at which the i.i.d decomposition of the baseline hazard will be computed}

\item{tau.max}{[numeric] latest time at which the i.i.d decomposition of the baseline hazard will be computed. Alternative to \code{tau.hazard}.}

\item{store.iid}{[character] the method used to compute the influence function and the standard error.
Can be \code{"full"}, \code{"approx"} or \code{"minimal"}. See the details section.}

\item{keep.times}{[logical] If \code{TRUE} add the evaluation times to the output.}

\item{return.object}{[logical] If \code{TRUE} return the object where the iid decomposition has been added.
Otherwise return a list (see the return section)}
}
\value{
For Cox models, the iid slot is a list containing:
 \itemize{
 \item{IFbeta}{Influence function for the regression coefficient.}
 \item{IFhazard}{Time differential of the influence function of the hazard.}
 \item{IFcumhazard}{Influence function of the cumulative hazard.}
 \item{calcIFhazard}{Elements used to compute the influence function at a given time.}
 \item{time}{Times at which the influence function has been evaluated.}
 \item{etime1.min}{Time of first event (i.e. jump) in each strata.}
 \item{etime.max}{Last observation time (i.e. jump or censoring) in each strata.}
 \item{indexObs}{Index of the observation in the original dataset.}
}
For Cause-Specific Cox models,
a list containing the iid decomposition relative to each cause is returned.
}
\description{
Compute the influence function for each observation used to estimate the model
}
\details{
This function implements the first three formula (no number,10,11) of the subsection
"Empirical estimates" in (Ozenne et al., 2017).

If there is no event in a strata, the influence function for the baseline hazard is set to 0.

\code{store.iid} equal to \code{"full"} exports the influence function for the coefficients
and the baseline hazard at each event time.
\code{store.iid} equal to \code{"approx"} does the same except that the terms that do not contributes
to the variance are not ignored (i.e. set to 0)
\code{store.iid} equal to \code{"minimal"} exports the influence function for the coefficients. For the
baseline hazard it only computes the quantities necessary to compute the influence function in order to save memory.
}
\examples{
library(survival)
library(data.table)
library(prodlim)
set.seed(10)
d <- sampleData(100, outcome = "survival")[,.(eventtime,event,X1,X6)]
setkey(d, eventtime)

m.cox <- coxph(Surv(eventtime, event) ~ X1+X6, data = d, y = TRUE, x = TRUE)
system.time(IF.cox <- iidCox(m.cox))
system.time(IF.cox_approx <- iidCox(m.cox, store.iid = "approx"))


IF.cox.all <- iidCox(m.cox, tau.hazard = sort(unique(c(7,d$eventtime))))
IF.cox.beta <- iidCox(m.cox, baseline.iid = FALSE)

}
\references{
Brice Ozenne, Anne Lyngholm Sorensen, Thomas Scheike, Christian Torp-Pedersen and Thomas Alexander Gerds.
riskRegression: Predicting the Risk of an Event using Cox Regression Models.
The R Journal (2017) 9:2, pages 440-460.
}
