% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/confint.ate.R
\name{confint.ate}
\alias{confint.ate}
\title{Confidence Intervals and Confidence Bands for the Predicted Absolute Risk (Cumulative Incidence Function)}
\usage{
\method{confint}{ate}(object, parm = NULL, level = 0.95,
  nsim.band = 10000, meanRisk.transform = "none",
  diffRisk.transform = "none", ratioRisk.transform = "none",
  seed = NA, bootci.method = "perc", ...)
}
\arguments{
\item{object}{A \code{ate} object, i.e. output of the \code{ate} function.}

\item{parm}{not used. For compatibility with the generic method.}

\item{level}{[numeric, 0-1] Level of confidence.}

\item{nsim.band}{[integer, >0]the number of simulations used to compute the quantiles for the confidence bands.}

\item{meanRisk.transform}{[character] the transformation used to improve coverage
of the confidence intervals for the mean risk in small samples.
Can be \code{"none"}, \code{"log"}, \code{"loglog"}, \code{"cloglog"}.}

\item{diffRisk.transform}{[character] the transformation used to improve coverage
of the confidence intervals for the risk difference in small samples.
Can be \code{"none"}, \code{"atanh"}.}

\item{ratioRisk.transform}{[character] the transformation used to improve coverage
of the confidence intervals for the risk ratio in small samples.
Can be \code{"none"}, \code{"log"}.}

\item{seed}{[integer, >0] seed number set when performing simulation for the confidence bands.
If not given or NA no seed is set.}

\item{bootci.method}{[character] Method for constructing bootstrap confidence intervals.
Either "perc" (the default), "norm", "basic", "stud", or "bca".}

\item{...}{not used.}
}
\description{
Confidence intervals and confidence Bands for the predicted absolute risk (cumulative incidence function).
}
\details{
Confidence bands and confidence intervals computed via the influence function
are automatically restricted to the interval [0;1]. \cr \cr

Confidence intervals obtained via bootstrap are computed
using the \code{boot.ci} function of the \code{boot} package.
p-value are obtained using test inversion method
(finding the smallest confidence level such that the interval contain the null hypothesis).
}
\examples{
library(survival)
library(data.table)

## ## generate data ####
set.seed(10)
d <- sampleData(70,outcome="survival")
d[, X1 := paste0("T",rbinom(.N, size = 2, prob = c(0.51)))]
## table(d$X1)

#### stratified Cox model ####
fit <- coxph(Surv(time,event)~X1 + strata(X2) + X6,
             data=d, ties="breslow", x = TRUE, y = TRUE)

#### average treatment effect ####
fit.ate <- ate(fit, treatment = "X1", times = 1:3, data = d,
               se = TRUE, iid = TRUE, band = TRUE)
print(fit.ate, type = "meanRisk")
dt.ate <- as.data.table(fit.ate)

## manual calculation of se
dd <- copy(d)
dd$X1 <- rep(factor("T0", levels = paste0("T",0:2)), NROW(dd))
out <- predictCox(fit, newdata = dd, se = TRUE, times = 1:3, average.iid = TRUE)
term1 <- -out$survival.average.iid
term2 <- sweep(1-out$survival, MARGIN = 2, FUN = "-", STATS = colMeans(1-out$survival))
sqrt(colSums((term1 + term2/NROW(d))^2)) 
## fit.ate$meanRisk[treatment=="T0",meanRisk.se]

## note
out2 <- predictCox(fit, newdata = dd, se = TRUE, times = 1:3, iid = TRUE)
mean(out2$survival.iid[,1,1])
out$survival.average.iid[1,1]

## check confidence intervals (no transformation)
dt.ate[,.(lower = pmax(0,value + qnorm(0.025) * se),
          lower2 = lower,
          upper = value + qnorm(0.975) * se,
          upper2 = upper)]

## add confidence intervals computed on the log-log scale
## and backtransformed
outCI <- confint(fit.ate,
                 meanRisk.transform = "loglog", diffRisk.transform = "atanh",
                 ratioRisk.transform = "log")
print(outCI, type = "meanRisk")

dt.ate[type == "ate", newse := se/(value*log(value))]
dt.ate[type == "ate", .(lower = exp(-exp(log(-log(value)) - 1.96 * newse)),
                        upper = exp(-exp(log(-log(value)) + 1.96 * newse)))]
}
\author{
Brice Ozenne
}
