% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/query_search.R
\name{orderly_search}
\alias{orderly_search}
\title{Search for orderly reports matching criteria}
\usage{
orderly_search(
  query,
  name,
  parameters = NULL,
  draft = FALSE,
  root = NULL,
  locate = TRUE,
  remote = NULL
)
}
\arguments{
\item{query}{The query string - see details and examples}

\item{name}{Name of the report to search.  Only a single report
can be searched at once.}

\item{parameters}{Named list of parameters (as would be passed to
\code{\link{orderly_run}}) if your query uses parameters on the
right-hand-side of an expression.}

\item{draft}{Should draft reports be used searched?  This should
be used only in development.  Valid values are logical
(\code{TRUE}, \code{FALSE}) or use the string \code{newer} to
use draft reports where they are newer than archive reports.
For consistency, \code{always} and \code{never} are equivalent
to \code{TRUE} and \code{FALSE}, respectively.}

\item{root}{The path to an orderly root directory, or \code{NULL}
(the default) to search for one from the current working
directory if \code{locate} is \code{TRUE}.}

\item{locate}{Logical, indicating if the configuration should be
searched for.  If \code{TRUE} and \code{config} is not given,
then orderly looks in the working directory and up through its
parents until it finds an \code{orderly_config.yml} file.}

\item{remote}{A remote to use, if you want to apply the query
remotely.  If this is used then \code{draft} cannot be set to
\code{TRUE} as remotes do not expose draft reports.}
}
\value{
A character vector of matching report ids, possibly
  zero-length.  If the query is a "latest" query, then exactly one
  report id, possibly NA.
}
\description{
Search for orderly reports matching criteria.  This can be used to
find reports where a particular parameter or tag was used (it will
likely be expanded as time goes on - let us know if that would be
useful).  We search within versions of a single report only.
}
\details{
The query syntax is deliberately very simple; it may expand a bit
later.  At this point you can search for parameters and for tags,
and these can be combined.  Note that if you are using OrderlyWeb,
then only orderly (and not OrderlyWeb) tags are searched.

The idea here is that the queries can be used to find ids that
match certain criteria for use as dependencies.  This function
lets you work out what would be resolved by the query, and using
this query string in a \code{depends:} section will let you select
a report that matches some criteria.  For example, suppose that
you have report \code{A} that takes a parameter "fruit" with
values like "apple", "banana", and a report \code{B} that depends
on A.  You could then write:

\preformatted{
depends:
  A:
    id: latest(parameter:fruit == "apple")
    uses:
      summary.csv: summary.csv
}

To get the \code{summary.csv} file out of the latest report
\code{A} that was run with the "fruit" parameter set to "apple".
If "B" itself takes parameters, you can use those parameters in
these query expressions like

\preformatted{
depends:
  A:
    id: latest(parameter:fruit == target_fruit)
    uses:
      summary.csv: summary.csv
}

(assuming that \code{B} takes a parameter \code{target_fruit}).

The syntax for tags is simpler, one uses \code{tag:tagname} to
test for presence of a tag called "tagname".

Search queries can be joined by \code{&&} and \code{||} and
grouped using parentheses, these groups (or tags) can be negated
with \code{!}, so a complicated query expression might look like:

\preformatted{
(parameter:fruit == "apple" && !tag:weekly) || parameter:fruit == "banana"
}

Be careful of comparing floating point numbers with \code{==} or
\code{!=} as they may not always return what you expect (for example
\code{sqrt(3)^2 == 3} is \code{FALSE}).

In the documentation and error messages we may refer to the
left-hand-side of \code{:} as a "namespace".  At this point the
only supported namespaces are \code{tag} and \code{parameter}.
}
\examples{
# We need a few reports here to actually query.  There is a report in
# the "demo" example called "other" that takes a parameter "nmin",
# which is used to filter data - it's not terribly important what it
# does here, but it can give us a set of reports to use.

# The demo set also includes configuration for two tags, called
# "dataset" and "plot" - the "dataset" tag will always be applied
# as it is listed in the orderly.yml but we can still add the
# "plot" tag interactively
root <- orderly::orderly_example("demo")

# A helper function to mass-produce reports will reduce noise a bit
run1 <- function(nmin, tags = NULL) {
  id <- orderly_run("other", root = root, echo = FALSE,
                    parameters = list(nmin = nmin), tags = tags)
  orderly_commit(id, root = root)
  id
}

ids <- c(run1(0.1), run1(0.2, "plot"), run1(0.3))

# We can then ask for all reports where the parameter nmin was more
# than some value
orderly::orderly_search("parameter:nmin > 0.15", "other", root = root)

# Or use "&&" to find tags within a range
orderly::orderly_search("parameter:nmin > 0.1 && parameter:nmin < 0.3",
                        "other", root = root)

# If a parameter is not present in some versions of a report you
# can use is.null to test for it (this is only ever the case if
# you have altered a report definition to add or remove a
# parameter)
orderly::orderly_search("is.null(parameter:nmin)", "other", root = root)

# We can look for tags
orderly::orderly_search("tag:plot", "other", root = root)

# or exclude them
orderly::orderly_search("!tag:plot", "other", root = root)

# or combine that with the presence/absence of a tag
orderly::orderly_search("parameter:nmin > 0.15 && !tag:plot",
                        "other", root = root)

# Use latest() over a query to find the latest report matching the
# query expression.
orderly::orderly_search("latest(parameter:nmin > 0.15)",
                        "other", root = root)

# If no reports are found, then a zero-length character vector is returned
orderly::orderly_search("parameter:nmin > 0.4", "other", root = root)

# Or, in the case of latest(), NA
orderly::orderly_search("latest(parameter:nmin > 0.4)",
                        "other", root = root)
}
