% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/nnt.R
\name{nnt}
\alias{nnt}
\alias{nnt.default}
\alias{nnt.meta}
\alias{print.nnt.meta}
\title{Calculate the number needed to treat (NNT)}
\usage{
nnt(x, ...)

\method{nnt}{meta}(x, p.c, common = x$common, random = x$random, small.values = "good", ...)

\method{nnt}{default}(x, p.c, sm, lower, upper, small.values = "good", ...)

\method{print}{nnt.meta}(
  x,
  common = x$common,
  random = x$random,
  digits = gs("digits"),
  digits.prop = gs("digits.prop"),
  big.mark = gs("big.mark"),
  ...
)
}
\arguments{
\item{x}{An object of class \code{meta}, or estimated treatment
effect, i.e., risk difference(s), risk ratio(s), or odds
ratio(s).}

\item{\dots}{Additional arguments (to catch deprecated arguments).}

\item{p.c}{Baseline risk (control group event probability).}

\item{common}{A logical indicating whether NNTs should be
calculated based on common effect estimate.}

\item{random}{A logical indicating whether NNTs should be
calculated based on random effects estimate.}

\item{small.values}{A character string specifying whether small
treatment effects indicate a beneficial (\code{"good"}) or
harmful (\code{"bad"}) effect, can be abbreviated.}

\item{sm}{Summary measure.}

\item{lower}{Lower confidence interval limit.}

\item{upper}{Upper confidence interval limit.}

\item{digits}{Minimal number of significant digits, see
\code{print.default}.}

\item{digits.prop}{Minimal number of significant digits for
proportions, see \code{print.default}.}

\item{big.mark}{A character used as thousands separator.}
}
\description{
Calculate the number needed to treat (NNT) from estimated risk
difference, risk ratio, or odds ratio, and a baseline risk.
}
\details{
The number needed to treat (NNT) is the estimated number of
patients who need to be treated with a new treatment instead of a
standard for one additional patient to benefit (Laupacis et al.,
1988; Cook & Sackett, 1995). This definition of the NNT implies
that the new treatment is more beneficial than the standard. If the
new treatment is indeed less beneficial than the standard, the NNT
gives the number of patients treated with the new treatment to
observe an additional harmful event. Accordingly, the abbreviations
NNTB and NNTH can be used to distinguish between beneficial and
harmful NNTs (Altman, 1998).

NNTs can be easily computed from an estimated risk difference (RD),
risk ratio (RR), or odds ratio (OR) and a given baseline risk
(Higgins et al., 2022, section 15.4.4). It is also possible to
calculate NNTs from hazard ratios (HR) (Altman & Andersen,
1999). Accordlingly, NNTs can be calculated for meta-analyses
generated with \code{\link{metabin}} or \code{\link{metagen}} if
argument \code{sm} was equal to \code{"RD"}, \code{"RR"},
\code{"OR"}, or \code{"HR"}. It is also possible to provide only
estimated treatment effects and baseline risks (see Examples).

The baseline risk can be specified using argument \code{p.c}. If
this argument is missing, the minimum, mean, and maximum of the
control event probabilities in the meta-analysis are used for
\code{\link{metabin}} and control event probabilities of 0.1, 0.2,
\dots, 0.9 are used for \code{\link{metagen}}.

Argument \code{small.values} can be used to specify whether small
treatment effects indicate a beneficial (\code{"good"}) or harmful
(\code{"bad"}) effect. For \code{small.values = "small"}, odds and
risk ratios below 1 and risk differences below 0 indicate that the
new treatment is beneficial. For \code{small.values = "bad"}, odds
and risk ratios above 1 and risk differences above 0 indicate that
the new treatment is beneficial.

\subsection{Interpretation of (positive and negative) NNTs}{
A positive value for the estimated NNT indicates that the new
treatment is beneficial, i.e., the NNT is actually an NNTB. On the
other hand, a negative value for the estimated NNT indicates that
the new treatment is harmful, i.e., the NNT is actually an NNTH.

The minimal value for the NNTB is 1. In this extreme case the new
treatment is 100\% effective and the standard treatment is 0\%
effective, i.e., only one patient has to be treated with the new
treatment for one additional patient to benefit. The NNTB increases
with decreasing difference between the two risks. If both risks are
equal, the NNTB is infinite.

The other extreme is an NNT of -1 if the new treatment is 0\%
effective and the standard is 100\% effective. Here, one additional
harmful event is observed for each patient treated with the new
treatment. The NNT approaches minus infinity if the difference
between the two risks decreases to 0. Finally, an NNT of -1
translates to an NNTH of 1 with possible values from 1 to infinity.
}

\subsection{Confidence interval for the NNT}{
Confidence limits for an NNT are derived from the lower and upper
confidence limits of the summary measure using the same formulae as
for the NNT (Higgins et al., 2022, section 15.4.4).

A peculiar problem arises if the confidence interval for the
summary measure includes the null effect (i.e., RR = 1, OR = 1, HR
= 1, or RD = 0). In this case the confidence interval for the NNT
contains both NNTB and NNTH values and it seemingly does not
include the estimated NNT.

As described above, a positive NNT value corresponds to an NNTB and
the absolute value of a negative NNT is equal to an
NNTH. Accordingly, a confidence interval for the NNT from 20 to -5
translates to NNTB values between 20 and infinity and NNTH values
between 5 and infinity (Altman, 1998).
}
}
\examples{
# Calculate NNT for RD = -0.12 and -0.22
# (Cochrane Handbook, version 6.3, subsection 15.4.4.1)
nnt(c(-0.12, -0.22), sm = "RD")

# Calculate NNT for RR = 0.92 and baseline risk p.c = 0.3
# (Cochrane Handbook, version 6.3, subsection 15.4.4.2)
nnt(0.92, p.c = 0.3, sm = "RR")

# Calculate NNT for OR = 0.73 and baseline risk p.c = 0.3
# (Cochrane Handbook, version 6.3, subsection 15.4.4.3)
nnt(0.73, p.c = 0.3, sm = "OR")

# Use Mantel-Haenszel odds ratio to calculate NNTs
data(Olkin1995)
m1 <- metabin(ev.exp, n.exp, ev.cont, n.cont, data = Olkin1995,
              random = FALSE)
nnt(m1)

}
\references{
Altman DG (1998):
Confidence intervals for the number needed to treat.
\emph{British Medical Journal},
\bold{317}, 1309--12

Altman DG, Andersen PK (1999):
Calculating the Number Needed to Treat for Trials Where the Outcome
Is Time to an Event
\emph{British Medical Journal},
\bold{319}, 1492--95

Cook RJ, Sackett DL (1995):
The Number Needed to Treat: A Clinically Useful Measure of
Treatment Effect.
\emph{British Medical Journal},
\bold{310}, 452--54

Higgins JPT, Thomas J, Chandler J, Cumpston M, Li T, Page MJ, Welch
VA (editors) (2022):
\emph{Cochrane Handbook for Systematic Reviews of Interventions
  Version 6.3 (updated February 2022)}.
Available from www.training.cochrane.org/handbook

Laupacis A, Sackett DL, Roberts RS (1988):
An Assessment of Clinically Useful Measures of the Consequences of
Treatment.
\emph{New England Journal of Medicine},
\bold{318}, 1728--33
}
\seealso{
\code{\link{metabin}}, \code{\link{metagen}}
}
\author{
Guido Schwarzer \email{guido.schwarzer@uniklinik-freiburg.de}
}
