% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/learn.R
\name{learn}
\alias{learn}
\title{Informatic sequence classification tree learning.}
\usage{
learn(x, model = NULL, refine = "Viterbi", iterations = 50, nstart = 20,
  minK = 2, maxK = 2, minscore = 0.9, probs = 0.5, retry = TRUE,
  resize = TRUE, maxsize = max(sapply(x, length)), recursive = TRUE,
  cores = 1, quiet = TRUE, ...)
}
\arguments{
\item{x}{an object of class\code{"DNAbin"} representing a list of
DNA sequences to be used as the training data for the tree-learning process.
All sequences should be from the same genetic region of interest
and be globally alignable (i.e. without unjustified end-gaps).
This object must have a "lineage" attribute, a vector the same length as the
sequence list that is composed of semicolon-delimited lineage strings.
See \code{\link{searchGB}} for details on creating a reference sequence database.}

\item{model}{an optional object of class \code{"PHMM"} to form the model
at the root node of the classification tree. Used to train (optimize
parameters for) subsequent nested models to be positioned at successive
sub-nodes. If NULL, the root model is derived from the
sequence list prior to the recursive partitioning process.}

\item{refine}{character string giving the iterative model refinement
method to be used in the partitioning process. Valid options are
\code{"Viterbi"} (Viterbi training; the default option) and
\code{"BaumWelch"} (a modified version of the Expectation-Maximization
algorithm).}

\item{iterations}{integer giving the maximum number of training-classification
iterations to be used in the splitting process.
Note that this is not necessarily the same as the number of Viterbi training
or Baum Welch iterations to be used in model training, which can be set
using the argument \code{"maxiter"} (eventually passed to
\code{\link[aphid]{train}} via the dots argument "...").}

\item{nstart}{integer. The number of random starting sets to be chosen
for initial k-means assignment of sequences to groups.}

\item{minK}{integer. The minimum number of furications allowed at each inner
node of the tree. Defaults to 2 (all inner nodes are bifuricating).}

\item{maxK}{integer. The maximum number of furications allowed at each inner
node of the tree. Defaults to 2 (all inner nodes are bifuricating).}

\item{minscore}{numeric between 0 and 1. The minimum acceptable value
for the \emph{n}th percentile of Akaike weights (where \emph{n} is
the value given in \code{"probs"}, for the node to be split and the
recursion process to continue.
At any given node, if the \emph{n}th percentile of Akaike weights
falls below this threshold, the recursion process for the node will
terminate. As an example, if \code{minscore = 0.9} and
\code{probs = 0.5} (the default settings), and after generating two
candidate PHMMs to occupy the candidate subnodes the median
of Akaike weights is 0.89, the splitting process will
terminate and the function will simply return the unsplit root node.}

\item{probs}{numeric between 0 and 1. The percentile of Akaike weights
to test against the minimum score threshold given in \code{"minscore"}.}

\item{retry}{logical indicating whether failure to split a node based on
the criteria outlined in 'minscore' and 'probs' should prompt a second
attempt with different initial groupings. These groupings are based on
maximum kmer frequencies rather than k-means division, which can give
suboptimal groupings when the cluster sizes are different (due to
the up-weighting of larger clusters in the k-means algorithm).}

\item{resize}{logical indicating whether the models should be free to
change size during the training process or if the number of modules
should be fixed. Defaults to TRUE. Only applicable if
\code{refine = "Viterbi"}.}

\item{maxsize}{integer giving the upper bound on the number of modules
in the PHMMs. If NULL (default) no maximum size is enforced.}

\item{recursive}{logical indicating whether the splitting process
should continue recursively until the discrimination criteria
are not met (TRUE; default), or whether a single split should
take place at the root node.}

\item{cores}{integer giving the number of CPUs to use
when training the models (only applicable if
\code{refine = 'Viterbi'}). Defaults to 1.
This argument may alternatively be a 'cluster' object,
in which case it is the user's responsibility to close the socket
connection at the conclusion of the operation,
e.g. by running \code{parallel::stopCluster(cores)}.
The string 'autodetect' is also accepted, in which case the maximum
number of cores to use is one less than the total number of cores
available.}

\item{quiet}{logical indicating whether feedback should be printed
to the console. Note that the output can be verbose.}

\item{...}{further arguments to be passed on to \code{\link[aphid]{train}}).}
}
\value{
an object of class \code{"insect"}.
}
\description{
This function learns a classification tree from a reference sequence database
  using a recursive partitioning procedure.
}
\details{
The "insect" object type is a dendrogram
  with several additional attributes stored at each node.
  These include:
  "clade" the index of the node (see further details below);
  "sequences" the indices of the sequences in the reference
  database used to create the object;
  "lineage" the lowest common taxon of the sequences belonging to the node;
  "minscore" the lowest likelihood among the training sequences given
  the profile HMM stored at the node;
  "minlength" the minimum length of the sequences belonging to the node;
  "maxlength" the maximum length of the sequences belonging to the node;
  "key" the hash key used for exact sequence matching
  (bypasses the classification procedure);
  "model" the profile HMM derived from the sequence subset belonging to the node;
  "nunique" the number of unique sequences belonging to the node;
  "ntotal" the total number of sequences belonging to the node (including duplicates).

  The clade indexing system used here is based on character strings,
  where "0" refers to the root node,
  "01" is the first child node, "02" is the second child node,
  "011" is the first child node of the first child node, etc.
  The leading zero may be omitted for brevity.
  Note that each node cannot have more than 9 child nodes.
}
\examples{
\donttest{
  data(whales)
  ## use sequences 2-19 to learn the tree
  ## note that training data must retain lineage attribute
  training_data <- subset.DNAbin(whales, subset = seq_along(whales) > 1)
  ## learn the tree
  set.seed(999)
  tree <- learn(training_data, quiet = FALSE, maxiter = 5)
  ## find predicted lineage for sequence #1
  classify(whales[[1]], tree)
  ## compare with actual lineage
  attr(whales, "lineage")[1]
}
}
\references{
Blackshields G, Sievers F, Shi W, Wilm A, Higgins DG (2010) Sequence embedding
  for fast construction of guide trees for multiple sequence alignment.
  \emph{Algorithms for Molecular Biology}, \strong{5}, 21.

  Durbin R, Eddy SR, Krogh A, Mitchison G (1998) Biological
  sequence analysis: probabilistic models of proteins and nucleic acids.
  Cambridge University Press, Cambridge, United Kingdom.

  Gerstein M, Sonnhammer ELL, Chothia C (1994) Volume changes in protein evolution.
  \emph{Journal of Molecular Biology}, \strong{236}, 1067-1078.

  Juang B-H, Rabiner LR (1990) The segmental K-means
  algorithm for estimating parameters of hidden Markov models.
  \emph{IEEE Transactions on Acoustics, Speech, and Signal Processing},
  \strong{38}, 1639-1641.
}
\author{
Shaun Wilkinson
}
