% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Shapley.R
\name{Shapley}
\alias{Shapley}
\title{Prediction explanations with game theory}
\format{\code{\link{R6Class}} object.}
\description{
\code{Shapley} computes feature contributions for single predictions with the Shapley value, an approach from cooperative game theory. 
The features values of an instance cooperate to achieve the prediction. 
The Shapley value fairly distributes the difference of the instance's prediction and the datasets average prediction among the features.
}
\section{Usage}{

\preformatted{
shapley = Shapley$new(predictor, x.interest = NULL, sample.size = 100, run = TRUE)

plot(shapley)
shapley$results
print(shapley)
shapley$explain(x.interest)
}
}

\section{Arguments}{
 
For Shapley$new():
\describe{
\item{predictor: }{(Predictor)\cr 
The object (created with Predictor$new()) holding the machine learning model and the data.}
\item{x.interest: }{(data.frame)\cr  Single row with the instance to be explained.}
\item{sample.size: }{(`numeric(1)`)\cr The number of  Monte Carlo samples for estimating the Shapley value.} 
\item{run: }{(`logical(1)`)\cr Should the Interpretation method be run?}
}
}

\section{Details}{

For more details on the algorithm see https://christophm.github.io/interpretable-ml-book/shapley.html
}

\section{Fields}{

\describe{
\item{predictor: }{(Predictor)\cr 
The object (created with Predictor$new()) holding the machine learning model and the data.}
\item{results: }{(data.frame)\cr data.frame with the Shapley values (phi) per feature.}
\item{sample.size: }{(`numeric(1)`)\cr The number of times coalitions/marginals are sampled from data X. The higher the more accurate the explanations become.}
\item{x.interest: }{(data.frame)\cr Single row with the instance to be explained.}
\item{y.hat.interest: }{(numeric)\cr Predicted value for instance of interest}
\item{y.hat.average: }{(`numeric(1)`)\cr Average predicted value for data \code{X}} 
}
}

\section{Methods}{

\describe{
\item{explain(x.interest)}{method to set a new data point which to explain.}
\item{plot()}{method to plot the Shapley value. See \link{plot.Shapley}}
\item{\code{run()}}{[internal] method to run the interpretability method. Use \code{obj$run(force = TRUE)} to force a rerun.}
\item{\code{clone()}}{[internal] method to clone the R6 object.}
\item{\code{initialize()}}{[internal] method to initialize the R6 object.}
}
}

\examples{
if (require("randomForest")) {
# First we fit a machine learning model on the Boston housing data
data("Boston", package  = "MASS")
rf =  randomForest(medv ~ ., data = Boston, ntree = 50)
X = Boston[-which(names(Boston) == "medv")]
mod = Predictor$new(rf, data = X)

# Then we explain the first instance of the dataset with the Shapley method:
x.interest = X[1,]
shapley = Shapley$new(mod, x.interest = x.interest)
shapley

# Look at the results in a table
shapley$results
# Or as a plot
plot(shapley)

# Explain another instance
shapley$explain(X[2,])
plot(shapley)

# Shapley() also works with multiclass classification
rf = randomForest(Species ~ ., data= iris, ntree=50)
X = iris[-which(names(iris) == "Species")]
predict.fun = function(object, newdata) predict(object, newdata, type = "prob")
mod = Predictor$new(rf, data = X, predict.fun = predict.fun)

# Then we explain the first instance of the dataset with the Shapley() method:
shapley = Shapley$new(mod, x.interest = X[1,])
shapley$results
plot(shapley) 

# You can also focus on one class
mod = Predictor$new(rf, data = X, predict.fun = predict.fun, class = "setosa")
shapley = Shapley$new(mod, x.interest = X[1,])
shapley$results
plot(shapley) 
}
}
\references{
Strumbelj, E., Kononenko, I. (2014). Explaining prediction models and individual predictions with feature contributions. Knowledge and Information Systems, 41(3), 647-665. https://doi.org/10.1007/s10115-013-0679-x
}
\seealso{
\link{Shapley}

A different way to explain predictions: \link{LocalModel}
}
