#' Draw Self-loop Arrows on a ggplot Object
#'
#' @description
#' This function overlays self-loop arrows to a ggplot object based on data describing their positions, sizes, orientations, and styles. Self-loop arrows can be drawn in one direction or bidirectionally with customizable parameters such as color, width, and arrow type. The data can come from a CSV file generated by the ggsem Shiny app or custom input.
#' @param p A ggplot object to which the self-loop arrows will be added.
#' @param loops_data A data frame containing information about the self-loop arrows. The expected columns include:
#' \itemize{
#'   \item \code{x_center}, \code{y_center}: Center coordinates of the loop.
#'   \item \code{radius}: Radius of the loop.
#'   \item \code{color}: Color of the loop (hexadecimal color code).
#'   \item \code{width}: Width of the loop line (numeric).
#'   \item \code{alpha}: Transparency of the loop line (numeric, 0 to 1).
#'   \item \code{arrow_type}: Type of arrow (\code{"closed"} or \code{"open"}).
#'   \item \code{arrow_size}: Size of the arrowhead.
#'   \item \code{gap_size}: Size of the gap in the loop, specified as a fraction of the full circle (numeric, 0 to 1).
#'   \item \code{loop_width}, \code{loop_height}: Width and height scaling factors for the loop.
#'   \item \code{orientation}: Rotation angle of the loop in degrees.
#'   \item \code{two_way}: Logical, whether the loop is bidirectional (adds arrows at both ends).
#' }
#'
#' @param zoom_level Numeric. Adjusts the size of line widths and arrowheads relative to the plot. Default is \code{1}.
#'
#' @return
#' A ggplot object with the specified self-loop arrows added.
#' @export
#'
#' @examples
#' library(ggplot2)
#'
#' loops_data <- data.frame(
#' x_center = -5, y_center = 5, radius = 2, color = '#000000', width = 1,
#' alpha = 1, arrow_type = 'closed', arrow_size = 0.1, gap_size = 0.2,
#' loop_width = 1, loop_height = 20, orientation = 0,
#' two_way = FALSE, locked = FALSE
#' )
#'
#' p <- ggplot()
#'
#' draw_loops(p, loops_data, zoom_level = 1.2)
draw_loops <- function(p, loops_data, zoom_level = 1) {
  # Ensure loops_data is valid and not empty
  if (!is.null(loops_data) && nrow(loops_data) > 0) {
    loops_data$color <- sapply(loops_data$color, valid_hex)
    loops_data$alpha <- sapply(loops_data$alpha, valid_alpha)


    for (i in 1:nrow(loops_data)) {
      t <- seq(0, 2 * pi, length.out = 100)
      gap_angle <- loops_data$gap_size[i] * pi
      loop_start <- t[t < (2 * pi - gap_angle)]

      x_ellipse <- loops_data$x_center[i] + (loops_data$loop_width[i]) * loops_data$radius[i] * cos(loop_start)
      y_ellipse <- loops_data$y_center[i] + (loops_data$loop_height[i]) * loops_data$radius[i] * sin(loop_start)

      # Rotate the ellipse by the orientation angle
      theta <- loops_data$orientation[i] * pi / 180
      x_rotated <- cos(theta) * (x_ellipse - loops_data$x_center[i]) - sin(theta) * (y_ellipse - loops_data$y_center[i]) + loops_data$x_center[i]
      y_rotated <- sin(theta) * (x_ellipse - loops_data$x_center[i]) + cos(theta) * (y_ellipse - loops_data$y_center[i]) + loops_data$y_center[i]

      if (loops_data$two_way[i] == FALSE) {
        p <- p + annotate("path",
                          x = x_rotated,
                          y = y_rotated,
                          color = loops_data$color[i],
                          linewidth = loops_data$width[i] / zoom_level,
                          alpha = loops_data$alpha[i],
                          arrow = arrow(type = loops_data$arrow_type[i],
                                        length = unit(loops_data$arrow_size[i] / zoom_level, "inches")))
      } else {
        p <- p + annotate("path",
                          x = x_rotated,
                          y = y_rotated,
                          color = loops_data$color[i],
                          linewidth = loops_data$width[i] / zoom_level,
                          alpha = loops_data$alpha[i],
                          arrow = arrow(type = loops_data$arrow_type[i],
                                        ends = "both",
                                        length = unit(loops_data$arrow_size[i] / zoom_level, "inches")))
      }
    }
  }
  return(p)
}

