% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/emfrail.R
\name{emfrail}
\alias{emfrail}
\title{Fitting shared frailty models with the EM algorithm}
\usage{
emfrail(.data, .formula, .distribution = emfrail_distribution(),
  .control = emfrail_control())
}
\arguments{
\item{.data}{A data frame in which the formula argument can be evaluated}

\item{.formula}{A formula that contains on the left hand side an object of the type \code{Surv} in the Andersen-Gill format, and on the right hand side a \code{+cluster(id)} statement}

\item{.distribution}{An object as created by \code{\link{emfrail_distribution}}}

\item{.control}{An object as created by \code{\link{emfrail_control}}}
}
\value{
An object of the class \code{emfrail}, that is in fact a list which contains (1) the object returned by the
"outer maximization" from \code{optimx}, (2) the object with all the estimates returned by the "inner maximization",
(3) the log-likelihood of the Cox model without frailty, (4) the variance-covariance matrix adjusted for the uncertainty in the
outer maximization, (5) the results of the Commenges-Andersen test for heterogeneity and (6,7,8) are copies of the original input arguments: .formula, .distribution and .control.
Two attributes are also present, \code{class} for determining the object type and \code{metadata} which contains some information that is used to
process the input for \code{predict.emfrail()}.
}
\description{
Fitting shared frailty models with the EM algorithm
}
\details{
The \code{emfrail} function fits shared frailty models for processes which have intensity
\deqn{\lambda(t) = z \lambda_0(t) \exp(\beta' \mathbf{x})}
with a non-parametric (Breslow) baseline intensity \eqn{\lambda_0(t)}.
The distribution of \eqn{z} is usually described by one parameter \eqn{\theta}.
The family of supported distributions can be one of gamma, positive stable or PVF (power-variance-family).
For details, see the vignette and \code{\link{emfrail_distribution}} .

The algorithm is described in detail in the vignette. The short version is as follows:
The objective is to maximize a marginal likelihood with respect to 3 sets of parameters, \eqn{\theta > 0}
(the parameter of the frailty distribution), \eqn{\beta} (a vector of regression coefficients) and \eqn{\lambda_0}
(a vector of "jumps" of the cumulative hazards at each event time point).
The essence of the algorithm relies on the observation that, if \eqn{\theta} would be fixed, then the maximization
with respect to \eqn{\beta, \theta} can be done with an EM algorithm. This procedure has been described, for the
gamma frailty, in Nielsen (1992).
The "inner" problem is, for a fixed of \eqn{\theta}, to calculate
\deqn{\widehat{L}(\theta) = \mathrm{max}_{\beta, \lambda_0} L(\beta, \lambda_0 | \theta).}
which is done via an EM algorithm. The "outer" problem is to calculate
\deqn{\widehat{L} = \mathrm{max}_{\theta} \widehat{L}(\theta).}

The "inner" problem is solved relying on \code{agreg.fit} from the \code{survival} package in the M step
and with an E step which is done in \code{R} when there are closed form solutions and in \code{C++} when the
solutions are calculated with a recursive algorithm. The information matrix (given fixed \eqn{\theta}) is calculated using Louis' formula.
The "outer" problem is solved relying on the maximizers in the \code{optimx} package, which also provides an estimate
of the Hessian matrix. The remaining elements of the information matrix are approximated numerically.

Several options are available to the user. In the \code{.distribution} argument, the frailty distribution and the
starting value for \eqn{\theta} can be specified, as well as whether the data consists of left truncated
survival times or not (the default). In the \code{.control} argument, several parameters control the maximization.
The convergence criterion for the EM can be specified (or a maximum number of iterations). The "outer" procedure can be
avoided altogether and then \code{emfrail} calculates only \eqn{\widehat{L}(\theta)} at the given starting value. Also,
some parameters for \code{optimx} can be set.
}
\note{
Some possible problems may appear when the maximum likelihood estimate lies on the border of the parameter space.
Usually, this will happen when the "outer" parameter MLE is infinity (i.e. variance 0 in case of gamma and PVF).
For small enough values of \eqn{1/\theta} the log-likelihood
of the Cox model is returned to avoid such problems. This option can be tweaked in \code{emfrail_control()}.
}
\examples{
dat <- survival::rats


m1 <- emfrail(.data =  dat,
              .formula = Surv(time, status) ~  rx + sex + cluster(litter))
m1
summary(m1)
\dontrun{
# for the Inverse Gaussian distribution
m2 <- emfrail(.data =  dat,
              .formula = Surv(time, status) ~  rx + sex + cluster(litter),
              .distribution = emfrail_distribution(dist = "pvf"))
m2

# for the PVF distribution with m = 0.75
m3 <- emfrail(.data =  dat,
              .formula = Surv(time, status) ~  rx + sex + cluster(litter),
              .distribution = emfrail_distribution(dist = "pvf", pvfm = 0.75))
m3

# for the positive stable distribution
m4 <- emfrail(.data =  dat,
              .formula = Surv(time, status) ~  rx + sex + cluster(litter),
              .distribution = emfrail_distribution(dist = "stable"))
m4
}
# Compare marginal log-likelihoods
\dontrun{
models <- list(m1, m2, m3, m4)

logliks <- lapply(models,
                  function(x) -x$outer_m$value)

names(logliks) <- lapply(models,
                         function(x) with(x$.distribution,
                                          ifelse(dist == "pvf",
                                                 paste(dist, "/", pvfm),
                                                 dist))
)

logliks
}
# Draw the profile log-likelihood
\dontrun{
fr_var <- seq(from = 0.01, to = 1.4, length.out = 20)

# For gamma the variance is 1/theta (see parametrizations)
pll_gamma <- emfrail_pll(.data =  dat,
                         .formula = Surv(time, status) ~  rx + sex + cluster(litter),
                         .values = 1/fr_var )
 plot(fr_var, pll_gamma,
     type = "l",
     xlab = "Frailty variance",
     ylab = "Profile log-likelihood")
}
\dontrun{
# The same can be done with coxph, where variance is refered to as "theta"
pll_cph <- sapply(fr_var, function(fr)
  coxph(data =  dat, formula = Surv(time, status) ~ rx + sex + frailty(litter, theta = fr),
        method = "breslow")$history[[1]][[3]])

# Same for inverse gaussian
pll_if <- emfrail_pll(.data =  dat,
                      .formula = Surv(time, status) ~  rx + sex + cluster(litter),
                      .distribution = emfrail_distribution(dist = "pvf"),
                      .values = 1/fr_var )

# Same for pvf with a psoitive pvfm parameter
pll_pvf <- emfrail_pll(.data =  dat,
                       .formula = Surv(time, status) ~  rx + sex + cluster(litter),
                       .distribution = emfrail_distribution(dist = "pvf", pvfm = 1.5),
                       .values = 1/fr_var )

miny <- min(c(pll_gamma, pll_cph, pll_if, pll_pvf))
maxy <- max(c(pll_gamma, pll_cph, pll_if, pll_pvf))

plot(fr_var, pll_gamma,
     type = "l",
     xlab = "Frailty variance",
     ylab = "Profile log-likelihood",
     ylim = c(miny, maxy))
points(fr_var, pll_cph, col = 2)
lines(fr_var, pll_if, col = 3)
lines(fr_var, pll_pvf, col = 4)

legend(legend = c("gamma (emfrail)", "gamma (coxph)", "inverse gaussian", "pvf, m=1.5"),
       col = 1:4,
       lty = 1,
       x = 0,
       y = (maxy + miny)/2)
}
# Recurrent events
mod_rec <- emfrail(bladder1, Surv(start, stop, status) ~ treatment + cluster(id))
# The warnings appear from the Surv object, they also appear in coxph.

summary(mod_rec)

# Create a histogram of the estimated frailties

hist_frail(mod_rec)

# or, with ggplot:
\dontrun{
library(ggplot2)
sum_mod_rec <- summary(mod_rec)

ggplot(sum_mod_rec$frail, aes(x = z)) +
  geom_histogram() +
  ggtitle("Estimated frailties")

# Plot the frailty estimates with quantiles of the estimated distribution
ggplot(sum_mod_rec$frail, aes(x = id, y = z)) +
  geom_point() +
  ggtitle("Estimated frailties") +
  geom_errorbar(aes(ymin = lower_q, ymax = upper_q))

# We can do the same plot but with the ordered frailties:
ord <- order(sum_mod_rec$frail$z)
# we need new x coordinates for that:
ordering <- 1:length(ord)

ggplot(sum_mod_rec$frail[ord,], aes(x = ordering, y = z)) +
  geom_point() +
  ggtitle("Estimated frailties") +
  geom_errorbar(aes(ymin = lower_q, ymax = upper_q))

# How do we know which id is which one now?
# We can make an interactive plot with ggplotly
# To add text to elements we add id in aes()

library(plotly)
ggplotly(
  ggplot(sum_mod_rec$frail[ord,], aes(x = ordering, y = z)) +
    geom_point(aes(id = id)) +
    ggtitle("Estimated frailties") +
    geom_errorbar(aes(ymin = lower_q, ymax = upper_q, id = id))
)
}

# Plot marginal and conditional curves
# For recurrent events, the survival is not very meaningful

plot_pred(mod_rec, quantity = "cumhaz")
#The strong frailty "drags down" the intensity



# Left truncation

# N.B. This code takes a longer time to run

\dontrun{
# We simulate some data with truncation times
set.seed(1)
x <- sample(c(0,1), 5 * 300, TRUE)
u <- rep(rgamma(300,1,1), each = 5)
stime <- rexp(5*300, rate = u * exp(0.5 * x))
status <- ifelse(stime > 5, 0, 1)
stime[status] <- 5

ltime <- runif(5 * 300, min = 0, max = 2)
d <- data.frame(id = rep(1:300, each = 5),
                x = x,
                stime = stime,
                u = u,
                ltime = ltime,
                status = 1)
d_left <- d[d$stime > d$ltime,]

# The worst thing that can be done is to
# Ignore the left truncation:
mod_1 <- emfrail(d_left,
                 Surv(stime, status)~ x + cluster(id))

# The so-and-so thing is to consider the delayed entry time,
# But do not "update" the frailty distribution accordingly
mod_2 <- emfrail(d_left,
                 Surv(ltime, stime, status)~ x + cluster(id))

# This is identical with
mod_cox <- coxph(Surv(ltime, stime, status)~ x + frailty(id), data = d_left)


# The correct thing is to update the frailty.
mod_3 <- emfrail(d_left,
                 Surv(ltime, stime, status)~ x + cluster(id),
                 .distribution = emfrail_distribution(left_truncation = TRUE))

summary(mod_1)
summary(mod_2)
summary(mod_3)
}
}
\seealso{
\code{\link{plot_emfrail}} for plot functions directly available, \code{\link{emfrail_pll}} for calculating \eqn{\widehat{L}(\theta)} at specific values of \eqn{\theta},
\code{\link{summary.emfrail}} for transforming the \code{emfrail} object into a more human-readable format and for
visualizing the frailty (empirical Bayes) estimates,
\code{\link{predict.emfrail}} for calculating and visalizing conditional and marginal survival and cumulative
hazard curves.
}
