% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/design.R
\name{design}
\alias{design}
\alias{design.gp}
\alias{design.dgp}
\alias{design.bundle}
\title{Sequential design of a (D)GP emulator or a bundle of (D)GP emulators}
\usage{
design(
  object,
  N,
  x_cand,
  y_cand,
  n_cand,
  limits,
  int,
  f,
  freq,
  x_test,
  y_test,
  target,
  method,
  eval,
  verb,
  check_point,
  cores,
  ...
)

\method{design}{gp}(
  object,
  N,
  x_cand = NULL,
  y_cand = NULL,
  n_cand = 200,
  limits = NULL,
  int = FALSE,
  f = NULL,
  freq = c(1, 1),
  x_test = NULL,
  y_test = NULL,
  target = NULL,
  method = mice,
  eval = NULL,
  verb = TRUE,
  check_point = NULL,
  cores = 1,
  ...
)

\method{design}{dgp}(
  object,
  N,
  x_cand = NULL,
  y_cand = NULL,
  n_cand = 200,
  limits = NULL,
  int = FALSE,
  f = NULL,
  freq = c(1, 1),
  x_test = NULL,
  y_test = NULL,
  target = NULL,
  method = mice,
  eval = NULL,
  verb = TRUE,
  check_point = NULL,
  cores = 1,
  train_N = 100,
  refit_cores = 1,
  ...
)

\method{design}{bundle}(
  object,
  N,
  x_cand = NULL,
  y_cand = NULL,
  n_cand = 200,
  limits = NULL,
  int = FALSE,
  f = NULL,
  freq = c(1, 1),
  x_test = NULL,
  y_test = NULL,
  target = NULL,
  method = mice,
  eval = NULL,
  verb = TRUE,
  check_point = NULL,
  cores = 1,
  train_N = 100,
  refit_cores = 1,
  ...
)
}
\arguments{
\item{object}{can be one of the following:
\itemize{
\item the S3 class \code{gp}.
\item the S3 class \code{dgp}.
\item the S3 class \code{bundle}.
}}

\item{N}{the number of steps for the sequential design.}

\item{x_cand}{a matrix (with each row being a design point and column being an input dimension) that gives a candidate set
in which the next design point is determined. If \code{x_cand = NULL}, the candidate set will be generated using \code{n_cand},
\code{limits}, and \code{int}. Defaults to \code{NULL}.}

\item{y_cand}{a matrix (with each row being a simulator evaluation and column being an output dimension) that gives the realizations
from the simulator at input positions in \code{x_cand}. Defaults to \code{NULL}.}

\item{n_cand}{an integer that gives
\itemize{
\item the size of the candidate set in which the next design point is determined, if \code{x_cand = NULL};
\item the size of a sub-set to be sampled from the candidate set \code{x_cand} at each step of the sequential design to determine the next
design point, if \code{x_cand} is not \code{NULL}.
}

Defaults to \code{200}.}

\item{limits}{a two-column matrix that gives the ranges of each input dimension, or a vector of length two if there is only one
input dimension. If a vector is provided, it will be converted to a two-column row matrix. The rows of the matrix correspond to input
dimensions, and its first and second columns correspond to the minimum and maximum values of the input dimensions. If
\code{limits = NULL}, the ranges of input dimensions will be determined from the training data contained in \code{object}. This argument
is used when \code{x_cand = NULL} and \code{y_cand = NULL}. Defaults to \code{NULL}.}

\item{int}{a bool or a vector of bools that indicates if an input dimension is an integer type. If a bool is given, it will be applied to
all input dimensions. If a vector is provided, it should have a length equal to the input dimensions and will be applied to individual
input dimensions. Defaults to \code{FALSE}.}

\item{f}{an R function that represents the simulator. \code{f} needs to be specified with the following basic rules:
\itemize{
\item the first argument of the function should be a matrix with rows being different design points and columns being input dimensions.
\item the output of the function can either
\itemize{
\item a matrix with rows being different outputs (corresponding to the input design points) and columns being output dimensions. If there is
only one output dimension, the matrix still needs to be returned with a single column.
\item a list with the first element being the output matrix described above and, optionally, additional named elements which will update values
of any arguments with the same names passed via \code{...}. The list output can be useful if some additional arguments of \code{f} and \code{aggregate}
need to be updated after each step of the sequential design.
}
}}

\item{freq}{a vector of two integers with the first element giving the frequency (in number of steps) to re-fit the
emulator, and the second element giving the frequency to implement the emulator validation (for RMSE). Defaults to \code{c(1, 1)}.}

\item{x_test}{a matrix (with each row being an input testing data point and each column being an input dimension) that gives the testing
input data to evaluate the emulator after each step of the sequential design. Set to \code{NULL} for the LOO-based emulator validation.
Defaults to \code{NULL}. This argument is only used if \code{eval = NULL}.}

\item{y_test}{the testing output data that correspond to \code{x_test} for the emulator validation after each step of the sequential design:
\itemize{
\item if \code{object} is an instance of the \code{gp} class, \code{y_test} is a matrix with only one column and each row being an testing output data point.
\item if \code{object} is an instance of the \code{dgp} class, \code{y_test} is a matrix with its rows being testing output data points and columns being
output dimensions.
}

Set to \code{NULL} for the LOO-based emulator validation. Defaults to \code{NULL}. This argument is only used if \code{eval = NULL}.}

\item{target}{a numeric or a vector that gives the target RMSEs at which the sequential design is terminated. Defaults to \code{NULL}, in which
case the sequential design stops after \code{N} steps. See \emph{Note} section below for further information about \code{target}.}

\item{method}{an R function that give indices of designs points in a candidate set. The function must satisfy the following basic rules:
\itemize{
\item the first argument is an emulator object that can be either an instance of
\itemize{
\item the \code{gp} class (produced by \code{\link[=gp]{gp()}});
\item the \code{dgp} class (produced by \code{\link[=dgp]{dgp()}});
\item the \code{bundle} class (produced by \code{\link[=pack]{pack()}}).
}
\item the second argument is a matrix with rows representing a set of different design points.
\item the output of the function
\itemize{
\item is a vector of indices if the first argument is an instance of the \code{gp} class;
\item is a matrix of indices if the first argument is an instance of the \code{dgp} class. If there are different design points to be added with
respect to different outputs of the DGP emulator, the column number of the matrix should equal to the number of the outputs. If design
points are common to all outputs of the DGP emulator, the matrix should be single-columned. If more than one design points are determined
for a given output or for all outputs, the indices of these design points are placed in the matrix with extra rows.
\item is a matrix of indices if the first argument is an instance of the \code{bundle} class. Each row of the matrix gives the indices of the design
points to be added to individual emulators in the bundle.
}
}

See \code{\link[=alm]{alm()}}, \code{\link[=mice]{mice()}}, and \code{\link[=pei]{pei()}} for examples on customizing \code{method}. Defaults to \code{\link[=mice]{mice()}}.}

\item{eval}{an R function that calculates the customized evaluating metric of the emulator.The function must satisfy the following basic rules:
\itemize{
\item the first argument is an emulator object that can be either an instance of
\itemize{
\item the \code{gp} class (produced by \code{\link[=gp]{gp()}});
\item the \code{dgp} class (produced by \code{\link[=dgp]{dgp()}});
\item the \code{bundle} class (produced by \code{\link[=pack]{pack()}}).
}
\item the output of the function can be
\itemize{
\item a single metric value, if the first argument is an instance of the \code{gp} class;
\item a single metric value or a vector of metric values with the length equal to the number of output dimensions, if the first argument is an
instance of the \code{dgp} class;
\item a single metric value metric or a vector of metric values with the length equal to the number of emulators in the bundle, if the first
argument is an instance of the \code{bundle} class.
}
}

If no customized function is provided, the built-in evaluation metric, RMSE, will be calculated. Defaults to \code{NULL}. See \emph{Note} section below for further information.}

\item{verb}{a bool indicating if the trace information will be printed during the sequential design.
Defaults to \code{TRUE}.}

\item{check_point}{a vector of integers that indicates at which steps the sequential design will pause and ask for the confirmation
from the user if the sequential design should continue or be terminated. Set to \code{NULL} to suspend the manual intervention. Defaults
to \code{NULL}.}

\item{cores}{an integers that gives the number of cores to be used for emulator validations. If set to \code{NULL}, the number of cores is
set to \verb{(max physical cores available - 1)}. Defaults to \code{1}. This argument is only used if \code{eval = NULL}.}

\item{...}{any arguments (with names different from those of arguments used in \code{\link[=design]{design()}}) that are used by \code{f}, \code{method}, and \code{eval}
can be passed here. \code{\link[=design]{design()}} will pass relevant arguments to \code{f}, \code{method}, and \code{eval} based on the names of additional arguments provided.}

\item{train_N}{an integer or a vector of integers that gives the number of training iterations to be used to re-fit the DGP emulator at each step
of the sequential design:
\itemize{
\item If \code{train_N} is an integer, then at each step the DGP emulator will re-fitted (based on the frequency of re-fit specified in \code{freq}) with \code{train_N} iterations.
\item If \code{train_N} is a vector, then its size must be \code{N} even the re-fit frequency specified in \code{freq} is not one.
}

Defaults to \code{100}.}

\item{refit_cores}{the number of cores/workers to be used to re-fit GP components (in the same layer of a DGP emulator)
at each M-step during the re-fitting. If set to \code{NULL}, the number of cores is set to \verb{(max physical cores available - 1)}.
Only use multiple cores when there is a large number of GP components in different layers and optimization of GP components
is computationally expensive. Defaults to \code{1}.}
}
\value{
An updated \code{object} is returned with a slot called \code{design} that contains:
\itemize{
\item \emph{S} slots, named \verb{wave1, wave2,..., waveS}, that contain information of \emph{S} waves of sequential designs that have been applied to the emulator.
Each slot contains the following elements:
\itemize{
\item \code{N}, an integer that gives the numbers of steps implemented in the corresponding wave;
\item \code{rmse}, a matrix that gives the RMSEs of emulators constructed during the corresponding wave, if \code{eval = NULL};
\item \code{metric}, a matrix that gives the customized evaluating metric values of emulators constructed during the corresponding wave,
if a customized function is supplied to \code{eval};
\item \code{freq}, an integer that gives the frequency that the emulator validations are implemented during the corresponding wave.
\item \code{enrichment}, a vector of size \code{N} that gives the number of new design points added after each step of the sequential design (if \code{object} is
an instance of the \code{gp} or \code{dgp} class), or a matrix that gives the number of new design points added to emulators in a bundle after each step of
the sequential design (if \code{object} is an instance of the \code{bundle} class).
}

If \code{target} is not \code{NULL}, the following additional elements are also included:
\itemize{
\item \code{target}, the target RMSE(s) to stop the sequential design.
\item \code{reached}, a bool (if \code{object} is an instance of the \code{gp} or \code{dgp} class) or a vector of bools (if \code{object} is an instance of the \code{bundle}
class) that indicate if the target RMSEs are reached at the end of the sequential design.
}
\item a slot called \code{type} that gives the type of validations:
\itemize{
\item either LOO (\code{loo}) or OOS (\code{oos}) if \code{eval = NULL}. See \code{\link[=validate]{validate()}} for more information about LOO and OOS.
\item the customized R function provided to \code{eval}.
}
\item two slots called \code{x_test} and \code{y_test} that contain the data points for the OOS validation if the \code{type} slot is \code{oos}.
}

See \emph{Note} section below for further information.
}
\description{
This function implements the sequential design of a (D)GP emulator or a bundle of (D)GP emulators.
}
\details{
See further examples and tutorials at \url{https://mingdeyu.github.io/dgpsi-R/}.
}
\note{
\itemize{
\item The re-fitting and validation of an emulator are forced after the final step of a sequential design even \code{N} is not multiples of elements in \code{freq}.
\item Any \code{loo} or \code{oos} slot that already exists in \code{object} will be cleaned, and a new slot called \code{loo} or \code{oos} will be created in the returned object
depending on whether \code{x_test} and \code{y_test} are provided. The new slot gives the validation information of the emulator constructed in the final step of
the sequential design. See \code{\link[=validate]{validate()}} for more information about the slots \code{loo} and \code{oos}.
\item If \code{object} has previously been used by \code{\link[=design]{design()}} for sequential designs, the information of the current wave of the sequential design will replace
those of old waves and be contained in the returned object, unless the following conditions are met:
\itemize{
\item the validation type (\code{loo}, \code{oos}, or the customized function provided to \code{eval}) of the current wave of the sequential design is the same as the
validation types in previous waves, and
\item if the validation type is OOS, \code{x_test} and \code{y_test} in the current wave of the sequential design are identical to those in the previous waves.
}

When the above conditions are met, the information of the current wave of the sequential design will be added to
the \code{design} slot of the returned object under the name \code{waveS}.
\item If \code{object} is an instance of the \code{gp} class and \code{eval = NULL}, the matrix in the \code{rmse} slot is single-columned. If \code{object} is an instance of
the \code{dgp} or \code{bundle} class and \code{eval = NULL}, the matrix in the \code{rmse} slot can have multiple columns that correspond to different output dimensions
or different emulators in the bundle.
\item If \code{object} is an instance of the \code{gp} class and \code{eval = NULL}, \code{target} needs to be a single value giving the RMSE threshold. If \code{object} is an instance
of the \code{dgp} or \code{bundle} class and \code{eval = NULL}, \code{target} can be a vector of values that gives the RMSE thresholds for different output dimensions or
different emulators. If a single value is provided, it will be used as the RMSE threshold for all output dimensions (if \code{object} is an instance of the \code{dgp}) or all emulators
(if \code{object} is an instance of the \code{bundle}). If a customized function is supplied to \code{eval}, the user needs to ensure that the length of \code{target} is equal
to that of the output from \code{eval}.
\item When defining \code{f}, it is important to ensure that:
\itemize{
\item the column order of the first argument of \code{f} is consistent with the training input used for the emulator;
\item the column order of the output matrix of \code{f} is consistent with the order of emulator output dimensions (if \code{object} is an instance of the \code{dgp} class),
or the order of emulators placed in \code{object} (if \code{object} is an instance of the \code{bundle} class).
}
\item When defining \code{eval}, the output metric needs to be positive if \code{\link[=draw]{draw()}} is used with \code{log = T}. And one needs to ensure that a lower metric value indicates
a better emulation performance if \code{target} is set.
\item Any R vector detected in \code{x_test} and \code{y_test} will be treated as a column vector and automatically converted into a single-column
R matrix. Thus, if \code{x_test} or \code{y_test} is a single testing data point with multiple dimensions, it must be given as a matrix.
}
}
\examples{
\dontrun{

# load packages and the Python env
library(lhs)
library(dgpsi)
init_py()

# construct a 2D non-stationary function that takes a matrix as the input
f <- function(x) {
  sin(1/((0.7*x[,1,drop=F]+0.3)*(0.7*x[,2,drop=F]+0.3)))
}

# generate the initial design
X <- maximinLHS(5,2)
Y <- f(X)

# generate the validation data
validate_x <- maximinLHS(30,2)
validate_y <- f(validate_x)

# training a 2-layered DGP emulator with the initial design
m <- dgp(X, Y)

# specify the ranges of the input dimensions
lim_1 <- c(0, 1)
lim_2 <- c(0, 1)
lim <- rbind(lim_1, lim_2)

# 1st wave of the sequential design with 10 steps
m <- design(m, N=10, limits = lim, f = f, x_test = validate_x, y_test = validate_y)

# 2nd wave of the sequential design with 10 steps
m <- design(m, N=10, limits = lim, f = f, x_test = validate_x, y_test = validate_y)

# 3rd wave of the sequential design with 10 steps
m <- design(m, N=10, limits = lim, f = f, x_test = validate_x, y_test = validate_y)

# draw the design created by the sequential design
draw(m,'design')

# inspect the trace of RMSEs during the sequential design
draw(m,'rmse')

# reduce the number of imputations for faster OOS
m_faster <- set_imp(m, 10)

# plot the OOS validation with the faster DGP emulator
plot(m_faster, x_test = validate_x, y_test = validate_y)
}
}
