% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bundle_keras.R
\name{bundle.keras.engine.training.Model}
\alias{bundle.keras.engine.training.Model}
\title{Bundle a \code{keras} object}
\usage{
\method{bundle}{keras.engine.training.Model}(x, ...)
}
\arguments{
\item{x}{An object returned from modeling functions in the
\link[keras:keras-package]{keras} package.}

\item{...}{Not used in this bundler and included for compatibility with
the generic only. Additional arguments passed to this method will return
an error.}
}
\value{
A bundle object with subclass \code{bundled_keras}.

Bundles are a list subclass with two components:

\item{object}{An R object. Gives the output of native serialization
methods from the model-supplying package, sometimes with additional
classes or attributes that aid portability. This is often
a \link[base:raw]{raw} object.}
\item{situate}{A function. The \code{situate()} function is defined when
\code{\link[=bundle]{bundle()}} is called, though is a loose analogue of an \code{\link[=unbundle]{unbundle()}} S3
method for that object. Since the function is defined on \code{\link[=bundle]{bundle()}}, it
has access to references and dependency information that can
be saved alongside the \code{object} component. Calling \code{\link[=unbundle]{unbundle()}} on a
bundled object \code{x} calls \code{x$situate(x$object)}, returning the
unserialized version of \code{object}. \code{situate()} will also restore needed
references, such as server instances and environmental variables.}

Bundles are R objects that represent a "standalone" version of their
analogous model object. Thus, bundles are ready for saving to a file; saving
with \code{\link[base:readRDS]{base::saveRDS()}} is our recommended serialization strategy for bundles,
unless documented otherwise for a specific method.

To restore the original model object \code{x} in a new environment, load its
bundle with \code{\link[base:readRDS]{base::readRDS()}} and run \code{\link[=unbundle]{unbundle()}} on it. The output
of \code{\link[=unbundle]{unbundle()}} is a model object that is ready to \code{\link[=predict]{predict()}} on new data,
and other restored functionality (like plotting or summarizing) is supported
as a side effect only.

The bundle package wraps native serialization methods from model-supplying
packages. Between versions, those model-supplying packages may change their
native serialization methods, possibly introducing problems with re-loading
objects serialized with previous package versions. The bundle package does
not provide checks for these sorts of changes, and ought to be used in
conjunction with tooling for managing and monitoring model environments
like \link[vetiver:vetiver-package]{vetiver} or \link[renv:renv-package]{renv}.

See \code{vignette("bundle")} for more information on bundling and its motivation.
}
\description{
Bundling a model prepares it to be saved to a file and later
restored for prediction in a new R session. See the 'Value' section for
more information on bundles and their usage.
}
\details{
This bundler does not currently support custom keras extensions,
such as use of a \code{\link[keras:new-classes]{keras::new_layer_class()}} or custom metric function.
In such situations, consider using \code{\link[keras:with_custom_object_scope]{keras::with_custom_object_scope()}}.
}
\examples{
\dontshow{if (rlang::is_installed(c("keras")) && identical(Sys.getenv("NOT_CRAN"), "true")) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
# fit model and bundle ------------------------------------------------
library(keras)

set.seed(1)

mnist <- dataset_mnist()
x_train <- mnist$train$x
y_train <- mnist$train$y
x_test <- mnist$test$x
y_test <- mnist$test$y

x_train <- array_reshape(x_train, c(nrow(x_train), 784))
x_test <- array_reshape(x_test, c(nrow(x_test), 784))

x_train <- x_train / 255
x_test <- x_test / 255

y_train <- to_categorical(y_train, 10)
y_test <- to_categorical(y_test, 10)

mod <- keras_model_sequential()

mod \%>\%
  layer_dense(units = 128, activation = 'relu', input_shape = c(784)) \%>\%
  layer_dropout(rate = 0.4) \%>\%
  layer_dense(units = 64, activation = 'relu') \%>\%
  layer_dropout(rate = 0.3) \%>\%
  layer_dense(units = 10, activation = 'softmax')

mod \%>\% compile(
  loss = 'categorical_crossentropy',
  optimizer = optimizer_rmsprop(),
  metrics = c('accuracy')
)

mod \%>\% fit(
  x_train, y_train,
  epochs = 5, batch_size = 128,
  validation_split = 0.2,
  verbose = 0
)

mod_bundle <- bundle(mod)

# then, after saveRDS + readRDS or passing to a new session ----------
mod_unbundled <- unbundle(mod_bundle)

predict(mod_unbundled, x_test)
\dontshow{\}) # examplesIf}
}
\seealso{
This method wraps \code{\link[keras:save_model_tf]{keras::save_model_tf()}} and
\code{\link[keras:save_model_tf]{keras::load_model_tf()}}.

Other bundlers: 
\code{\link{bundle}()},
\code{\link{bundle.H2OAutoML}()},
\code{\link{bundle.bart}()},
\code{\link{bundle.luz_module_fitted}()},
\code{\link{bundle.model_fit}()},
\code{\link{bundle.model_stack}()},
\code{\link{bundle.recipe}()},
\code{\link{bundle.step_umap}()},
\code{\link{bundle.train}()},
\code{\link{bundle.workflow}()},
\code{\link{bundle.xgb.Booster}()}
}
\concept{bundlers}
