% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/backbone_from_projection.R
\name{backbone_from_projection}
\alias{backbone_from_projection}
\title{Extract the backbone from a weighted bipartite or hypgergraph projection}
\usage{
backbone_from_projection(
  B,
  alpha = 0.05,
  model = "sdsm",
  signed = FALSE,
  mtc = "none",
  missing_as_zero = FALSE,
  narrative = FALSE,
  trials = NULL,
  return = "backbone"
)
}
\arguments{
\item{B}{An unweighted bipartite network or hypergraph as an incidence matrix or \link[Matrix]{Matrix}, or as a bipartite \link[igraph]{igraph} object}

\item{alpha}{real: significance level of hypothesis test(s)}

\item{model}{string: backbone model, one of: \code{"sdsm"}, \code{"fdsm"}, \code{"fixedrow"}, \code{"fixedcol"}, or \code{"fixedfill"}}

\item{signed}{boolean: return a signed backbone}

\item{mtc}{string: type of Multiple Test Correction, either \code{"none"} or a method allowed by \code{\link[=p.adjust]{p.adjust()}}.}

\item{missing_as_zero}{boolean: treat missing edges as edges with zero weight and test them for significance}

\item{narrative}{boolean: display suggested text & citations}

\item{trials}{numeric: if \code{model = "fdsm"}, the number of graphs generated using fastball to approximate the edge weight distribution}

\item{return}{string: return either only the \code{"backbone"} or \code{"everything"}}
}
\value{
If \code{return = "backbone"}, a backbone in the same class as \code{B}. If \code{return = "everything"}, then the backbone
is returned as an element in a list that also includes the original network, raw projection, backbone, edgewise p-values, narrative
description, and original function call.
}
\description{
\code{backbone_from_projection()} extracts the unweighted backbone from the weighted projection of a bipartite network or hypergraph
}
\details{
The \code{backbone_from_projection} function extracts the backbone from the weighted projection of a bipartite network or hypergraph.
The backbone is an unweighted unipartite network of agents that contains only edges whose weights in the projection are statistically
significant. When \code{signed = FALSE}, the backbone contains edges that are statistically significantly strong under a one-tailed test.
When \code{signed = TRUE}, the backbone contains positive edges that are statistically significantly strong, and negative edges that are
statistically significantly weak, under a two-tailed test.

The \code{model} parameter controls the null model used to evaluate the statistical significance of edge weights. All available models
are \emph{statistical models} that are controlled by \code{alpha}, and differ in the constraints they impose on \code{B}:
\itemize{
\item \code{sdsm} (default) - The "Stochastic Degree Sequence Model" (SDSM; Neal et al., 2021) approximately constrains the agent and artifact degrees, and exactly constrains edges that are prohibited (weight = 10) or required (weight = 11; Neal & Neal, 2023)
\item \code{fdsm} - The "Fixed Degree Sequence Model" (Neal et al., 2021) exactly constrains the agent and artifact degrees
\item \code{fixedfill} - The "fixed fill" model (Neal et al., 2021) exactly constrains the total number of edges (i.e., sum)
\item \code{fixedrow} - The "fixed row" model (Neal et al., 2021) exactly constrains the agent degrees (i.e., row sums)
\item \code{fixedcol} - The "fixed column" model (Neal et al., 2021) exactly constrains the artifact degrees (i.e., column sums)
}

Although \code{backbone_from_projection} extracts the backbone from the weighted projection of a bipartite network or hypergraph,
the input \code{B} \emph{must be the bipartite network or hypergraph itself, and not the weighted projection}. This is necessary
because these backbone models use information in the bipartite network that is missing from the projection. The "agent" nodes that
appear in the projection must be represented by rows if \code{B} is an incidence matrix, or by \code{type = FALSE} nodes if \code{B}
is a bipartite igraph object. In either case, the source network must be binary (i.e., unweighted), unless \code{model = "sdsm"},
when "prohibited" edges can be represented with weight = 10 and "required" edges can be represented with weight = 11.
}
\examples{
#A binary bipartite network of 30 agents & 75 artifacts
#The agents form three communities
B <- rbind(cbind(matrix(rbinom(250,1,.8),10),
                 matrix(rbinom(250,1,.2),10),
                 matrix(rbinom(250,1,.2),10)),
           cbind(matrix(rbinom(250,1,.2),10),
                 matrix(rbinom(250,1,.8),10),
                 matrix(rbinom(250,1,.2),10)),
           cbind(matrix(rbinom(250,1,.2),10),
                 matrix(rbinom(250,1,.2),10),
                 matrix(rbinom(250,1,.8),10)))
B <- igraph::graph_from_biadjacency_matrix(B)

P <- igraph::bipartite_projection(B, which = "true")  #An ordinary weighted projection...
plot(P)                                               #...is a dense hairball

backbone <- backbone_from_projection(B)  #A backbone...
plot(backbone)                           #...is sparse with clear communities
}
\references{
package: {Neal, Z. P. (2025). backbone: An R Package to Extract Network Backbones. CRAN. \doi{10.32614/CRAN.package.backbone}}

sdsm-ec model: {Neal, Z. P. and Neal, J. W. (2023). Stochastic Degree Sequence Model with Edge Constraints (SDSM-EC) for Backbone Extraction. \emph{International Conference on Complex Networks and Their Applications, 12}, 127-136. \doi{10.1007/978-3-031-53468-3_11}}

all other models: {Neal, Z. P., Domagalski, R., and Sagan, B. (2021). Comparing Alternatives to the Fixed Degree Sequence Model for Extracting the Backbone of Bipartite Projections. \emph{Scientific Reports, 11}, 23929. \doi{10.1038/s41598-021-03238-3}}
}
