uiDIF <-
  navbarMenu("DIF/Fairness",
             # * SUMMARY ####
             "Description",
             tabPanel('About DIF',
                      h3('Differential Item Functioning / Item Fairness'),
                      p('Differential item functioning (DIF) occurs when people from different
                        social groups (commonly gender or ethnicity) with the same underlying true
                        ability have a different probability of answering the item correctly.
                        If item functions differently for two groups, it is potentially unfair.
                        In general, two type of DIF can be recognized: if the item has different
                        difficulty for given two groups with the same discrimination, ',
                        strong('uniform'), 'DIF is present (left figure). If the item has different
                        discrimination and possibly also different difficulty for given two groups, ',
                        strong('non-uniform'), 'DIF is present (right figure)'),
                      br(),
                      img(src = "fig_DIF_uniform.png",
                          style = "float: left; width: 32%; margin-right: 2%; margin-left: 16%; margin-bottom: 0.5em;"),
                      img(src = "fig_DIF_nonuniform.png",
                          style = "float: left; width: 32%; margin-right: 16%; margin-left: 2%; margin-bottom: 0.5em;"),
                      br(),
                      br()
             ),
             "----",
             "Used methods",
             # * TOTAL SCORES ####
             tabPanel("Total scores",
                      h3("Total scores"),
                      p('DIF is not about total scores! Two groups may have the same distribution of total scores, yet,
                        some item may function differently for two groups. Also, one of the groups may have signifficantly
                        lower total score, yet, it may happen that there is no DIF item ',
                        a('(Martinkova et al., 2017). ',
                          href = "https://www.lifescied.org/doi/10.1187/cbe.16-10-0307",
                          target = "_blank")),
                      h4("Summary of total scores for groups"),
                      tableOutput('resultsgroup'),
                      h4("Histograms of total scores for groups"),
                      sliderInput("inSlider2group", "Cut-score", min = 0, value = 1, max = 10,
                                  step = 1, animate = animationOptions(interval = 1200)),
                      p('For selected cut-score, blue part of histogram shows respondents with total score
                        above the cut-score, grey column shows respondents with total score equal
                        to cut-score and red part of histogram shows respondents below the cut-score.'),
                      splitLayout(cellWidths = c("50%", "50%"), plotlyOutput('histbyscoregroup0'), plotlyOutput('histbyscoregroup1')),
                      splitLayout(cellWidths = c("50%", "50%"), downloadButton("DP_histbyscoregroup0", label = "Download figure"),
                                  downloadButton("DP_histbyscoregroup1", label = "Download figure")),
                      br(),
                      h4("Comparison of total scores"),
                      p("Test for difference in total scores between reference and focal group is based od Welch two sample
                        t-test. ", br(),
                        strong("Explanation: "), strong("Diff. (CI)"), "- difference in means of total scores with 95% confidence interval, ",
                        strong("t-value"), "- test statistic, ", strong("df"), "- degrees of freedom, ",
                        strong("p-value"), "- if it is lower than 0.05, it means significant difference in total scores. "),
                      tableOutput("DIF_scores_ttest"),
                      br(),
                      h4("Selected R code"),
                      div(code(HTML("library(difNLR)<br>library(ggplot2)<br>library(moments)<br><br>#&nbsp;loading&nbsp;data<br>data(GMAT)<br>data&nbsp;<-&nbsp;GMAT[,&nbsp;1:20]<br>group&nbsp;<-&nbsp;GMAT[,&nbsp;\"group\"]<br><br>#&nbsp;total&nbsp;score&nbsp;calculation&nbsp;wrt&nbsp;group<br>score0&nbsp;<-&nbsp;apply(data,&nbsp;1,&nbsp;sum)[group&nbsp;==&nbsp;0]<br>score1&nbsp;<-&nbsp;apply(data,&nbsp;1,&nbsp;sum)[group&nbsp;==&nbsp;1]<br><br>#&nbsp;summary&nbsp;of&nbsp;total&nbsp;score&nbsp;<br>rbind(c(length(score0),&nbsp;min(score0),&nbsp;max(score0),&nbsp;mean(score0),&nbsp;median(score0),&nbsp;sd(score0),&nbsp;skewness(score0),&nbsp;kurtosis(score0)),<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;c(length(score1),&nbsp;min(score1),&nbsp;max(score1),&nbsp;mean(score1),&nbsp;median(score1),&nbsp;sd(score1),&nbsp;skewness(score1),&nbsp;kurtosis(score1)))<br><br>#&nbsp;colors&nbsp;by&nbsp;cut-score&nbsp;wrt&nbsp;group<br>cut&nbsp;<-&nbsp;12&nbsp;#&nbsp;cut-score&nbsp;<br>color0&nbsp;<-&nbsp;c(rep(\"red\",&nbsp;cut&nbsp;-&nbsp;min(score0)),&nbsp;\"gray\",&nbsp;rep(\"blue\",&nbsp;max(score0)&nbsp;-&nbsp;cut))<br>color1&nbsp;<-&nbsp;c(rep(\"red\",&nbsp;cut&nbsp;-&nbsp;min(score1)),&nbsp;\"gray\",&nbsp;rep(\"blue\",&nbsp;max(score1)&nbsp;-&nbsp;cut))<br><br>#&nbsp;histogram&nbsp;for&nbsp;reference&nbsp;group<br>ggplot(data&nbsp;=&nbsp;data.frame(score0),&nbsp;aes(score0))&nbsp;+&nbsp;<br>&nbsp;&nbsp;geom_histogram(binwidth&nbsp;=&nbsp;1,&nbsp;fill&nbsp;=&nbsp;color0,&nbsp;col&nbsp;=&nbsp;\"black\")&nbsp;+&nbsp;<br>&nbsp;&nbsp;xlab(\"Total&nbsp;score\")&nbsp;+&nbsp;<br>&nbsp;&nbsp;ylab(\"Number&nbsp;of&nbsp;respondents\")&nbsp;+&nbsp;<br>&nbsp;&nbsp;ggtitle(\"Reference&nbsp;group\")&nbsp;+&nbsp;<br>&nbsp;&nbsp;theme_app()<br><br>#&nbsp;histogram&nbsp;for&nbsp;focal&nbsp;group<br>ggplot(data&nbsp;=&nbsp;data.frame(score1),&nbsp;aes(score1))&nbsp;+&nbsp;<br>&nbsp;&nbsp;geom_histogram(binwidth&nbsp;=&nbsp;1,&nbsp;fill&nbsp;=&nbsp;color1,&nbsp;col&nbsp;=&nbsp;\"black\")&nbsp;+&nbsp;<br>&nbsp;&nbsp;xlab(\"Total&nbsp;score\")&nbsp;+&nbsp;<br>&nbsp;&nbsp;ylab(\"Number&nbsp;of&nbsp;respondents\")&nbsp;+&nbsp;<br>&nbsp;&nbsp;ggtitle(\"Focal&nbsp;group\")&nbsp;+&nbsp;<br>&nbsp;&nbsp;theme_app()<br><br>#&nbsp;t-test&nbsp;to&nbsp;compare&nbsp;total&nbsp;scores<br>t.test(score0,&nbsp;score1)"))),
                      br()
             ),
             # * DELTA PLOTS ####
             tabPanel("Delta plots",
                      h3("Delta plot"),
                      p('Delta plot (Angoff & Ford, 1973) compares the proportions of correct answers per
                        item in the two groups. It displays non-linear transformation of these proportions using
                        quantiles of standard normal distributions (so called delta scores) for each item for the two
                        genders in a scatterplot called diagonal plot or delta plot (see Figure). Item is under
                        suspicion of DIF if the delta point considerably departs from the diagonal. The detection
                        threshold is either fixed to value 1.5 or based on bivariate normal approximation (Magis &
                        Facon, 2012).'),

                      radioButtons('type_threshold', 'Threshold',
                                   list("Fixed", "Normal")
                      ),
                      checkboxInput('puri_DP', 'Item purification', FALSE),
                      conditionalPanel(
                        condition = "input.puri_DP",
                        selectInput("puri_DP_type", "Purification method",
                                    c("IPP1" = "IPP1",
                                      "IPP2" = "IPP2",
                                      "IPP3" = "IPP3"
                                    ),
                                    selected = "IPP1")),
                      plotOutput('deltaplot'),
                      downloadButton("DP_deltaplot", label = "Download figure"),
                      br(),
                      br(),
                      verbatimTextOutput("dp_text_normal"),
                      br(),
                      h4("Selected R code"),
                      div(code('library(deltaPlotR)'),
                          br(),
                          code('library(difNLR)'),
                          br(),
                          code('data(GMAT)'),
                          br(),
                          code('data <- GMAT[, 1:20]'),
                          br(),
                          code('group <- GMAT[, "group"]'),
                          br(),
                          br(),
                          code('# Delta scores with fixed threshold'),
                          br(),
                          code('deltascores <- deltaPlot(data.frame(data, group), group = "group",
                               focal.name = 1, thr = 1.5)'),
                          br(),
                          code('deltascores'),
                          br(),
                          code('# Delta plot'),
                          br(),
                          code('diagPlot(deltascores, thr.draw = T)'),
                          br(),
                          br(),
                          code('# Delta scores with normal threshold'),
                          br(),
                          code('deltascores <- deltaPlot(data.frame(data, group), group = "group",
                               focal.name = 1, thr = "norm", purify = F)'),
                          br(),
                          code('deltascores'),
                          br(),
                          code('# Delta plot'),
                          br(),
                          code('diagPlot(deltascores, thr.draw = T)')),
                      br()
             ),
             # * MANTEL-HAENSZEL ####
             tabPanel("Mantel-Haenszel",
                      tabsetPanel(
                        # Summary
                        tabPanel("Summary",
                                 h3("Mantel-Haenszel test"),
                                 p('Mantel-Haenszel test is DIF detection method based on contingency
                                   tables that are calculated for each level of total score (Mantel &
                                   Haenszel, 1959).'),
                                 h4('Summary table'),
                                 p('Here you can select ',strong('correction method'),' for multiple comparison or', strong('item purification.')),
                                 selectInput("correction_method_MZ_print", "Correction method",
                                             choices = c("Benjamini-Hochberg" = "BH",
                                                         "Benjamini-Yekutieli" = "BY",
                                                         "Holm" = "holm",
                                                         "Hochberg" = "hochberg",
                                                         "Hommel" = "hommel",
                                                         "None" = "none"),
                                             selected = "none"),
                                 checkboxInput('puri_MH', 'Item purification', FALSE),
                                 verbatimTextOutput("print_DIF_MH"),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(difNLR)'),
                                     br(),
                                     code('library(difR)'),
                                     br(),
                                     code('data(GMAT)'),
                                     br(),
                                     code('data <- GMAT[, 1:20]'),
                                     br(),
                                     code('group <- GMAT[, "group"]'),
                                     br(),
                                     br(),
                                     code('# Mantel-Haenszel test'),
                                     br(),
                                     code('fit <- difMH(Data = data, group = group, focal.name = 1,
                                          p.adjust.method = "none", purify = F)'),
                                     br(),
                                     code('fit')),
                                 br()
                        ),
                        tabPanel('Items',
                                 h3("Mantel-Haenszel test"),
                                 p('Mantel-Haenszel test is DIF detection method based on contingency
                                   tables that are calculated for each level of total score (Mantel &
                                   Haenszel, 1959).'),
                                 h4('Contingency tables and odds ratio calculation'),
                                 fluidPage(div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                               sliderInput("difMHSlider_item",
                                                           "Item",
                                                           animate = animationOptions(interval = 1200),
                                                           min = 1,
                                                           max = 10,
                                                           value = 1,
                                                           step = 1)),
                                           div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                           div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                               sliderInput("difMHSlider_score",
                                                           "Cut-score",
                                                           min = 0,
                                                           max = 10,
                                                           value = 1,
                                                           step = 1))),
                                 fluidRow(column(12, align = "center", tableOutput('table_DIF_MH'))),
                                 uiOutput('ORcalculation'),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(difNLR)'),
                                     br(),
                                     code('library(difR)'),
                                     br(),
                                     code('data(GMAT)'),
                                     br(),
                                     code('data <- GMAT[, 1:20]'),
                                     br(),
                                     code('group <- GMAT[, "group"]'),
                                     br(),
                                     br(),
                                     code('# Contingency table for item 1 and score 12'),
                                     br(),
                                     code('df <- data.frame(data[, 1], group)'),
                                     br(),
                                     code('colnames(df) <- c("Answer", "Group")'),
                                     br(),
                                     code('df$Answer <- relevel(factor(df$Answer, labels = c("Incorrect", "Correct")), "Correct")'),
                                     br(),
                                     code('df$Group <- factor(df$Group, labels = c("Reference Group", "Focal Group"))'),
                                     br(),
                                     code('score <- apply(data, 1, sum)'),
                                     br(),

                                     code('df <- df[score == 12, ]'),
                                     br(),

                                     code('tab <- dcast(data.frame(xtabs(~ Group + Answer, data = df)),
                                          Group ~ Answer,
                                          value.var = "Freq",
                                          margins = T,
                                          fun = sum)'),
                                     br(),
                                     code('tab'),
                                     br(),

                                     code('# Mantel-Haenszel estimate of OR'),
                                     br(),
                                     code('fit <- difMH(Data = data, group = group, focal.name = 1,
                                          p.adjust.method = "none", purify = F)'),
                                     br(),
                                     code('fit$alphaMH')),
                                 br()
                        )
                      )
             ),
             # * LOGISTIC ####
             tabPanel("Logistic regression",
                      tabsetPanel(
                        # ** Summary ####
                        tabPanel('Summary',
                                 h3('Logistic regression on total scores'),
                                 p('Logistic regression allows for detection of uniform and non-uniform DIF (Swaminathan & Rogers, 1990) by adding a group
                                   specific intercept', strong('\\(b_{2}\\)'), '(uniform DIF) and group specific interaction', strong('\\(b_{3}\\)'), '(non-uniform DIF) into model and
                                   by testing for their significance.'),
                                 h4("Equation"),
                                 ('$$\\mathrm{P}\\left(Y_{ij} = 1 | X_i, G_i, b_0, b_1, b_2, b_3\\right) = \\frac{e^{b_0 + b_1 X_i + b_2 G_i + b_3 X_i G_i}}{1+e^{b_0 + b_1 X_i + b_2 G_i + b_3 X_i G_i}} $$'),
                                 h4("Summary table"),
                                 p('Here you can choose what ',strong('type'),' of DIF to test. You can also select ',strong('correction method'),' for multiple comparison or', strong('item purification.')),
                                 fluidPage(div(style = "display: inline-block; vertical-align: top; width: 27%; ",
                                               withMathJax(),
                                               radioButtons(inputId = 'type_print_DIF_logistic',
                                                            label = 'Type',
                                                            choices = c("\\(H_{0}\\): Any DIF vs. \\(H_{1}\\): No DIF" = 'both',
                                                                        "\\(H_{0}\\): Uniform DIF vs. \\(H_{1}\\): No DIF" = 'udif',
                                                                        "\\(H_{0}\\): Non-Uniform DIF vs. \\(H_{1}\\): Uniform DIF" = 'nudif'),
                                                            selected = 'both')),
                                           div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                           div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                               selectInput("correction_method_logSummary", "Correction method",
                                                           choices = c("Benjamini-Hochberg" = "BH",
                                                                       "Benjamini-Yekutieli" = "BY",
                                                                       "Holm" = "holm",
                                                                       "Hochberg" = "hochberg",
                                                                       "Hommel" = "hommel",
                                                                       "None" = "none"),
                                                           selected = "none"),
                                               checkboxInput(inputId = 'puri_LR',
                                                             label = 'Item purification',
                                                             value = FALSE))),
                                 uiOutput("DIF_logistic_na_alert"),
                                 verbatimTextOutput('print_DIF_logistic'),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(difNLR)'),
                                     br(),
                                     code('library(difR)'),
                                     br(),
                                     code('data(GMAT)'),
                                     br(),
                                     code('data <- GMAT[, 1:20]'),
                                     br(),
                                     code('group <- GMAT[, "group"]'),
                                     br(),
                                     br(),

                                     code('# Logistic regression DIF detection method'),
                                     br(),
                                     code('fit <- difLogistic(Data = data, group = group, focal.name = 1,
                                          type = "both",
                                          p.adjust.method = "none",
                                          purify = F)'),
                                     br(),
                                     code('fit')),
                                 br()
                        ),
                        # ** Items ####
                        tabPanel('Items',
                                 h3('Logistic regression on total scores'),
                                 p('Logistic regression allows for detection of uniform and non-uniform DIF (Swaminathan & Rogers, 1990) by adding a group
                                   specific intercept', strong('\\(b_{2}\\)'), '(uniform DIF) and group specific interaction', strong('\\(b_{3}\\)'), '(non-uniform DIF) into model and
                                   by testing for their significance.'),
                                 h4("Plot with estimated DIF logistic curve"),
                                 p('Here you can choose what ',strong('type'),' of DIF to test. You can also select ',strong('correction method'),' for multiple comparison or', strong('item purification.')),
                                 fluidPage(div(style = "display: inline-block; vertical-align: top; width: 27%; ",
                                               radioButtons(inputId = 'type_plot_DIF_logistic',
                                                            label = 'Type',
                                                            choices = c("\\(H_{0}\\): Any DIF vs. \\(H_{1}\\): No DIF" = 'both',
                                                                        "\\(H_{0}\\): Uniform DIF vs. \\(H_{1}\\): No DIF" = 'udif',
                                                                        "\\(H_{0}\\): Non-Uniform DIF vs. \\(H_{1}\\): Uniform DIF" = 'nudif'),
                                                            selected = 'both')),
                                           div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                           div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                               selectInput(inputId = "correction_method_logItems",
                                                           label = "Correction method",
                                                           choices = c("Benjamini-Hochberg" = "BH",
                                                                       "Benjamini-Yekutieli" = "BY",
                                                                       "Holm" = "holm",
                                                                       "Hochberg" = "hochberg",
                                                                       "Hommel" = "hommel",
                                                                       "None" = "none"),
                                                           selected = "none"),
                                               checkboxInput(inputId = 'puri_LR_plot',
                                                             label = 'Item purification',
                                                             value = FALSE)),
                                           div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                           div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                               sliderInput("diflogSlider", "Item",
                                                           min = 1,
                                                           value = 1,
                                                           max = 10,
                                                           step = 1,
                                                           animate = animationOptions(interval = 1600)))),
                                 p('Points represent proportion of correct answer with respect to total score.
                                   Their size is determined by count of respondents who achieved given level of
                                   total score with respect to the group membership.'),
                                 p('NOTE: Plots and tables are based on DIF logistic procedure without any correction method. '),
                                 plotOutput('plot_DIF_logistic'),
                                 downloadButton("DP_plot_DIF_logistic", label = "Download figure"),
                                 h4("Equation"),
                                 ('$$\\mathrm{P}\\left(Y_{ij} = 1 | X_i, G_i, b_0, b_1, b_2, b_3\\right) = \\frac{e^{b_0 + b_1 X_i + b_2 G_i + b_3 X_i G_i}}{1+e^{b_0 + b_1 X_i + b_2 G_i + b_3 X_i G_i}} $$'),
                                 h4("Table of parameters"),
                                 fluidRow(column(12, align = "center", tableOutput('tab_coef_DIF_logistic'))),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(difNLR)'),
                                     br(),
                                     code('library(difR)'),
                                     br(),
                                     code('data(GMAT)'),
                                     br(),
                                     code('data <- GMAT[, 1:20]'),
                                     br(),
                                     code('group <- GMAT[, "group"]'),
                                     br(),
                                     br(),

                                     code('# Logistic regression DIF detection method'),
                                     br(),
                                     code('fit <- difLogistic(Data = data, group = group, focal.name = 1,
                                          type = "both",
                                          p.adjust.method = "none", purify = F)'),
                                     br(),
                                     code('fit'),
                                     br(),

                                     code('# Plot of characteristic curve for item 1'),
                                     br(),
                                     code('plotDIFLogistic(data, group,
                                          type = "both",
                                          item =  1,
                                          IRT = F,
                                          p.adjust.method = "none",
                                          purify = F)'),
                                     br(),
                                     code('# Coefficients'),
                                     br(),
                                     code('fit$logitPar')),
                                 br()
                        )
                      )
             ),

             # # * LOGISTIC Z ####
             # tabPanel("Logistic IRT Z",
             #          tabsetPanel(
             #            # ** Summary ####
             #            tabPanel('Summary',
             #                     h3('Logistic regression on standardized total scores with IRT parameterization'),
             #                     p('Logistic regression allows for detection of uniform and non-uniform DIF (Swaminathan & Rogers, 1990) by adding a group
             #                       specific intercept', strong('bDIF'), '(uniform DIF) and group specific interaction', strong('aDIF'), '(non-uniform DIF) into model and
             #                       by testing for their significance.'),
             #                     h4("Equation"),
             #                     ('$$\\mathrm{P}\\left(Y_{ij} = 1 | Z_i, G_i, a_j, b_j, a_{\\text{DIF}j}, b_{\\text{DIF}j}\\right) = \\frac{e^{\\left(a_j + a_{\\text{DIF}j} G_i\\right)
             #                      \\left(Z_i -\\left(b_j + b_{\\text{DIF}j} G_i\\right)\\right)}}{1+e^{\\left(a_j + a_{\\text{DIF}j} G_i\\right)
             #                      \\left(Z_i -\\left(b_j + b_{\\text{DIF}j} G_i\\right)\\right)}} $$'),
             #                     h4('Summary table'),
             #                     p('Here you can choose what', strong('type'), 'of DIF to test. You can also select ',
             #                       strong('correction method'), 'for multiple comparison.'),
             #                     fluidPage(div(style = "display: inline-block; vertical-align: top; width: 27%; ",
             #                                   radioButtons(inputId = 'type_print_DIF_logistic_IRT_Z',
             #                                                label = 'Type',
             #                                                choices = c("H0: Any DIF vs. H1: No DIF" = 'both',
             #                                                            "H0: Uniform DIF vs. H1: No DIF" = 'udif',
             #                                                            "H0: Non-Uniform DIF vs. H1: Uniform DIF" = 'nudif'),
             #                                                selected = 'both')),
             #                               div(style = "display: inline-block; vertical-align: top; width: 5%; "),
             #                               div(style = "display: inline-block; vertical-align: top; width: 20%; ",
             #                                   selectInput(inputId = "correction_method_logZSummary",
             #                                               label = "Correction method",
             #                                               choices = c("BH" = "BH",
             #                                                           "Holm" = "holm",
             #                                                           "Hochberg" = "hochberg",
             #                                                           "Hommel" = "hommel",
             #                                                           "BY" = "BY",
             #                                                           "FDR" = "fdr",
             #                                                           "none" = "none"),
             #                                               selected = "none"))),
             #                     verbatimTextOutput('print_DIF_logistic_IRT_Z'),
             #                     br(),
             #                     h4("Selected R code"),
             #                     div(code('library(difNLR)'),
             #                         br(),
             #                         code('library(difR)'),
             #                         br(),
             #                         code('data(GMAT)'),
             #                         br(),
             #                         code('data <- GMAT[, 1:20]'),
             #                         br(),
             #                         code('group <- GMAT[, "group"]'),
             #                         br(),
             #                         code('scaled.score <- scale(score)'),
             #                         br(),
             #                         br(),
             #                         code('# Logistic regression DIF detection method'),
             #                         br(),
             #                         code('fit <- difLogistic(Data = data, group = group, focal.name = 1,
             #                              type = "both",
             #                              match = scaled.score,
             #                              p.adjust.method = "none",
             #                              purify = F)'),
             #                         br(),
             #                         code('fit')),
             #                     br()
             #                     ),
             #            # ** Items ####
             #            tabPanel('Items',
             #                     h3('Logistic regression on standardized total scores with IRT parameterization'),
             #                     p('Logistic regression allows for detection of uniform and non-uniform DIF by adding a group
             #                       specific intercept', strong('bDIF'), '(uniform DIF) and group specific interaction', strong('aDIF'), '(non-uniform DIF) into model and
             #                       by testing for their significance.'),
             #                     h4("Plot with estimated DIF logistic curve"),
             #                     p('Here you can choose what', strong('type'), 'of DIF to test. You can also select ',
             #                       strong('correction method'), 'for multiple comparison.'),
             #                     fluidPage(div(style = "display: inline-block; vertical-align: top; width: 27%; ",
             #                                   radioButtons(inputId = 'type_plot_DIF_logistic_IRT_Z',
             #                                                label = 'Type',
             #                                                choices = c("H0: Any DIF vs. H1: No DIF" = 'both',
             #                                                            "H0: Uniform DIF vs. H1: No DIF" = 'udif',
             #                                                            "H0: Non-Uniform DIF vs. H1: Uniform DIF" = 'nudif'),
             #                                                selected = 'both')),
             #                               div(style = "display: inline-block; vertical-align: top; width: 5%; "),
             #                               div(style = "display: inline-block; vertical-align: top; width: 20%; ",
             #                                   selectInput(inputId = "correction_method_logZItems",
             #                                               label = "Correction method",
             #                                               choices = c("BH" = "BH",
             #                                                           "Holm" = "holm",
             #                                                           "Hochberg" = "hochberg",
             #                                                           "Hommel" = "hommel",
             #                                                           "BY" = "BY",
             #                                                           "FDR" = "fdr",
             #                                                           "none" = "none"),
             #                                               selected = "none")),
             #                               div(style = "display: inline-block; vertical-align: top; width: 5%; "),
             #                               div(style = "display: inline-block; vertical-align: top; width: 20%; ",
             #                                   sliderInput("diflog_irtSlider", "Item",
             #                                               min = 1,
             #                                               value = 1,
             #                                               max = 10,
             #                                               step = 1,
             #                                               animate = animationOptions(interval = 1200)))),
             #                     p('Points represent proportion of correct answer with respect to standardized
             #                       total score. Their size is determined by count of respondents who achieved
             #                       given level of standardized total score with respect to the group membership.'),
             #                     p('NOTE: Plots and tables are based on DIF logistic procedure without any correction method. '),
             #                     plotOutput('plot_DIF_logistic_IRT_Z'),
             #                     downloadButton("DP_plot_DIF_logistic_IRT_Z", label = "Download figure"),
             #                     h4("Equation"),
             #                     ('$$\\mathrm{P}\\left(Y_{ij} = 1 | Z_i, G_i, a_j, b_j, a_{\\text{DIF}j}, b_{\\text{DIF}j}\\right) =
             #                      \\frac{e^{\\left(a_j + a_{\\text{DIF}j} G_i\\right)\\left(Z_i -\\left(b_j + b_{\\text{DIF}j} G_i\\right)\\right)}}
             #                      {1+e^{\\left(a_j + a_{\\text{DIF}j} G_i\\right)\\left(Z_i -\\left(b_j + b_{\\text{DIF}j} G_i\\right)\\right)}} $$'),
             #                     h4("Table of parameters"),
             #                     fluidRow(column(12, align = "center", tableOutput('tab_coef_DIF_logistic_IRT_Z'))),
             #                     br(),
             #                     h4("Selected R code"),
             #                     div(code('library(difNLR)'),
             #                         br(),
             #                         code('library(difR)'),
             #                         br(),
             #                         code('data(GMAT)'),
             #                         br(),
             #                         code('data <- GMAT[, 1:20]'),
             #                         br(),
             #                         code('group <- GMAT[, "group"]'),
             #                         br(),
             #                         code('scaled.score <- scale(score)'),
             #                         br(),
             #                         br(),
             #                         code('# Logistic regression DIF detection method'),
             #                         br(),
             #                         code('fit <- difLogistic(Data = data, group = group, focal.name = 1,
             #                              type = "both",
             #                              match = scaled.score,
             #                              p.adjust.method = "none",
             #                              purify = F)'),
             #                         br(),
             #                         code('fit'),
             #                         br(),
             #
             #                         code('# Plot of characteristic curve for item 1'),
             #                         br(),
             #                         code('plotDIFLogistic(data, group,
             #                              type = "both",
             #                              item =  1,
             #                              IRT = T,
             #                              p.adjust.method = "BH")'),
             #                         br(),
             #                         code('# Coefficients for item 1 - recalculation'),
             #                         br(),
             #                         code('coef_old <- fit$logitPar[1, ]'),
             #                         br(),
             #                         code('coef <- c()'),
             #                         br(),
             #                         code('# a = b1, b = -b0/b1, adif = b3, bdif = -(b1b2-b0b3)/(b1(b1+b3))'),
             #                         br(),
             #                         code('coef[1] <- coef_old[2]'),
             #                         br(),
             #                         code('coef[2] <- -(coef_old[1] / coef_old[2])'),
             #                         br(),
             #                         code('coef[3] <- coef_old[4]'),
             #                         br(),
             #                         code('coef[4] <- -(coef_old[2] * coef_old[3] + coef_old[1] * coef_old[4] ) /
             #                              (coef_old[2] * (coef_old[2] + coef_old[4]))')),
             #                     br()
             #                         )
             #                     )
             #            ),
             # * GENERALIZED LOGISTIC ####
             tabPanel("Generalized logistic",
                      tabsetPanel(
                        # ** Summary ####
                        tabPanel('Summary',
                                 h3('Generalized logistic regression'),
                                 p('Generalized logistic regression models can be seen as proxies of IRT models for
                                   DIF detection using standardized total score as estimate of knowledge.
                                   They can allow for nonzero lower asymptote - pseudoguessing \\(c\\)',
                                   a('(Drabinova & Martinkova, 2017) ',
                                     href = "http://onlinelibrary.wiley.com/doi/10.1111/jedm.12158/full",
                                     target = "_blank"),
                                   'or upper asymptote lower than one - inattention \\(d\\). Similarly to logistic
                                   regression, also its extensions provide detection of uniform and non-uniform DIF by
                                   letting difficulty parameter \\(b\\) (uniform) and discrimination parameter \\(a\\)
                                   (non-uniform) differ for groups and by testing for significance difference in their
                                   values. Moreover, these extensions allow for testing differences in pseudoguessing and
                                   inattention parameters. '),
                                 withMathJax(),
                                 p('With ', strong('model'), 'you can specify what parameters should be kept the same for
                                   both groups and what parameters should differ. The notation is similar to IRT models.
                                   In 3PL and 4PL models abbreviations \\(c_{g}\\) or \\(d_{g}\\) mean that parameters \\(c\\) or \\(d\\) are the same for
                                   both groups. With ', strong('type'), 'you can choose parameters in which difference between
                                   groups should be tested.'),
                                 h4("Equation"),
                                 p("Displayed equation is based on selected model"),
                                 fluidRow(column(12, align = "center", uiOutput("DIF_NLR_equation_print"))),
                                 h4("Summary table"),
                                 p('Here you can choose what ',strong('type'),' of DIF to test. You can also select ',strong('correction method'),' for multiple comparison or', strong('item purification.')),
                                 fluidRow(column(3,
                                                 selectInput(inputId = "DIF_NLR_model_print",
                                                             label = "Model",
                                                             choices = c("Rasch" = "Rasch",
                                                                         "1PL" = "1PL",
                                                                         "2PL" = "2PL",
                                                                         "3PLcg" = "3PLcg",
                                                                         "3PLdg" = "3PLdg",
                                                                         "3PLc" = "3PLc",
                                                                         "3PLd" = "3PLd",
                                                                         "4PLcgdg" = "4PLcgdg",
                                                                         "4PLcgd" = "4PLcgd",
                                                                         "4PLcdg" = "4PLcdg",
                                                                         "4PL" = "4PL"),
                                                             selected = "3PLcg")),
                                          column(1,
                                                 checkboxGroupInput(inputId = 'DIF_NLR_type_print',
                                                                    label = 'Type',
                                                                    choices = c("\\(a\\)" = "a",
                                                                                "\\(b\\)" = "b",
                                                                                "\\(c\\)" = "c",
                                                                                "\\(d\\)" = "d"),
                                                                    selected = c("\\(a\\)", "b"))),
                                          column(3,
                                                 selectInput(inputId = "DIF_NLR_correction_method_print",
                                                             label = "Correction method",
                                                             choices = c("Benjamini-Hochberg" = "BH",
                                                                         "Benjamini-Yekutieli" = "BY",
                                                                         "Holm" = "holm",
                                                                         "Hochberg" = "hochberg",
                                                                         "Hommel" = "hommel",
                                                                         "None" = "none"),
                                                             selected = "none"),
                                                 checkboxInput(inputId = 'DIF_NLR_purification_print',
                                                               label = 'Item purification',
                                                               value = FALSE))
                                 ),
                                 uiOutput("DIF_NLR_na_alert"),
                                 verbatimTextOutput('print_DIF_NLR'),
                                 br(),
                                 h4("Selected R code"),
                                 div(code(HTML("library(difNLR)&nbsp;<br><br>#&nbsp;loading&nbsp;data&nbsp;<br>data(GMAT)&nbsp;<br>Data&nbsp;<-&nbsp;GMAT[,&nbsp;1:20]&nbsp;<br>group&nbsp;<-&nbsp;GMAT[,&nbsp;\"group\"]&nbsp;<br><br>#&nbsp;generalized&nbsp;logistic&nbsp;regression&nbsp;DIF&nbsp;method&nbsp;<br>#&nbsp;using&nbsp;3PL&nbsp;model&nbsp;with&nbsp;the&nbsp;same&nbsp;guessing&nbsp;parameter&nbsp;for&nbsp;both&nbsp;groups&nbsp;<br>fit&nbsp;<-&nbsp;difNLR(Data&nbsp;=&nbsp;Data,&nbsp;group&nbsp;=&nbsp;group,&nbsp;focal.name&nbsp;=&nbsp;1,&nbsp;model&nbsp;=&nbsp;\"3PLcg\",&nbsp;type&nbsp;=&nbsp;\"both\",&nbsp;p.adjust.method&nbsp;=&nbsp;\"none\")&nbsp;<br>fit"))),
                                 br()
                        ),
                        # ** Items ####
                        tabPanel('Items',
                                 h3('Generalized logistic regression'),
                                 p('Generalized logistic regression models can be seen as proxies of IRT models for
                                   DIF detection using standardized total score as estimate of knowledge.
                                   They can allow for nonzero lower asymptote - pseudoguessing \\(c\\)',
                                   a('(Drabinova & Martinkova, 2017) ',
                                     href = "https://doi.org/10.1111/jedm.12158",
                                     target = "_blank"),
                                   'or upper asymptote lower than one - inattention \\(d\\). Similarly to logistic
                                   regression, also its extensions provide detection of uniform and non-uniform DIF by
                                   letting difficulty parameter \\(b\\) (uniform) and discrimination parameter \\(a\\)
                                   (non-uniform) differ for groups and by testing for significance difference in their
                                   values. Moreover, these extensions allow for testing differences in pseudoguessing and
                                   inattention parameters. '),
                                 p('With ', strong('model'), 'you can specify what parameters should be kept the same for
                                   both groups and what parameters should differ. The notation is similar to IRT models.
                                   In 3PL and 4PL models abbreviations \\(c_{g}\\) or \\(d_{g}\\) mean that parameters \\(c\\) or \\(d\\) are the same for
                                   both groups. With ', strong('type'), 'you can choose parameters in which difference between
                                   groups should be tested.'),
                                 h4("Plot with estimated DIF generalized logistic curve"),
                                 p('Here you can choose what ', strong('type'),' of DIF to test. You can also select ',
                                   strong('correction method'), ' for multiple comparison or', strong('item purification.')),
                                 fluidRow(column(3,
                                                 selectInput(inputId = "DIF_NLR_model_plot",
                                                             label = "Model",
                                                             choices = c("Rasch" = "Rasch",
                                                                         "1PL" = "1PL",
                                                                         "2PL" = "2PL",
                                                                         "3PLcg" = "3PLcg",
                                                                         "3PLdg" = "3PLdg",
                                                                         "3PLc" = "3PLc",
                                                                         "3PLd" = "3PLd",
                                                                         "4PLcgdg" = "4PLcgdg",
                                                                         "4PLcgd" = "4PLcgd",
                                                                         "4PLcdg" = "4PLcdg",
                                                                         "4PL" = "4PL"),
                                                             selected = "3PLcg")),
                                          column(1,
                                                 withMathJax(),
                                                 checkboxGroupInput(inputId = 'DIF_NLR_type_plot',
                                                                    label = 'Type',
                                                                    choices = c("\\(a\\)" = "a",
                                                                                "\\(b\\)" = "b",
                                                                                "\\(c\\)" = "c",
                                                                                "\\(d\\)" = "d"),
                                                                    selected = c("a", "b"))),
                                          column(3,
                                                 selectInput(inputId = "DIF_NLR_correction_method_plot",
                                                             label = "Correction method",
                                                             choices = c("Benjamini-Hochberg" = "BH",
                                                                         "Benjamini-Yekutieli" = "BY",
                                                                         "Holm" = "holm",
                                                                         "Hochberg" = "hochberg",
                                                                         "Hommel" = "hommel",
                                                                         "None" = "none"),
                                                             selected = "none"),
                                                 checkboxInput(inputId = 'DIF_NLR_purification_plot',
                                                               label = 'Item purification',
                                                               value = FALSE)),
                                          column(3,
                                                 sliderInput(inputId = "DIF_NLR_item_plot",
                                                             label = "Item",
                                                             min = 1,
                                                             value = 1,
                                                             max = 10,
                                                             step = 1,
                                                             animate = animationOptions(interval = 1600)))),
                                 p('Points represent proportion of correct answer with respect to standardized
                                   total score. Their size is determined by count of respondents who achieved
                                   given level of standardized total score with respect to the group membership.'),
                                 plotOutput('plot_DIF_NLR'),
                                 downloadButton("DP_plot_DIF_NLR", label = "Download figure"),
                                 h4("Equation"),
                                 fluidRow(column(12, align = "center", uiOutput("DIF_NLR_equation_plot"))),
                                 h4("Table of parameters"),
                                 fluidRow(column(12, align = "center", tableOutput('tab_coef_DIF_NLR'))),
                                 br(),
                                 h4("Selected R code"),
                                 div(code(HTML("library(difNLR)&nbsp;<br><br>#&nbsp;loading&nbsp;data&nbsp;<br>data(GMAT)&nbsp;<br>Data&nbsp;<-&nbsp;GMAT[,&nbsp;1:20]&nbsp;<br>group&nbsp;<-&nbsp;GMAT[,&nbsp;\"group\"]&nbsp;<br><br>#&nbsp;generalized&nbsp;logistic&nbsp;regression&nbsp;DIF&nbsp;method&nbsp;<br>#&nbsp;using&nbsp;3PL&nbsp;model&nbsp;with&nbsp;the&nbsp;same&nbsp;guessing&nbsp;parameter&nbsp;for&nbsp;both&nbsp;groups&nbsp;<br>fit&nbsp;<-&nbsp;difNLR(Data&nbsp;=&nbsp;Data,&nbsp;group&nbsp;=&nbsp;group,&nbsp;focal.name&nbsp;=&nbsp;1,&nbsp;model&nbsp;=&nbsp;\"3PLcg\",&nbsp;type&nbsp;=&nbsp;\"both\",&nbsp;p.adjust.method&nbsp;=&nbsp;\"none\")&nbsp;<br><br>#&nbsp;plot&nbsp;of&nbsp;characteristic&nbsp;curve&nbsp;of&nbsp;item&nbsp;1&nbsp;<br>plot(fit,&nbsp;item&nbsp;=&nbsp;1)&nbsp;<br><br>#&nbsp;table&nbsp;of&nbsp;estimated&nbsp;coefficients&nbsp;<br>fit$nlrPAR"))),
                                 br()
                        )
                      )
             ),
             # * IRT LORD ####
             tabPanel("IRT Lord",
                      tabsetPanel(
                        # ** Summary ####
                        tabPanel('Summary',
                                 h3('Lord test for IRT models'),
                                 p('Lord test (Lord, 1980) is based on IRT model
                                   (1PL, 2PL, or 3PL with the same guessing). It uses the
                                   difference between item parameters for the two groups
                                   to detect DIF. In statistical terms, Lord statistic is
                                   equal to Wald statistic.'),
                                 br(),
                                 img(src = "fig_lord_uniform.png",
                                     style = "float: left; width: 32%; margin-right: 2%; margin-left: 16%; margin-bottom: 0.5em;"),
                                 img(src = "fig_lord_nonuniform.png",
                                     style = "float: left; width: 32%; margin-right: 16%; margin-left: 2%; margin-bottom: 0.5em;"),
                                 br(),
                                 h4('Summary table'),
                                 p('Here you can choose ',strong('model'),' to test. You can also select ',strong('correction method'),' for multiple comparison or', strong('item purification.')),
                                 fluidPage(div(style = "display: inline-block; vertical-align: top; width: 10%; ",
                                               radioButtons(inputId = 'type_print_DIF_IRT_lord',
                                                            label = 'Model',
                                                            choices = c("1PL" = '1PL',
                                                                        "2PL" = '2PL',
                                                                        "3PL" = '3PL'),
                                                            selected = '2PL')),
                                           div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                           div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                               selectInput(inputId = "correction_method_DIF_IRT_lordSummary",
                                                           label = "Correction method",
                                                           choices = c("Benjamini-Hochberg" = "BH",
                                                                       "Benjamini-Yekutieli" = "BY",
                                                                       "Holm" = "holm",
                                                                       "Hochberg" = "hochberg",
                                                                       "Hommel" = "hommel",
                                                                       "None" = "none"),
                                                           selected = "none"),
                                               checkboxInput('puri_Lord', 'Item purification', FALSE))),
                                 uiOutput("DIF_IRT_LORD_na_alert"),
                                 verbatimTextOutput('print_DIF_IRT_Lord'),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(difNLR)'),
                                     br(),
                                     code('library(difR)'),
                                     br(),
                                     code('data(GMAT)'),
                                     br(),
                                     code('data <- GMAT[, 1:20]'),
                                     br(),
                                     code('group <- GMAT[, "group"]'),
                                     br(),
                                     br(),
                                     code('# 1PL IRT MODEL'),
                                     br(),
                                     code('fit1PL <- difLord(Data = data, group = group, focal.name = 1,
                                          model = "1PL",
                                          p.adjust.method = "none", purify = F)'),
                                     br(),
                                     code('fit1PL'),
                                     br(),
                                     br(),
                                     code('# 2PL IRT MODEL'),
                                     br(),
                                     code('fit2PL <- difLord(Data = data, group = group, focal.name = 1,
                                          model = "2PL",
                                          p.adjust.method = "none", purify = F)'),
                                     br(),
                                     code('fit2PL'),
                                     br(),
                                     br(),
                                     code('# 3PL IRT MODEL with the same guessing for groups'),
                                     br(),
                                     code('guess <- itemParEst(data, model = "3PL")[, 3]'),
                                     br(),
                                     code('fit3PL <- difLord(Data = data, group = group, focal.name = 1,
                                          model = "3PL", c = guess,
                                          p.adjust.method = "none", purify = F)'),
                                     br(),
                                     code('fit3PL')),
                                 br()
                        ),
                        # ** Items ####
                        tabPanel('Items',
                                 h3('Lord test for IRT models'),
                                 p('Lord test (Lord, 1980) is based on IRT model
                                   (1PL, 2PL, or 3PL with the same guessing). It uses the
                                   difference between item parameters for the two groups
                                   to detect DIF. In statistical terms, Lord statistic is
                                   equal to Wald statistic.'),
                                 br(),
                                 h4('Plot with estimated DIF characteristic curve'),
                                 p('Here you can choose ',strong('model'),' to test. You can also select ',strong('correction method'),' for multiple comparison or', strong('item purification.')),
                                 fluidPage(div(style = "display: inline-block; vertical-align: top; width: 10%; ",
                                               radioButtons(inputId = 'type_plot_DIF_IRT_lord',
                                                            label = 'Model',
                                                            choices = c("1PL" = '1PL',
                                                                        "2PL" = '2PL',
                                                                        "3PL" = '3PL'),
                                                            selected = '2PL')),
                                           div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                           div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                               selectInput(inputId = "correction_method_DIF_IRT_lordItems",
                                                           label = "Correction method",
                                                           choices = c("Benjamini-Hochberg" = "BH",
                                                                       "Benjamini-Yekutieli" = "BY",
                                                                       "Holm" = "holm",
                                                                       "Hochberg" = "hochberg",
                                                                       "Hommel" = "hommel",
                                                                       "None" = "none"),
                                                           selected = "none"),
                                               checkboxInput('puri_Lord_plot', 'Item purification', FALSE)),
                                           div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                           div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                               sliderInput(inputId = "difirt_lord_itemSlider",
                                                           label = "Item",
                                                           min = 1,
                                                           value = 1,
                                                           max = 10,
                                                           step = 1,
                                                           animate = animationOptions(interval = 1600)))),
                                 p('NOTE: Plots and tables are based on larger DIF IRT model. '),
                                 plotOutput('plot_DIF_IRT_Lord'),
                                 downloadButton("DP_plot_DIF_IRT_Lord", label = "Download figure"),
                                 h4("Equation"),
                                 uiOutput('irtint_lord'),
                                 fluidRow(column(12, align = "center", uiOutput('irteq_lord'))),
                                 h4("Table of parameters"),
                                 fluidRow(column(12, align = "center", tableOutput('tab_coef_DIF_IRT_Lord'))),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(difNLR)'),
                                     br(),
                                     code('library(difR)'),
                                     br(),
                                     code('data(GMAT)'),
                                     br(),
                                     code('data <- GMAT[, 1:20]'),
                                     br(),
                                     code('group <- GMAT[, "group"]'),
                                     br(),
                                     br(),
                                     code('# 1PL IRT MODEL'),
                                     br(),
                                     code('fit1PL <- difLord(Data = data, group = group, focal.name = 1,
                                          model = "1PL",
                                          p.adjust.method = "none", purify = F)'),
                                     br(),
                                     code('fit1PL'),
                                     br(),
                                     code('# Coefficients for all items'),
                                     br(),
                                     code('tab_coef1PL <- fit1PL$itemParInit'),
                                     br(),
                                     code('# Plot of characteristic curve of item 1'),
                                     br(),
                                     code('plotDIFirt(parameters = tab_coef1PL, item = 1, test = "Lord")'),
                                     br(),
                                     br(),
                                     code('# 2PL IRT MODEL'),
                                     br(),
                                     code('fit2PL <- difLord(Data = data, group = group, focal.name = 1,
                                          model = "2PL",
                                          p.adjust.method = "none", purify = F)'),
                                     br(),
                                     code('fit2PL'),
                                     br(),
                                     code('# Coefficients for all items'),
                                     br(),
                                     code('tab_coef2PL <- fit2PL$itemParInit'),
                                     br(),
                                     code('# Plot of characteristic curve of item 1'),
                                     br(),
                                     code('plotDIFirt(parameters = tab_coef2PL, item = 1, test = "Lord")'),
                                     br(),
                                     br(),
                                     code('# 3PL IRT MODEL with the same guessing for groups'),
                                     br(),
                                     code('guess <- itemParEst(data, model = "3PL")[, 3]'),
                                     br(),
                                     code('fit3PL <- difLord(Data = data, group = group, focal.name = 1,
                                          model = "3PL", c = guess,
                                          p.adjust.method = "none", purify = F)'),
                                     br(),
                                     code('fit3PL'),
                                     br(),
                                     code('# Coefficients for all items'),
                                     br(),
                                     code('tab_coef3PL <- fit3PL$itemParInit'),
                                     br(),
                                     code('# Plot of characteristic curve of item 1'),
                                     br(),
                                     code('plotDIFirt(parameters = tab_coef3PL, item = 1, test = "Lord")')),
                                 br()
                        )
                      )
             ),
             # * IRT RAJU ####
             tabPanel("IRT Raju",
                      tabsetPanel(
                        # ** Summary ####
                        tabPanel('Summary',
                                 h3('Raju test for IRT models'),
                                 p('Raju test (Raju, 1988, 1990) is based on IRT
                                   model (1PL, 2PL, or 3PL with the same guessing). It
                                   uses the area between the item charateristic curves
                                   for the two groups to detect DIF.'),
                                 br(),
                                 img(src = "fig_raju_uniform.png",
                                     style = "float: left; width: 32%; margin-right: 2%; margin-left: 16%; margin-bottom: 0.5em;"),
                                 img(src = "fig_raju_nonuniform.png",
                                     style = "float: left; width: 32%; margin-right: 16%; margin-left: 2%; margin-bottom: 0.5em;"),
                                 br(),
                                 h4('Summary table'),
                                 p('Here you can choose ',strong('model'),' to test. You can also select ',strong('correction method'),' for multiple comparison or', strong('item purification.')),
                                 fluidPage(div(style = "display: inline-block; vertical-align: top; width: 10%; ",
                                               radioButtons(inputId = 'type_print_DIF_IRT_raju',
                                                            label = 'Model',
                                                            choices = c("1PL" = '1PL',
                                                                        "2PL" = '2PL',
                                                                        "3PL" = '3PL'),
                                                            selected = '2PL')),
                                           div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                           div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                               selectInput(inputId = "correction_method_DIF_IRT_rajuSummary",
                                                           label = "Correction method",
                                                           choices = c("Benjamini-Hochberg" = "BH",
                                                                       "Benjamini-Yekutieli" = "BY",
                                                                       "Holm" = "holm",
                                                                       "Hochberg" = "hochberg",
                                                                       "Hommel" = "hommel",
                                                                       "None" = "none"),
                                                           selected = "none"),
                                               checkboxInput(inputId = 'puri_Raju',
                                                             label = 'Item purification',
                                                             value = FALSE))),
                                 uiOutput("DIF_IRT_Raju_na_alert"),
                                 verbatimTextOutput('print_DIF_IRT_Raju'),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(difNLR)'),
                                     br(),
                                     code('library(difR)'),
                                     br(),
                                     code('data(GMAT)'),
                                     br(),
                                     code('data <- GMAT[, 1:20]'),
                                     br(),
                                     code('group <- GMAT[, "group"]'),
                                     br(),
                                     br(),
                                     code('# 1PL IRT MODEL'),
                                     br(),
                                     code('fit1PL <- difRaju(Data = data, group = group, focal.name = 1,
                                          model = "1PL",
                                          p.adjust.method = "none", purify = F)'),
                                     br(),
                                     code('fit1PL'),
                                     br(),
                                     br(),
                                     code('# 2PL IRT MODEL'),
                                     br(),
                                     code('fit2PL <- difRaju(Data = data, group = group, focal.name = 1,
                                          model = "2PL",
                                          p.adjust.method = "none", purify = F)'),
                                     br(),
                                     code('fit2PL'),
                                     br(),
                                     br(),
                                     code('# 3PL IRT MODEL with the same guessing for groups'),
                                     br(),
                                     code('guess <- itemParEst(data, model = "3PL")[, 3]'),
                                     br(),
                                     code('fit3PL <- difRaju(Data = data, group = group, focal.name = 1,
                                          model = "3PL", c = guess,
                                          p.adjust.method = "none", purify = F)'),
                                     br(),
                                     code('fit3PL')),
                                 br()
                        ),
                        # ** Items ####
                        tabPanel('Items',
                                 h3('Raju test for IRT models'),
                                 p('Raju test (Raju, 1988, 1990) is based on IRT
                                   model (1PL, 2PL, or 3PL with the same guessing). It
                                   uses the area between the item charateristic curves
                                   for the two groups to detect DIF.'),
                                 br(),
                                 h4('Plot with estimated DIF characteristic curve'),
                                 p('Here you can choose ', strong('model'), ' to test. You can also select ', strong('correction method'),
                                   ' for multiple comparison or', strong('item purification.')),
                                 fluidPage(div(style = "display: inline-block; vertical-align: top; width: 10%; ",
                                               radioButtons(inputId = 'type_plot_DIF_IRT_raju',
                                                            label = 'Model',
                                                            choices = c("1PL" = '1PL',
                                                                        "2PL" = '2PL',
                                                                        "3PL" = '3PL'),
                                                            selected = '2PL')),
                                           div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                           div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                               selectInput(inputId = "correction_method_DIF_IRT_rajuItems",
                                                           label = "Correction method",
                                                           choices = c("Benjamini-Hochberg" = "BH",
                                                                       "Benjamini-Yekutieli" = "BY",
                                                                       "Holm" = "holm",
                                                                       "Hochberg" = "hochberg",
                                                                       "Hommel" = "hommel",
                                                                       "None" = "none"),
                                                           selected = "none"),
                                               checkboxInput(inputId = 'puri_Raju_plot',
                                                             label = 'Item purification',
                                                             value = FALSE)),
                                           div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                           div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                               sliderInput(inputId = "difirt_raju_itemSlider",
                                                           label = "Item",
                                                           min = 1,
                                                           value = 1,
                                                           max = 10,
                                                           step = 1,
                                                           animate = animationOptions(interval = 1600)))),
                                 p('NOTE: Plots and tables are based on larger DIF IRT model. '),
                                 plotOutput('plot_DIF_IRT_Raju'),
                                 downloadButton("DP_plot_DIF_IRT_Raju", label = "Download figure"),
                                 h4("Equation"),
                                 uiOutput('irtint_raju'),
                                 fluidRow(column(12, align = "center", uiOutput('irteq_raju'))),
                                 h4("Table of parameters"),
                                 fluidRow(column(12, align = "center", tableOutput('tab_coef_DIF_IRT_Raju'))),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(difNLR)'),
                                     br(),
                                     code('library(difR)'),
                                     br(),
                                     code('data(GMAT)'),
                                     br(),
                                     code('data <- GMAT[, 1:20]'),
                                     br(),
                                     code('group <- GMAT[, "group"]'),
                                     br(),
                                     br(),
                                     code('# 1PL IRT MODEL'),
                                     br(),
                                     code('fit1PL <- difRaju(Data = data, group = group, focal.name = 1,
                                          model = "1PL",
                                          p.adjust.method = "none", purify = F)'),
                                     br(),
                                     code('fit1PL'),
                                     br(),
                                     code('# Coefficients for all items'),
                                     br(),
                                     code('tab_coef1PL <- fit1PL$itemParInit'),
                                     br(),
                                     code('# Plot of characteristic curve of item 1'),
                                     br(),
                                     code('plotDIFirt(parameters = tab_coef1PL, item = 1, test = "Raju")'),
                                     br(),
                                     br(),
                                     code('# 2PL IRT MODEL'),
                                     br(),
                                     code('fit2PL <- difRaju(Data = data, group = group, focal.name = 1,
                                          model = "2PL",
                                          p.adjust.method = "none", purify = F)'),
                                     br(),
                                     code('fit2PL'),
                                     br(),
                                     code('# Coefficients for all items'),
                                     br(),
                                     code('tab_coef2PL <- fit2PL$itemParInit'),
                                     br(),
                                     code('# Plot of characteristic curve of item 1'),
                                     br(),
                                     code('plotDIFirt(parameters = tab_coef2PL, item = 1, test = "Raju")'),
                                     br(),
                                     br(),
                                     code('# 3PL IRT MODEL with the same guessing for groups'),
                                     br(),
                                     code('guess <- itemParEst(data, model = "3PL")[, 3]'),
                                     br(),
                                     code('fit3PL <- difRaju(Data = data, group = group, focal.name = 1,
                                          model = "3PL", c = guess,
                                          p.adjust.method = "none", purify = F)'),
                                     br(),
                                     code('fit3PL'),
                                     br(),
                                     code('# Coefficients for all items'),
                                     br(),
                                     code('tab_coef3PL <- fit3PL$itemParInit'),
                                     br(),
                                     code('# Plot of characteristic curve of item 1'),
                                     br(),
                                     code('plotDIFirt(parameters = tab_coef3PL, item = 1, test = "Raju")')),
                                 br())
                      )
             ),
             # * SIBTEST ####
             tabPanel("SIBTEST",
                      h3("SIBTEST"),
                      p("The SIBTEST method (Shealy & Stout, 1993) allows for detection of uniform DIF without requiring
                        an item response model approach. Its modified version, the Crossing-SIBTEST (Chalmers, 2018; Li & Stout, 1996),
                        focuses on detection of non-uniform DIF."),
                      h4("Summary table"),
                      p('Here you can choose ',strong('type'),' of DIF to test.  With uniform DIF, SIBTEST is applied, while with non-uniform DIF, the Crossing-SIBTEST method is used instead. You can also select ',strong('correction method'),' for multiple comparison or', strong('item purification.')),
                      fluidRow(column(2,
                                      radioButtons(inputId = "DIF_SIBTEST_type",
                                                   label = "Type",
                                                   choices = c("Uniform" = "udif",
                                                               "Non-uniform" = "nudif"),
                                                   selected = "udif")),
                               column(3,
                                      selectInput(inputId = "DIF_SIBTEST_correction_method",
                                                  label = "Correction method",
                                                  choices = c("Benjamini-Hochberg" = "BH",
                                                              "Benjamini-Yekutieli" = "BY",
                                                              "Holm" = "holm",
                                                              "Hochberg" = "hochberg",
                                                              "Hommel" = "hommel",
                                                              "None" = "none"),
                                                  selected = "none"),
                                      checkboxInput(inputId = "DIF_SIBTEST_purification",
                                                    label = "Item purification",
                                                    value = FALSE))),
                      uiOutput("DIF_SIBTEST_na_alert"),
                      verbatimTextOutput("DIF_SIBTEST_print"),
                      br(),
                      h4("Selected code"),
                      div(code(HTML("library(difNLR)<br>library(difR)<br><br>#&nbsp;loading&nbsp;data<br>data(GMAT)<br>data&nbsp;<-&nbsp;GMAT[,&nbsp;1:20]<br>group&nbsp;<-&nbsp;GMAT[,&nbsp;\"group\"]<br><br>#&nbsp;SIBTEST&nbsp;(uniform&nbsp;DIF)<br>fit&nbsp;<-&nbsp;difMH(Data&nbsp;=&nbsp;data,&nbsp;group&nbsp;=&nbsp;group,&nbsp;focal.name&nbsp;=&nbsp;1,&nbsp;type&nbsp;=&nbsp;\"udif\",&nbsp;p.adjust.method&nbsp;=&nbsp;\"none\",&nbsp;purify&nbsp;=&nbsp;F)<br>fit<br><br>#&nbsp;Crossing-SIBTEST&nbsp;(non-uniform&nbsp;DIF)<br>fit&nbsp;<-&nbsp;difMH(Data&nbsp;=&nbsp;data,&nbsp;group&nbsp;=&nbsp;group,&nbsp;focal.name&nbsp;=&nbsp;1,&nbsp;type&nbsp;=&nbsp;\"nudif\",&nbsp;p.adjust.method&nbsp;=&nbsp;\"none\",&nbsp;purify&nbsp;=&nbsp;F)<br>fit"))),
                      br()),
             # * DDF ####
             tabPanel("DDF",
                      tabsetPanel(
                        # ** Summary ####
                        tabPanel('Summary',
                                 h3('Differential Distractor Functioning with multinomial log-linear regression model'),
                                 p('Differential Distractor Functioning (DDF) occurs when people from different
                                   groups but with the same knowledge have different probability of selecting
                                   at least one distractor choice. DDF is here examined by multinomial log-linear
                                   regression model with Z-score and group membership as covariates. '),
                                 h4('Equation'),
                                 p('For ', strong('K'), ' possible test choices is the probability of the correct answer for
                                   person ', strong('i'), ' with standardized total score ', strong('Z'), ' and group
                                   membership ', strong('G'),' in item ', strong('j'), 'given by the following equation: '),
                                 ('$$\\mathrm{P}(Y_{ij} = K|Z_i, G_i, b_{jl0}, b_{jl1}, b_{jl2}, b_{jl3}, l = 1, \\dots, K-1) =
                                  \\frac{1}{1 + \\sum_l e^{\\left( b_{il0} + b_{il1} Z + b_{il2} G + b_{il3} Z:G\\right)}}$$'),
                                 p('The probability of choosing distractor ', strong('k'), ' is then given by: '),
                                 ('$$\\mathrm{P}(Y_{ij} = k|Z_i, G_i, b_{jl0}, b_{jl1}, b_{jl2}, b_{jl3}, l = 1, \\dots, K-1) =
                                  \\frac{e^{\\left( b_{jk0} + b_{jk1} Z_i + b_{jk2} G_i + b_{jk3} Z_i:G_i\\right)}}
                                  {1 + \\sum_l e^{\\left( b_{jl0} + b_{jl1} Z_i + b_{jl2} G_i + b_{jl3} Z_i:G_i\\right)}}$$'),
                                 br(),
                                 h4('Summary table'),
                                 p('Here you can choose ',strong('type'),' of DIF to test.  With uniform DIF, SIBTEST is applied, while with non-uniform DIF, the Crossing-SIBTEST method is used instead. You can also select ',strong('correction method'),' for multiple comparison or', strong('item purification.')),
                                 fluidPage(div(style = "display: inline-block; vertical-align: top; width: 27%; ",
                                               withMathJax(),
                                               radioButtons(inputId = 'type_print_DDF',
                                                            label = 'Type',
                                                            choices = c("\\(H_{0}\\): Any DIF vs. \\(H_{1}\\): No DIF" = 'both',
                                                                        "\\(H_{0}\\): Uniform DIF vs. \\(H_{1}\\): No DIF" = 'udif',
                                                                        "\\(H_{0}\\): Non-Uniform DIF vs. \\(H_{1}\\): Uniform DIF" = 'nudif'),
                                                            selected = 'both')),
                                           div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                           div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                               selectInput(inputId = "correction_method_print_DDF",
                                                           label = "Correction method",
                                                           choices = c("Benjamini-Hochberg" = "BH",
                                                                       "Benjamini-Yekutieli" = "BY",
                                                                       "Holm" = "holm",
                                                                       "Hochberg" = "hochberg",
                                                                       "Hommel" = "hommel",
                                                                       "None" = "none"),
                                                           selected = "none"),
                                               checkboxInput(inputId = 'puri_DDF_print',
                                                             label = 'Item purification',
                                                             value = FALSE))),
                                 uiOutput("DDF_na_alert"),
                                 verbatimTextOutput('print_DDF'),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(difNLR)'),
                                     br(),
                                     code('data(GMATtest, GMATkey)'),
                                     br(),
                                     code('Data <- GMATtest[, 1:20]'),
                                     br(),
                                     code('group <- GMATtest[, "group"]'),
                                     br(),
                                     code('key <- GMATkey'),
                                     br(),
                                     br(),
                                     code('# DDF with difNLR package'),
                                     br(),
                                     code('fit <- ddfMLR(Data, group, focal.name = 1, key, type = "both",
                                          p.adjust.method = "none")'),
                                     br(),
                                     code('fit')),
                                 br()
                        ),
                        # ** Items ####
                        tabPanel('Items',
                                 h3('Differential Distractor Functioning with multinomial log-linear regression model'),
                                 p('Differential Distractor Functioning (DDF) occurs when people from different
                                   groups but with the same knowledge have different probability of selecting
                                   at least one distractor choice. DDF is here examined by Multinomial Log-linear
                                   Regression model with Z-score and group membership as covariates. '),
                                 h4("Plot with estimated DDF curves"),
                                 p('Here you can choose ',strong('type'),' of DIF to test.  With uniform DIF, SIBTEST is applied, while with non-uniform DIF, the Crossing-SIBTEST method is used instead. You can also select ',strong('correction method'),' for multiple comparison or', strong('item purification.')),
                                 fluidPage(div(style = "display: inline-block; vertical-align: top; width: 27%; ",
                                               withMathJax(),
                                               radioButtons(inputId = 'type_plot_DDF',
                                                            label = 'Type',
                                                            choices = c("\\(H_{0}\\): Any DIF vs. \\(H_{1}\\): No DIF" = 'both',
                                                                        "\\(H_{0}\\): Uniform DIF vs. \\(H_{1}\\): No DIF" = 'udif',
                                                                        "\\(H_{0}\\): Non-Uniform DIF vs. \\(H_{1}\\): Uniform DIF" = 'nudif'),
                                                            selected = 'both')),
                                           div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                           div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                               selectInput(inputId = "correction_method_plot_DDF",
                                                           label = "Correction method",
                                                           choices = c("Benjamini-Hochberg" = "BH",
                                                                       "Benjamini-Yekutieli" = "BY",
                                                                       "Holm" = "holm",
                                                                       "Hochberg" = "hochberg",
                                                                       "Hommel" = "hommel",
                                                                       "None" = "none"),
                                                           selected = "none"),
                                               checkboxInput(inputId = 'puri_DDF_plot',
                                                             label = 'Item purification',
                                                             value = FALSE)),
                                           div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                           div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                               sliderInput(inputId = "ddfSlider",
                                                           label = "Item",
                                                           min = 1,
                                                           value = 1,
                                                           max = 10,
                                                           step = 1,
                                                           animate = animationOptions(interval = 1600)))),
                                 p('Points represent proportion of selected answer with respect to standardized
                                   total score. Their size is determined by count of respondents who achieved
                                   given level of standardized total score and who selected given option with
                                   respect to the group membership.'),
                                 plotOutput('plot_DDF'),
                                 downloadButton("DP_plot_DDF", label = "Download figure"),
                                 h4("Equation"),
                                 fluidRow(column(12, align = "center", uiOutput('DDFeq'))),
                                 h4("Table of parameters"),
                                 fluidRow(column(12, align = "center", tableOutput('tab_coef_DDF'))),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(difNLR)'),
                                     br(),
                                     code('data(GMATtest, GMATkey)'),
                                     br(),
                                     code('Data <- GMATtest[, 1:20]'),
                                     br(),
                                     code('group <- GMATtest[, "group"]'),
                                     br(),
                                     code('key <- GMATkey'),
                                     br(),
                                     br(),
                                     code('# DDF with difNLR package'),
                                     br(),
                                     code('fit <- ddfMLR(Data, group, focal.name = 1, key, type = "both",
                                          p.adjust.method = "none")'),
                                     br(),
                                     code('# Estimated coefficients of item 1'),
                                     br(),
                                     code('fit$mlrPAR[[1]]')),
                                 br()
                        )
                      )
             ),
             # * METHOD COMPARISON ####
             tabPanel("Method comparison",
                      h3("Method comparison"),
                      p("Here you can compare all offered DIF detection methods. In the table below, columns represent DIF detection methods, and rows represent item number. If
                        the method detects item as DIF, value 1 is assigned to that item, otherwise 0 is assigned. In case that any method fail to converge or cannot be
                        fitted, NA is displayed instead of 0/1 values. Methods in columns: "),
                      tags$ul(
                        tags$li(strong('Delta'), 'is delta plot method (Angoff & Ford, 1973; Magis & Facon, 2012),'),
                        tags$li(strong('MH'), 'is Mantel-Haenszel test (Mantel & Haenszel, 1959), '),
                        tags$li(strong('Logistic'), 'is logistic regression (Swaminathan & Rogers, 1990),'),
                        tags$li(strong('GLogistic'), 'is generalized logistic regression (Drabinova & Martinkova, 2017),'),
                        tags$li(strong('Lord'), 'is Lord chi-square test (Lord, 1980),'),
                        tags$li(strong('Raju'), 'is Raju area method (Raju, 1990),'),
                        tags$li(strong('SIBTEST'), 'is SIBTEST (Shealy & Stout, 1993) and crossing-SIBTEST (Chalmers, 2018; Li & Stout, 1996) methods, '),
                        tags$li(strong('DDF'), 'is differential distractor functioning with multinomial log-linear regression model. ')
                      ),
                      h3("Table with method comparison"),
                      p("Methods in the table using all settings that were set in the previous tabs. The last column shows how
                        many methods out of 8 detect certain item as DIF. The last row shows how many items are detected as DIF by certain method. "),
                      fluidRow(column(12, align = "center", tableOutput("method_comparison_table"))),
                      br(),
                      br()
             )
  )
