% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Olink_anova.R
\name{olink_anova}
\alias{olink_anova}
\title{Function which performs an ANOVA per protein}
\usage{
olink_anova(
  df,
  variable,
  outcome = "NPX",
  covariates = NULL,
  return.covariates = FALSE,
  verbose = TRUE
)
}
\arguments{
\item{df}{NPX data frame in long format with at least protein name (Assay), OlinkID, UniProt, Panel and a factor with at least 3 levels.}

\item{variable}{Single character value or character array.
Variable(s) to test. If length > 1, the included variable names will be used in crossed analyses .
Also takes ':' or '*' notation.}

\item{outcome}{Character. The dependent variable. Default: NPX.}

\item{covariates}{Single character value or character array. Default: NULL.
Covariates to include. Takes ':' or '*' notation. Crossed analysis will not be inferred from main effects.}

\item{return.covariates}{Boolean. Default: False. Returns F-test results for the covariates. Note: Adjusted p-values will be NA for the covariates.}

\item{verbose}{Boolean. Default: True. If information about removed samples, factor conversion and final model formula is to be printed to the console.}
}
\value{
A "tibble" containing the ANOVA results for every protein. The tibble is arranged by ascending p-values.
Columns include:

\itemize{
 \item{Assay:} "character" Protein symbol
 \item{OlinkID:} "character" Olink specific ID  
 \item{UniProt:} "character" Olink specific ID  
 \item{Panel:} "character" Name of Olink Panel
 \item{term:} "character" term in model
 \item{df:} "numeric" degrees of freedom
 \item{sumsq:} "numeric" sum of square
 \item{meansq:} "numeric" mean of square
 \item{statistic:} "numeric" value of the statistic
 \item{p.value:} "numeric" nominal p-value
 \item{Adjusted_pval:} "numeric" adjusted p-value for the test (Benjamini&Hochberg)
 \item{Threshold:} "character" if adjusted p-value is significant or not (< 0.05)
}
}
\description{
Performs an ANOVA F-test for each assay (by OlinkID) in every panel using car::Anova and Type III sum of squares.
The function handles both factor and numerical variables and/or covariates. \cr\cr
Samples that have no variable information or missing factor levels are automatically removed from the analysis (specified in a messsage if verbose = TRUE).
Character columns in the input dataframe are automatically converted to factors (specified in a message if verbose = TRUE).
Numerical variables are not converted to factors.
If a numerical variable is to be used as a factor, this conversion needs to be done on the dataframe before the function call. \cr\cr
Crossed analysis, i.e. A*B formula notation, is inferred from the variable argument in the following cases: \cr
\itemize{
\item c('A','B')
\item c('A: B')
\item c('A: B', 'B') or c('A: B', 'A')
}
Inference is specified in a message if verbose = TRUE. \cr
For covariates, crossed analyses need to be specified explicity, i.e. two main effects will not be expaned with a c('A','B') notation. Main effects present in the variable takes precedence.
The formula notation of the final model is specified in a message if verbose = TRUE. \cr\cr
Adjusted p-values are calculated by stats::p.adjust according to the Benjamini & Hochberg (1995) method (“fdr”).
The threshold is determined by logic evaluation of Adjusted_pval < 0.05. Covariates are not included in the p-value adjustment.
}
\examples{
\donttest{

library(dplyr)

npx_df <- npx_data1 \%>\% filter(!grepl('control',SampleID, ignore.case = TRUE))

#One-way ANOVA, no covariates.
#Results in a model NPX~Time
anova_results <- olink_anova(df = npx_df, variable = "Time")

#Two-way ANOVA, one main effect covariate.
#Results in model NPX~Treatment*Time+Site.
anova_results <- olink_anova(df = npx_df,
                             variable=c("Treatment:Time"),
                             covariates="Site")

#One-way ANOVA, interaction effect covariate.
#Results in model NPX~Treatment+Site:Time+Site+Time.
anova_results <- olink_anova(df = npx_df,
                             variable="Treatment",
                             covariates="Site:Time")}
}
