% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/SensAnalysisMLP.R
\name{SensAnalysisMLP}
\alias{SensAnalysisMLP}
\alias{SensAnalysisMLP.default}
\alias{SensAnalysisMLP.train}
\alias{SensAnalysisMLP.H2OMultinomialModel}
\alias{SensAnalysisMLP.H2ORegressionModel}
\alias{SensAnalysisMLP.list}
\alias{SensAnalysisMLP.mlp}
\alias{SensAnalysisMLP.nn}
\alias{SensAnalysisMLP.nnet}
\title{Sensitivity of NNET models}
\usage{
SensAnalysisMLP(MLP.fit, .returnSens = TRUE, ...)

\method{SensAnalysisMLP}{default}(MLP.fit, .returnSens = TRUE, trData,
  actfunc = c("linear", "sigmoid", "linear"), preProc = NULL,
  terms = NULL, ...)

\method{SensAnalysisMLP}{train}(MLP.fit, .returnSens = TRUE, ...)

\method{SensAnalysisMLP}{H2OMultinomialModel}(MLP.fit,
  .returnSens = TRUE, ...)

\method{SensAnalysisMLP}{H2ORegressionModel}(MLP.fit, .returnSens = TRUE,
  ...)

\method{SensAnalysisMLP}{list}(MLP.fit, .returnSens = TRUE, trData, ...)

\method{SensAnalysisMLP}{mlp}(MLP.fit, .returnSens = TRUE, trData,
  preProc = NULL, terms = NULL, ...)

\method{SensAnalysisMLP}{nn}(MLP.fit, .returnSens = TRUE,
  preProc = NULL, terms = NULL, ...)

\method{SensAnalysisMLP}{nnet}(MLP.fit, .returnSens = TRUE, trData,
  preProc = NULL, terms = NULL, ...)
}
\arguments{
\item{MLP.fit}{fitted model from caret package using nnet method}

\item{.returnSens}{logical value. If \code{TRUE}, sensibility of the model is returned.}

\item{...}{additional arguments passed to or from other methods}

\item{trData}{data frame containing the training data of the model}

\item{actfunc}{character vector indicating the activation function of each neurons layer.}

\item{preProc}{preProcess structure applied to the training data}

\item{terms}{function applied to the training data to create factors}
}
\value{
dataframe with the sensitivities obtained for each variable if .returnSens \code{TRUE}.
If there is more than one output, the sensitivities of each output are given in a list.
}
\description{
Function for evaluating the sensitivities of the inputs variables in a mlp model
}
\section{Output}{

\itemize{
  \item Plot 1: colorful plot with the classification of the classes in a 2D map
  \item Plot 2: b/w plot with probability of the chosen class in a 2D map
  \item Plot 3: plot with the stats::predictions of the data provided
}
}

\examples{
## Load data -------------------------------------------------------------------
data("DAILY_DEMAND_TR")
fdata <- DAILY_DEMAND_TR

## Parameters of the NNET ------------------------------------------------------
hidden_neurons <- 5
iters <- 250
decay <- 0.1

################################################################################
#########################  REGRESSION NNET #####################################
################################################################################
## Regression dataframe --------------------------------------------------------
# Scale the data
fdata.Reg.tr <- fdata[,2:ncol(fdata)]
fdata.Reg.tr[,2:3] <- fdata.Reg.tr[,2:3]/10
fdata.Reg.tr[,1] <- fdata.Reg.tr[,1]/1000

# Normalize the data for some models
preProc <- caret::preProcess(fdata.Reg.tr, method = c("center","scale"))
nntrData <- predict(preProc, fdata.Reg.tr)

#' ## TRAIN nnet NNET --------------------------------------------------------
# Create a formula to train NNET
form <- paste(names(fdata.Reg.tr)[2:ncol(fdata.Reg.tr)], collapse = " + ")
form <- formula(paste(names(fdata.Reg.tr)[1], form, sep = " ~ "))

set.seed(150)
nnetmod <- nnet::nnet(form,
                           data = nntrData,
                           linear.output = TRUE,
                           size = hidden_neurons,
                           decay = decay,
                           maxit = iters)
# Try SensAnalysisMLP
NeuralSens::SensAnalysisMLP(nnetmod, trData = nntrData)

\donttest{

## Train caret NNET ------------------------------------------------------------
# Create trainControl
ctrl_tune <- caret::trainControl(method = "boot",
                                 savePredictions = FALSE,
                                 summaryFunction = caret::defaultSummary)
set.seed(150) #For replication
caretmod <- caret::train(form = DEM~.,
                              data = fdata.Reg.tr,
                              method = "nnet",
                              linout = TRUE,
                              tuneGrid = data.frame(size = 3,
                                                    decay = decay),
                              maxit = iters,
                              preProcess = c("center","scale"),
                              trControl = ctrl_tune,
                              metric = "RMSE")

# Try SensAnalysisMLP
NeuralSens::SensAnalysisMLP(caretmod)

## Train h2o NNET --------------------------------------------------------------
# Creaci?n de un cluster local con todos los cores disponibles
h2o::h2o.init(ip = "localhost",
              # -1 indica que se empleen todos los cores disponibles.
              nthreads = 4,
              # M?xima memoria disponible para el cluster.
              max_mem_size = "2g")

# Se eliminan los datos del cluster por si ya hab?a sido iniciado.
h2o::h2o.removeAll()
fdata_h2o <- h2o::as.h2o(x = fdata.Reg.tr, destination_frame = "fdata_h2o")

set.seed(150)
h2omod <-h2o:: h2o.deeplearning(x = names(fdata.Reg.tr)[2:ncol(fdata.Reg.tr)],
                                     y = names(fdata.Reg.tr)[1],
                                     distribution = "AUTO",
                                     training_frame = fdata_h2o,
                                     standardize = TRUE,
                                     activation = "Tanh",
                                     hidden = c(hidden_neurons),
                                     stopping_rounds = 0,
                                     epochs = iters,
                                     seed = 150,
                                     model_id = "nnet_h2o",
                                     adaptive_rate = FALSE,
                                     rate_decay = decay,
                                     export_weights_and_biases = TRUE)

# Try SensAnalysisMLP
NeuralSens::SensAnalysisMLP(h2omod)

# Apaga el cluster
h2o::h2o.shutdown(prompt = FALSE)
rm(fdata_h2o)

## Train neural NNET -----------------------------------------------------------
set.seed(150)
neuralmod <- neural::mlptrain(as.matrix(nntrData[,2:ncol(nntrData)]),
                                   hidden_neurons,
                                   as.matrix(nntrData[1]),
                                   it=iters,
                                   visual=FALSE)

# Try SensAnalysisMLP
trData <- nntrData
names(trData)[1] <- ".outcome"
NeuralSens::SensAnalysisMLP(neuralmod, trData = trData)

## Train RSNNS NNET ------------------------------------------------------------
# Normalize data using RSNNS algorithms
trData <- as.data.frame(RSNNS::normalizeData(fdata.Reg.tr))
names(trData) <- names(fdata.Reg.tr)
set.seed(150)
RSNNSmod <-RSNNS::mlp(x = trData[,2:ncol(trData)],
                           y = trData[,1],
                           size = hidden_neurons,
                           linOut = TRUE,
                           learnFuncParams=c(decay),
                           maxit=iters)
names(trData)[1] <- ".outcome"

# Try SensAnalysisMLP
NeuralSens::SensAnalysisMLP(RSNNSmod, trData = trData)

## TRAIN neuralnet NNET --------------------------------------------------------
# Create a formula to train NNET
form <- paste(names(fdata.Reg.tr)[2:ncol(fdata.Reg.tr)], collapse = " + ")
form <- formula(paste(names(fdata.Reg.tr)[1], form, sep = " ~ "))

set.seed(150)
nnmod <- neuralnet::neuralnet(form,
                                   nntrData,
                                   linear.output = TRUE,
                                   rep = 1,
                                   hidden = hidden_neurons,
                                   lifesign = "minimal",
                                   threshold = 7,
                                   stepmax = iters,
                                   learningrate = decay,
                                   act.fct = "tanh")

# Try SensAnalysisMLP
NeuralSens::SensAnalysisMLP(nnmod)


################################################################################
#########################  CLASSIFICATION NNET #################################
################################################################################
## Regression dataframe --------------------------------------------------------
# Scale the data
fdata.Reg.cl <- fdata[,2:ncol(fdata)]
fdata.Reg.cl[,2:3] <- fdata.Reg.cl[,2:3]/10
fdata.Reg.cl[,1] <- fdata.Reg.cl[,1]/1000

# Normalize the data for some models
preProc <- caret::preProcess(fdata.Reg.cl, method = c("center","scale"))
nntrData <- predict(preProc, fdata.Reg.cl)

# Factorize the output
fdata.Reg.cl$DEM <- factor(round(fdata.Reg.cl$DEM, digits = 1))

# Normalize the data for some models
preProc <- caret::preProcess(fdata.Reg.cl, method = c("center","scale"))
nntrData <- predict(preProc, fdata.Reg.cl)

## Train caret NNET ------------------------------------------------------------
# Create trainControl
ctrl_tune <- caret::trainControl(method = "boot",
                                 savePredictions = FALSE,
                                 summaryFunction = caret::defaultSummary)
set.seed(150) #For replication
caretmod <- caret::train(form = DEM~.,
                                data = fdata.Reg.cl,
                                method = "nnet",
                                linout = FALSE,
                                tuneGrid = data.frame(size = hidden_neurons,
                                                      decay = decay),
                                maxit = iters,
                                preProcess = c("center","scale"),
                                trControl = ctrl_tune,
                                metric = "Accuracy")

# Try SensAnalysisMLP
NeuralSens::SensAnalysisMLP(caretmod)

## Train h2o NNET --------------------------------------------------------------
# Creaci?n de un cluster local con todos los cores disponibles
h2o::h2o.init(ip = "localhost",
              # -1 indica que se empleen todos los cores disponibles.
              nthreads = 4,
              # M?xima memoria disponible para el cluster.
              max_mem_size = "2g")

# Se eliminan los datos del cluster por si ya hab?a sido iniciado.
h2o::h2o.removeAll()
fdata_h2o <- h2o::as.h2o(x = fdata.Reg.cl, destination_frame = "fdata_h2o")

set.seed(150)
h2omod <- h2o::h2o.deeplearning(x = names(fdata.Reg.cl)[2:ncol(fdata.Reg.cl)],
                                       y = names(fdata.Reg.cl)[1],
                                       distribution = "AUTO",
                                       training_frame = fdata_h2o,
                                       standardize = TRUE,
                                       activation = "Tanh",
                                       hidden = c(hidden_neurons),
                                       stopping_rounds = 0,
                                       epochs = iters,
                                       seed = 150,
                                       model_id = "nnet_h2o",
                                       adaptive_rate = FALSE,
                                       rate_decay = decay,
                                       export_weights_and_biases = TRUE)

# Try SensAnalysisMLP
NeuralSens::SensAnalysisMLP(h2omod)

# Apaga el cluster
h2o::h2o.shutdown(prompt = FALSE)
rm(fdata_h2o)

## Train neural NNET -----------------------------------------------------------
# set.seed(150)
# neuralmod <-mlptrain(as.matrix(nntrData[,2:ncol(nntrData)]),
#                           hidden_neurons,
#                           as.matrix(nntrData[1]),
#                           it=iters,
#                           visual=FALSE)
#
# # Try SensAnalysisMLP
# NeuralSens::SensAnalysisMLP(neuralmod, trData = trData)

# ## Train RSNNS NNET ------------------------------------------------------------
# # Normalize data using RSNNS algorithms
# trData <- as.data.frame(RSNNS::normalizeData(fdata.Reg.cl))
# names(trData) <- names(fdata.Reg.tr)
# set.seed(150)
# RSNNSmod <- RSNNS::mlp(x = trData[,2:ncol(trData)],
#                      y = trData[,1],
#                      size = hidden_neurons,
#                      linOut = FALSE,
#                      learnFuncParams=c(decay),
#                      maxit=iters)
# names(trData)[1] <- ".outcome"
#
# # Try SensAnalysisMLP
# NeuralSens::SensAnalysisMLP(RSNNSmod, trData = trData)

## TRAIN neuralnet NNET --------------------------------------------------------
# Create a formula to train NNET
# form <- paste(names(fdata.Reg.tr)[2:ncol(fdata.Reg.tr)], collapse = " + ")
# form <- formula(paste(names(fdata.Reg.tr)[1], form, sep = " ~ "))
#
# set.seed(150)
# nnmod <- neuralnet(form,
#                    nntrData,
#                    linear.output = FALSE,
#                    rep = 1,
#                    hidden = hidden_neurons,
#                    lifesign = "minimal",
#                    threshold = 4,
#                    stepmax = iters,
#                    learningrate = decay,
#                    act.fct = "tanh")
#
# # Try SensAnalysisMLP
# NeuralSens::SensAnalysisMLP(nnmod)

## TRAIN nnet NNET --------------------------------------------------------
# Create a formula to train NNET
form <- paste(names(fdata.Reg.tr)[2:ncol(fdata.Reg.tr)], collapse = " + ")
form <- formula(paste(names(fdata.Reg.tr)[1], form, sep = " ~ "))

set.seed(150)
nnetmod <- nnet::nnet(form,
                      data = nntrData,
                      linear.output = TRUE,
                      size = hidden_neurons,
                      decay = decay,
                      maxit = iters)
# Try SensAnalysisMLP
NeuralSens::SensAnalysisMLP(nnetmod, trData = nntrData)
}
}
