##' Read simulation results based on NMsim's track of model runs
##' @param x Path to the simulation-specific rds file generated by
##'     NMsim, typically called `NMsim_paths.rds`. Or a table of
##'     simulation runs as returned by NMsim when `wait=FALSE`.
##' @param check.time If found, check whether `fst` file modification
##'     time is newer than `rds` file. The `fst` is generated based on
##'     information in `rds`, but notice that some systems don't
##'     preserve the file modification times. Becasue of that,
##'     `check.time` is `FALSE` by default.
##' @param as.fun The default is to return data as a data.frame. Pass
##'     a function (say `tibble::as_tibble`) in as.fun to convert to
##'     something else. If data.tables are wanted, use
##'     as.fun="data.table". The default can be configured using
##'     NMdataConf.
##' @return A data set of class defined by as.fun
##' @import NMdata
##' @import data.table
##' @import fst
##' @importFrom xfun relative_path
##' @export



NMreadSim <- function(x,check.time=FALSE,as.fun){

#### Section start: Dummy variables, only not to get NOTE's in pacakge checks ####

    . <- NULL
    ROWMODEL2 <- NULL
    args.NMscanData <- NULL
    path.sim.lst <- NULL
    file.res.data <- NULL
    pathResFromSims <- NULL
    funs.transform <- NULL

### Section end: Dummy variables, only not to get NOTE's in pacakge checks

    if(missing(as.fun)) as.fun <- NULL
    as.fun <- NMdata:::NMdataDecideOption("as.fun",as.fun)


### rexognized formats:
    ## NMsimRes - return x

    ## NMSimModels - read results in tab

    ## path to rds - read rds, then fst or lst
    
    
    ## if mltiple rds are provided, loop over them
    if(!is.list(x) && is.character(x)) {
        if( length(x)>1 ){
            res <- rbindlist(lapply(x,NMreadSim),fill=TRUE)
            ## setattr(res,"NMsim-models",tab.paths)
            addClass(res,"NMsimRes")
            return(as.fun(res))
        }
    }

    if(!is.list(x) && is.character(x)) {
        ##  an rds, read it, make sure its NMSimModels, check for fst,  and proceed with NMSimModels
        tab.paths <- readRDS(x)
        
        if(!inherits(tab.paths,"NMsimModTab")) {
            stop("The provided rds file does not contain a NMsimModTab object")
        }
        
        file.res.data <- fnAppend(fnExtension(x,"fst"),"res")

### if we have an fst, read it and return results
        if(!is.null(file.res.data) &&
           file.exists(file.res.data) ){
            time.ok <- TRUE
            if(check.time){
                time.ok <- file.mtime(file.res.data)>file.mtime(x)
            }
            if(time.ok){
                res <- read_fst(file.res.data,as.data.table=TRUE)
                addClass(res,"NMsimRes")
                return(as.fun(res))
            } 
        }
    } else if(is.NMsimModTab(x)){
        ## a NMSimModels already, go to procecssing that
        tab.paths <- x
    } else if (is.NMsimRes(x)){
        ## NMsimRes - nothing to do at all
        return(x)
    } else {
        stop("x is not recognized as being associated with NMsim. NMreadSim is only intended for reading output generated by NMsim.")
    }
    
####### Now we have a NMSimModels object to process.
    



### read all sim results

####  must read each model into list elements. Then rbind(fill=T)
### this is to make sure results from different models with
### incompatible columns can be combined.


    wrap.trans <- function(dt,...){
        funs <- list(...)
        for(name.fun in names(funs)){
            dt[,(name.fun):=funs[[name.fun]](get(name.fun))]
        }
        dt
    }
    
    res.list <- lapply(split(tab.paths,by="ROWMODEL2"),function(dat){
        res <- dat[,{
            ## cat(ROWMODEL2," ")     
            ## the rds table must keep NMscanData arguments
            args.NM <- args.NMscanData[[1]]
            if(! "quiet" %in% names(args.NM)){
                args.NM$quiet <- TRUE
            }
            
            ## this.res <- do.call(NMscanData,c(list(file=file.path(dirname(pathResFromSims),pathResFromSims,relative_path(path.sim.lst,dirname(pathResFromSims)))),args.NM))
            this.res <- do.call(NMscanData,c(list(file=file.path(dirname(x),pathResFromSims,relative_path(path.sim.lst,dirname(x)))),args.NM))
            
            if(!is.null(.SD$funs.transform)){
                this.funs <- .SD[1,funs.transform][[1]]
                this.res <- do.call(wrap.trans,c(list(dt=this.res),this.funs))
                this.res
            }


            this.res
        },by=.(ROWMODEL2)]


    })
    
    res <- rbindlist(res.list,fill=TRUE)
    res[,ROWMODEL2:=NULL]

    if(!is.null(file.res.data)){
        NMwriteData(res,
                    file=file.res.data,
                    formats.write="fst",
                    genText=F,
                    quiet=TRUE)
    }


    setattr(res,"NMsimModTab",tab.paths)
    addClass(res,"NMsimRes")
    return(as.fun(res))
}
