% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/GetCDLComp.R
\name{GetCDLComp}
\alias{GetCDLComp}
\title{Request for data from CropScape on land cover changes over time}
\usage{
GetCDLComp(
  aoi,
  year1,
  year2,
  type,
  mat = TRUE,
  crs = NULL,
  tol_time = 20,
  manual_try = TRUE
)
}
\arguments{
\item{aoi}{Area of interest. Could be a 5-digit FIPS code of a county, three coordinates that defines a triangle,
or four corner points that defines a rectangle (or a box), or a single coordinate. The default coordinate system used by CDL is a projected
coordinate system called Albers projection (or Albers equal-area conic projection). Users could specify coordinates based on a
different coordinate system (defined by the \code{crs} argument), including the geographic coordinate system such as latitude-longitude.}

\item{year1}{Crop year. Should be a 4-digit numerical value.}

\item{year2}{Crop year. Should be a 4-digit numerical value.}

\item{type}{Type of AOI. 'f' for county, 'ps' for triangle with multiple coordinates, 'b' for box with four corner points.}

\item{mat}{TRUE/FALSE. If TRUE (default), return a data table. If FALSE, return a raster file;}

\item{crs}{Coordinate system. NULL if use the default coordinate system (e.g., Albers projection); Use '+init=epsg:4326' for longitude/latitude.}

\item{tol_time}{Number of seconds to wait for a response until giving up. Default is 20 seconds.}

\item{manual_try}{True (default) for trying calculating land cover changes using the~\code{manualrotate} function. If False, no attempt is made.}
}
\value{
The function returns a data table or a raster file.
}
\description{
A function that makes HTTP GET requests from CropScape for land cover changes over time. This function implements the GetCDLComp service provided by CropScape \url{https://nassgeodata.gmu.edu/CropScape}.
}
\details{
Land cover changes are obtained by first merging two raster files in two different years (year1 and year 2) together based on the geographic coordinates. The merged file reports
land cover for each pixel used in year1 and year2. Secondly, the function aggregates the data by counting the number of pixels by types of land cover changes, such as corn to soybeans.
The final product of this function returns pixel counts and acreages (calculated from pixel counts) for each type of land cover from year1 to year2 in the AOI.

Note that the two rasters must have the same spatial resolutions and identical coordinates to be directly merged together. This is necessary for obtaining data on land cover change directly
from CropScape. However, the \code{GetCDLComp} function allows users to obtain land cover changes from two raster files that have different resolutions. This is achieved by resample the raster
data using the nearest neighbor resampling technique such that both rasters have the same resolutions (the finer resolution raster is downscaled to lower resolution). Then, crop land changes are
obtained from a manual processing based on the \code{manualrotate} function.  This functionality is useful considering that the rasters in 2006 and 2007 are at 56-meter resolution,
different to rasters in other years that are at 30-meter resolution. Also note that the resampling process will lead to sampling errors.
Whenever the manual calculation of land cover changes is used, a warning message will show up to alert users.
If without warning, the data are directly from the CropScape GetCDLComp service.

In rare cases, CropScape fails to generate land cover change data even without the issue of unequal spatial resolution. A common issue is mismatch in data sizes: the raster files in two years
have different pixel numbers. It is unclear that why this would happen. Nevertheless, when there is data mismatch, the \code{GetCDLComp} function will attempt to calculate for land cover change
manually using the \code{manual_rotate} function. Data associated with the unmatched coordinates are
discarded at the merging process. Again, a warning message will show up to alert users.

In some cases, the two raster files for the same AOI do not have matched coordinates (the reasons are unknown). CropScape cannot calculate land cover changes,
neither does the \code{manual_rotate} function. A warning message will show up to alert users.

The \code{tol_time} argument specifies the time limit for making the HTTP GET request. This is useful particularly when the CropScape server has an issue with processing the data.
CropScape could sometime take minutes before returning an error saying that the requested data are not available. By setting a time limit, the \code{GetCDLComp} function would stop
making the data requests when the time limit is reached. The default time limit is 20 seconds.
}
\examples{
\donttest{
# Example 1. Retrieve data for the Champaign county in Illinois (FIPS = 17109) in 2017-2018.
data <- GetCDLComp(aoi = '17019', year1 = 2017, year2 = 2018, type = 'f')
head(data, 5)

# Example 2. Retrieve data for a triangle defined by three coordinates in 2017-2018.
aoi <- c(175207,2219600,175207,2235525,213693,2219600)
data <- GetCDLComp(aoi = aoi, year1 = 2017, year2 = 2018, type = 'ps')
head(data, 5)

# Example 3. Retrieve data for a rectangle box defined by four corner points in 2018.
data <- GetCDLComp(aoi = c(130783,2203171,153923,2217961), year1 = 2017, year2 = 2018, type = 'b')
head(data, 5)
}
}
