## ----setup, include=FALSE-----------------------------------------------------
library(svyCausalGLM)
library(survey)
library(nnet)
library(ggplot2)
knitr::opts_chunk$set(
  echo = TRUE,
  message = FALSE,
  warning = FALSE
)

## ----synthetic-data-----------------------------------------------------------
set.seed(123)
n <- 800
synthetic_svy <- data.frame(
  psu     = sample(1:80, n, replace = TRUE),
  strata  = sample(1:40, n, replace = TRUE),
  weight  = runif(n, 0.5, 3),
  exposure = rbinom(n, 1, 0.45),
  age     = round(rnorm(n, 50, 12)),
  sex     = factor(sample(c("Male", "Female"), n, replace = TRUE)),
  bmi     = round(rnorm(n, 27, 4), 1)
)
linpred <- -2 + 0.8 * synthetic_svy$exposure + 0.03 * synthetic_svy$age
synthetic_svy$outcome <- rbinom(n, 1, plogis(linpred))
head(synthetic_svy)


## -----------------------------------------------------------------------------
#final_svyglm(data, dep_var, covariates, id_var,  strata_var, weight_var, family = "binomial", level = 0.95, interaction_terms = NULL) 

## ----final-svyglm-example-----------------------------------------------------
fit_main<-final_svyglm(data=synthetic_svy,
                         dep_var="outcome",
                         covariates = c("age", "sex", "bmi"),
                         id_var      = "psu",           # Changed from psu_var
                         strata_var  = "strata",
                         weight_var  = "weight",
                         family = "binomial",
                         level = 0.95,
                         interaction_terms = NULL)
fit_main$OR_table

## -----------------------------------------------------------------------------
#final_prop_svyglm(data,  dep_var,  covariates,  exposure,  id_var,   strata_var,  weight_var, exposure_type = "binary", outcome_covariates = NULL, level = 0.95)

## ----Synthetic data 2, eval=TRUE----------------------------------------------
set.seed(123)
n <- 1500
dat <- data.frame(
  psu = sample(1:10, n, replace = TRUE),
  strata = sample(1:5, n, replace = TRUE),
  weight = runif(n, 0.5, 2),
  age = rnorm(n, 50, 10),
  sex = factor(sample(c("Male", "Female"), n, replace = TRUE)),
  exposure_bin = rbinom(n, 1, 0.5)
)
dat$outcome <- rbinom(n, 1, plogis(-2 + 0.03*dat$age + 0.5*dat$exposure_bin))
## Multinomial exposure
dat$exposure_3cat <- cut(dat$age,
                         breaks = quantile(dat$age, probs = c(0, 1/3, 2/3, 1)),
                         labels = c("Low", "Medium", "High"),
                         include.lowest = TRUE)
exp_eff <- ifelse(dat$exposure_3cat == "Low", 0,
                  ifelse(dat$exposure_3cat == "Medium", 0.6, 1.2))
dat$outcome_cat <- rbinom(n, 1, plogis(-3 + 0.02 * dat$age + 0.4*(dat$sex=="Male") + exp_eff))

## ----final-prop-svyglm-example-binary, eval=TRUE------------------------------
## Binary exposure
fit_bin <- final_prop_svyglm(
  dat,
  dep_var = "outcome",
  covariates = c("age", "sex"),
  exposure = "exposure_bin",
  id_var = "psu",
  strata_var = "strata",
  weight_var = "weight"
)
fit_bin$OR_table

## Continuous exposure
fit_cont <- final_prop_svyglm(
  dat,
  dep_var = "outcome",
  covariates = c("sex"),
  exposure = "age",
  id_var = "psu",
  strata_var = "strata",
  weight_var = "weight",
  exposure_type = "continuous"
)
fit_cont$OR_table


## Multinomial exposure
fit_multi <- final_prop_svyglm(
  dat,
  dep_var = "outcome_cat",
  covariates = c("age", "sex"),
  exposure = "exposure_3cat",
  id_var = "psu",
  strata_var = "strata",
  weight_var = "weight",
  exposure_type = "multinomial"
)
fit_multi$OR_table



## -----------------------------------------------------------------------------
##final_prog_svyglm(data, dep_var, covariates, exposure, id_var,  strata_var,  weight_var, exposure_type = "binary", level = 0.95, ...)

## ----final-prog-svyglm-example------------------------------------------------
fit_prog <- final_prog_svyglm(
  data        = synthetic_svy,
  dep_var     = "outcome",
  exposure    = "exposure",      # Added (required by your function)
  covariates  = c("age", "sex", "bmi"), # Changed from indep_vars
  id_var      = "psu",           # Changed from psu_var
  strata_var  = "strata",
  weight_var  = "weight"
)
fit_prog$OR_table

## -----------------------------------------------------------------------------
##viz_auc_svyglm(fit_object, title = "Weighted ROC Curve", line_color = "#0072B2")

## ----viz-AUC-example1, fig.width=6, fig.height=4------------------------------
viz_auc_svyglm(fit_main)

## ----viz-AUC-example2, fig.width=6, fig.height=4------------------------------
viz_auc_svyglm(fit_bin)

## ----viz-AUC-example3, fig.width=6, fig.height=4------------------------------
viz_auc_svyglm(fit_prog)

