## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>"
)

## ----eval = FALSE-------------------------------------------------------------
#  install.packages("mfcurve")

## ----eval = FALSE-------------------------------------------------------------
#  # If devtools is not already installed
#  install.packages("devtools")
#  
#  # Install the mfcurve development version from GitHub
#  devtools::install_github("XAM12/mfcurve_R")

## ----message=FALSE------------------------------------------------------------
library(mfcurve)
library(dplyr)
set.seed(123)

# Simulate data with 1,000 observations (5 × 2 × 2 design)
df <- data.frame(
  drug = sample(c("A", "B", "C", "D", "E"), 1000, replace = TRUE),
  dose = sample(c("low", "high"), 1000, replace = TRUE),
  setup = sample(c("single-blind", "double-blind"), 1000, replace = TRUE)
)

# Simulate health outcome (1–10 scale)
df$health <- 6 +
  ifelse(df$drug == "B", 0.5,        # beneficial
  ifelse(df$drug == "E", -0.5, 0)) + # detrimental
  ifelse(df$setup == "single-blind", 0.4, 0) +  # stronger effect for single-blind
  rnorm(1000, 0, 1.2)  # noise

# Ensure values are between 1 and 10
df$health <- pmin(pmax(df$health, 1), 10)

## -----------------------------------------------------------------------------
mfcurve(data = df, outcome = "health", factors = c("dose", "drug", "setup"), mode = "expanded")

## ----include=FALSE, eval=FALSE------------------------------------------------
#  # Export static graph for vignette
#  library(plotly)
#  mfcurve(df, outcome = "health", factors = c("dose", "drug", "setup"), mode = "expanded") %>%
#    config(
#      toImageButtonOptions = list(
#        format = "png",
#        filename = "example1",
#        width = 600,
#        height = 550,
#        scale = 3
#      )
#    )

## ----echo = FALSE, out.width = "100%"-----------------------------------------
knitr::include_graphics("example1.png")

## ----message=FALSE------------------------------------------------------------
library(mfcurve)
library(dplyr)
set.seed(456)

# Simulate sociodemographic data for 8,000 observations
df <- data.frame(
  gender = sample(c("male", "female"), 8000, replace = TRUE),
  age = sample(c("20–29", "30–39", "40–49", "50–59", "60+"), 8000, replace = TRUE),
  education = sample(c("high", "low"), 8000, replace = TRUE)
)

# Define a modest linear age effect
age_effect <- c("20–29" = 0, "30–39" = 0.7, "40–49" = 1.4, "50–59" = 2.1, "60+" = 2.8)

# Generate hourly wage as a function of age, gender, and education (base wage: $12)
df$wage <- 12 +
  age_effect[df$age] +
  ifelse(df$gender == "male", 1, 0) +  # male wage premium
  ifelse(df$gender == "male" & df$education == "high", 2.5, 0) +  # differential returns to education
  rnorm(8000, 0, 2) #noise

## -----------------------------------------------------------------------------
mfcurve(data = df, outcome = "wage", factors = c("gender", "age", "education"), mode = "expanded", showSigStars = FALSE)

## ----include=FALSE, eval=FALSE------------------------------------------------
#  # Export static graph for vignette
#  mfcurve(df, outcome = "wage", factors = c("gender", "age", "education"), mode = "expanded", showSigStars = FALSE) %>%
#    config(
#      toImageButtonOptions = list(
#        format = "png",
#        filename = "example2",
#        width = 600,
#        height = 550,
#        scale = 3
#      )
#    )

## ----echo = FALSE, out.width = "100%"-----------------------------------------
knitr::include_graphics("example2.png")

